/*
 * This file is part of CityGuide
 *
 * Copyright (C) 2007 Pekka Rönkkö (pronkko@gmail.com).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <glib.h>
#include <stdlib.h>

#include <libosso.h>
#include <errno.h>
#include <libgnomevfs/gnome-vfs.h>
#include <math.h>
#include <string.h>

#include "parser.h"
#include "map_widget_defs.h"


#include <gconf/gconf-client.h>

#define NAME_START "<name>"
#define NAME_END "</name>"

#define TITLE_START "<title>"
#define TITLE_END "</title>"

#define LAT_START "<lat>"
#define LAT_END "</lat>"

#define LNG_START "<lng>"
#define LNG_END "</lng>"



/*source: 1 wikipedia, 2 source */
#define GCONF_KEY_SOURCE "/apps/cityguide/source"

#define GCONF_KEY_COUNTRY "/apps/cityguide/country"



/* for calculating distance.*/


/* radius where looking places */
#define RADIUS 20




gdouble calculate_distance(gdouble lat1, gdouble lon1, gdouble lat2,
			   gdouble lon2)
{
    gdouble dlat, dlon, slat, slon, a;
    gdouble distance;


    /* Convert to radians. */
    lat1 *= (PI / 180.f);
    lon1 *= (PI / 180.f);
    lat2 *= (PI / 180.f);
    lon2 *= (PI / 180.f);

    dlat = lat2 - lat1;
    dlon = lon2 - lon1;

    slat = sinf(dlat / 2.f);
    slon = sinf(dlon / 2.f);
    a = (slat * slat) + (cosf(lat1) * cosf(lat2) * slon * slon);

    distance = ((2.f * atan2f(sqrtf(a), sqrtf(1.f - a))) * EARTH_RADIUS);

//    g_debug("distance: %f\n", distance);
    return distance;

}


gchar *parse_place(Coordinates coords)
{

    GnomeVFSHandle *handle;
    GnomeVFSResult result;
    GString *string;
    gchar buffer[1024];
    GnomeVFSFileSize bytes_read;
    gchar *location;



    gchar *parse_url;

    parse_url =
	g_strdup_printf
	("http://ws.geonames.org/findNearbyPostalCodes?lang=fi&lat=%f&lng=%f",
	 coords.lat, coords.lng);

    if ((result =
	 gnome_vfs_open(&handle, parse_url,
			GNOME_VFS_OPEN_READ)) != GNOME_VFS_OK) {
	g_debug("%s\n", gnome_vfs_result_to_string(result));
	return "Place not found_4";
    }

    string = g_string_new("");
    while ((result =
	    gnome_vfs_read(handle, buffer, 1024,
			   &bytes_read)) == GNOME_VFS_OK) {
	g_string_append_len(string, buffer, bytes_read);
    }


    gnome_vfs_close(handle);

    if (result != GNOME_VFS_OK && result != GNOME_VFS_ERROR_EOF) {
	g_debug("2:%s\n", gnome_vfs_result_to_string(result));
	return "Place not found_3";
    }

    if ((location =
	 g_strstr_len(string->str, string->len, NAME_END)) == NULL) {
	g_debug("3:%s\n", "Error while parsing web page.");
	return "Place not found_2";
    }

    *location = 0;

    if ((location =
	 g_strstr_len(string->str, string->len, NAME_START)) == NULL) {
	g_debug("4:%s\n", "Error while parsing web page.");
	return "Place not found_1";
    }

    g_debug("You've been located to %s\n", location + strlen(NAME_START));

    g_string_free(string, TRUE);
    return location + strlen(NAME_START);




}

/* This function parse wikipedia articles from  geoname service called:
findNearbyWikipedia */

GPtrArray *parse_wikipedia_articles(Coordinates coords)
{

/* creating url */

    gchar *place_url = NULL;
    gchar *lang = NULL;

//for gconf 
    GConfClient *gconf = NULL;

    gconf = gconf_client_get_default();


    lang = gconf_client_get_string(gconf, GCONF_KEY_COUNTRY, NULL);

    g_debug("lang from gconf %s", lang);

    g_object_unref(gconf);

    place_url =
	g_strdup_printf
	("http://ws.geonames.org/findNearbyWikipedia?lang=%s&lat=%f&lng=%f&radius=%d",
	 lang, coords.lat, coords.lng, RADIUS);

    GnomeVFSHandle *handle;
    GnomeVFSResult result;
    GString *string = NULL;
    gchar buffer[1024];
    GnomeVFSFileSize bytes_read;
    gchar **parsed_buffer = NULL;
    gint i;
    gchar *location = NULL;
    gchar *parsed_buffer_copy = NULL;
    gchar *tempdata = NULL;

    GPtrArray *place_array = NULL;
    GPtrArray *error_array = NULL;
    PlacesInfo *placeinfo = NULL;
    PlacesInfo *error_placeinfo = NULL;
    place_array = g_ptr_array_new();

    error_placeinfo = g_new(PlacesInfo, 1);
    error_placeinfo->name = "Error";
    error_placeinfo->lng = "0";
    error_placeinfo->lat = "0";

    error_array = g_ptr_array_new();

    g_ptr_array_add(error_array, error_placeinfo);



    if ((result =
	 gnome_vfs_open(&handle, place_url,
			GNOME_VFS_OPEN_READ)) != GNOME_VFS_OK) {
	g_debug("%s\n", gnome_vfs_result_to_string(result));
	return error_array;

    }
    string = g_string_new("");
    while ((result =
	    gnome_vfs_read(handle, buffer, 1024,
			   &bytes_read)) == GNOME_VFS_OK) {
	g_string_append_len(string, buffer, bytes_read);
    }

    


    parsed_buffer = g_strsplit(string->str, "</entry>", -1);


    //free original string
    g_string_free(string, TRUE);


    for (i = 0; parsed_buffer[i] != NULL; i++) {

	placeinfo = g_new(PlacesInfo, 1);
	if (parsed_buffer[i] != NULL) {
	    g_debug("count %d\n", i);


	    /* parsing name */

	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      TITLE_END)) == NULL) {
		g_debug("1.Error while parsing web page.\n");
	        break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      TITLE_START)) == NULL) {
		g_debug("2.Error while parsing web page.\n");
		break;
	    }

	    tempdata = g_strdup(location + strlen(TITLE_START));
	    placeinfo->name = tempdata;


	    g_free(parsed_buffer_copy);

	    tempdata = NULL;

	    /* parsing lat */
	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LAT_END)) == NULL) {
		g_debug("3Error while parsing web page.\n");
		break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LAT_START)) == NULL) {
		g_debug("4Error while parsing web page.\n");
	    }


	    tempdata = g_strdup(location + strlen(LAT_START));
	    placeinfo->lat = tempdata;
	    g_free(parsed_buffer_copy);

	    tempdata = NULL;

	    /* parsing lng */

	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LNG_END)) == NULL) {
		g_debug("5Error while parsing web page.\n");
		break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LNG_START)) == NULL) {
		g_debug("6Error while parsing web page.\n");
	    }

	    tempdata = g_strdup(location + strlen(LNG_START));
	    placeinfo->lng = tempdata;
	    g_free(parsed_buffer_copy);
		
	     placeinfo->icon = WIKI_W;

	    tempdata = NULL;
	 /**/}
	/* Adding placeinfo struct to array */
	g_ptr_array_add(place_array, placeinfo);
    }

    g_strfreev(parsed_buffer);
    g_free(place_url);
    g_debug("end\n");

    return place_array;

    
}



/* This function parse populated places from geoname service called:
findNearbyPlaceName */

GPtrArray *parse_populated_places(Coordinates coords)
{

/* creating url, markers */

    gchar *place_url;
    place_url =
	g_strdup_printf
	("http://ws.geonames.org/findNearbyPlaceName?lat=%f&lng=%f&radius=%d",
	 coords.lat, coords.lng, RADIUS);


    GnomeVFSHandle *handle;
    GnomeVFSResult result;
    GString *string;
    gchar buffer[1024];
    GnomeVFSFileSize bytes_read;
    gchar **parsed_buffer = NULL;
    int i;
    gchar *location = NULL;
    gchar *parsed_buffer_copy = NULL;
    gchar *tempdata = NULL;


    GPtrArray *place_array = NULL;
    GPtrArray *error_array = NULL;
    PlacesInfo *placeinfo = NULL;
    PlacesInfo *error_placeinfo = NULL;


    place_array = g_ptr_array_new();

    error_placeinfo = g_new(PlacesInfo, 1);
    error_placeinfo->name = "Error";
    error_placeinfo->lng = "0";
    error_placeinfo->lat = "0";

    error_array = g_ptr_array_new();

    g_ptr_array_add(error_array, error_placeinfo);

    if ((result =
	 gnome_vfs_open(&handle, place_url,
			GNOME_VFS_OPEN_READ)) != GNOME_VFS_OK) {
	g_debug("%s\n", gnome_vfs_result_to_string(result));
	return error_array;

    }

    string = g_string_new("");
    while ((result =
	    gnome_vfs_read(handle, buffer, 1024,
			   &bytes_read)) == GNOME_VFS_OK) {
	g_string_append_len(string, buffer, bytes_read);
    }
    gnome_vfs_close(handle);
    if (result != GNOME_VFS_OK && result != GNOME_VFS_ERROR_EOF) {
	g_debug("Error while parsing web page.\n");
	return error_array;
    }



    parsed_buffer = g_strsplit(string->str, "</geoname>", -1);

    //free original string
    g_string_free(string, TRUE);


    for (i = 0; parsed_buffer[i] != NULL; i++) {
	placeinfo = g_new(PlacesInfo, 1);
	if (parsed_buffer[i] != NULL) {
	    g_debug("count %d\n", i);


	    /* parsing name */
	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      NAME_END)) == NULL) {
		g_debug("1Error while parsing web page.\n");
		break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      NAME_START)) == NULL) {
		g_debug("2.1.Error while parsing web page.\n");
		break;
	    }

	    tempdata = g_strdup(location + strlen(NAME_START));
	    placeinfo->name = tempdata;
	    g_free(parsed_buffer_copy);

	    tempdata = NULL;

	    /* parsing lat */
	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LAT_END)) == NULL) {
		g_debug("3.1.Error while parsing web page.\n");
		break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LAT_START)) == NULL) {
		g_debug("4.1.Error while parsing web page.\n");
	    }


	    tempdata = g_strdup(location + strlen(LAT_START));
	    placeinfo->lat = tempdata;
	    g_free(parsed_buffer_copy);

	    tempdata = NULL;

	    /* parsing lng */

	    parsed_buffer_copy = g_strdup(parsed_buffer[i]);

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LNG_END)) == NULL) {
		g_debug("5Error while parsing web page.\n");
		break;
	    }
	    *location = 0;

	    if ((location =
		 g_strstr_len(parsed_buffer_copy,
			      strlen(parsed_buffer_copy),
			      LNG_START)) == NULL) {
		g_debug("6Error while parsing web page.\n");
	    }

	    tempdata = g_strdup(location + strlen(LNG_START));
	    placeinfo->lng = tempdata;
	    g_free(parsed_buffer_copy);

	    placeinfo->icon = PEOPLE_BLUE;

	    tempdata = NULL;
	 /**/}
	/* Adding placeinfo struct to array */
	g_ptr_array_add(place_array, placeinfo);
    }

    g_strfreev(parsed_buffer);
    g_free(place_url);
    g_debug("end\n");


    return place_array;
}



