/*
 * This file is part of CityGuide
 *
 * Copyright (C) 2007 Pekka Rönkkö (pronkko@gmail.com).
 *
 * Part of code is ripped from John Costigan's Maemo Mapper
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <hildon/hildon-banner.h>
#include <hildon-uri.h>


#include <gdk/gdkkeysyms.h>
#include <libosso.h>
#include <errno.h>

#include <gps.h>
#include <pthread.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gpsbt.h>
#include <libgnomevfs/gnome-vfs.h>



#include <gconf/gconf-client.h>
#include <math.h>
#include <glib/gstdio.h>
#include <dbus/dbus-glib.h>

#include "appdata.h"
#include "parser.h"
#include "map_widget.h"

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>


#define PACKAGE "cityguide"
#define VERSION "1.0"
#define SERVICE_URI "org.o4f.pronkko.cityguide"
#define OBJECT_PATH "/org/o4f/pronkko/cityguide"


#define ERROR_BUFFER_MAX 255
#define DEBUGLEVEL 0
#define UPDATE_SECONDS 1

//for gconf 

#define GCONF_KEY_SOURCE "/apps/cityguide/source"
#define GCONF_KEY_COUNTRY "/apps/cityguide/country"
#define GCONF_KEY_MAPSOURCE "/apps/cityguide/mapsource"


GtkWidget *map;
LocationGPSDevice *device;
LocationGPSDControl *cont;

void check_place(void);
void add_coords(void);
void add_current_city(void);

AppData *appdata;
gboolean is_fullscreen;
gboolean is_gps_used=FALSE;

static gboolean city_added = 0;
static GtkWidget *current_location = NULL;
static guint timeout_id = 0;

static Coordinates coords;

static GPtrArray *pop_places = NULL;
static gboolean array_filled = 0;


static gint previus = -1;
static gint gps_closed = 0;

static gboolean is_screen_draw = 0;

gint dbus_req_handler(const gchar * interface, const gchar * method,
		      GArray * arguments, gpointer data,
		      osso_rpc_t * retval)
{
    return OSSO_OK;
}

static gboolean window_state_event(GtkWidget * widget,
				   GdkEventWindowState * event,
				   HildonWindow * window)
{
    if (event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN) {
	is_fullscreen =
	    event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN;
    }
    return FALSE;
}

static gboolean key_press_event(GtkWidget * widget, GdkEventKey * event,
				HildonWindow * window)
{
    switch (event->keyval) {
    case GDK_F6:
	if (is_fullscreen) {
	   // gtk_window_unfullscreen(GTK_WINDOW(window));
	} else {
	   // gtk_window_fullscreen(GTK_WINDOW(window));
	}
	return TRUE;
    }
    return FALSE;
}


void zoom_in(GtkWidget * widget, gpointer data)
{

    DEBUG_FUNC_START
    gfloat zoom;

    zoom = map_widget_get_current_zoom_level(map) + 1;

    if (zoom >0 && zoom <17)
    	map_widget_set_zoom(map, zoom);

    gchar *current_zoom;

    current_zoom = g_strdup_printf("%d",map_widget_get_current_zoom_level(map));

    gtk_label_set_text(data,current_zoom);
    g_free(current_zoom);

    DEBUG_FUNC_END

}

void zoom_out(GtkWidget * widget, gpointer data)
{

    DEBUG_FUNC_START

    gfloat zoom;

    zoom = map_widget_get_current_zoom_level(map) - 1;

    if (zoom >0 && zoom <17)
   	 map_widget_set_zoom(map,zoom);

    gchar *current_zoom;
    current_zoom = g_strdup_printf("%d",map_widget_get_current_zoom_level(map));

    gtk_label_set_text(data,current_zoom);
    g_free(current_zoom);

    DEBUG_FUNC_END
}



/* If there is problems, error dialog
will show and program closes */
void error_message(gchar * error_msg)
{
    DEBUG_FUNC_START
    GtkWidget *dialog = NULL;
    GtkWidget *label = NULL;
    gchar *error_text = NULL;

    dialog = gtk_dialog_new_with_buttons("Error", NULL,
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    error_text = g_strdup_printf("%s\nPress ok to quit CG", error_msg);
    label = gtk_label_new(error_text);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);


    gtk_widget_show_all(dialog);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) 
    {
    exit(0);
    }
    gtk_widget_destroy(dialog);
    g_free(error_text);
    DEBUG_FUNC_END
}


/* Ask if gps is used */
gboolean gps_start_message(void)
{
    DEBUG_FUNC_START
    GtkWidget *dialog = NULL;
    GtkWidget *label = NULL;

    dialog = gtk_dialog_new_with_buttons("GPS", NULL,
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_YES, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_NO, GTK_RESPONSE_REJECT, 
					 NULL);

    
    label = gtk_label_new("Do you use GPS");
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), label);


    gtk_widget_show_all(dialog);

    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

	is_gps_used = TRUE;
	
    }
    gtk_widget_destroy(dialog);
  
    if (is_gps_used)
    {
        DEBUG_FUNC_END
	return TRUE;
    }
    else
    {
    	DEBUG_FUNC_END
    	return FALSE;
    }

}

gboolean check_place_poll(gpointer user_data)
{

    DEBUG_FUNC_START
    add_coords();



    MapPoint *new_cen=NULL;
    new_cen = g_new0(MapPoint, 1);

   if (is_gps_used)
   {
    if (device->fix->mode == 3)
    {
	coords.lat = device->fix->latitude;
	coords.lng = device->fix->longitude;
         add_current_city();
	city_added = 1;
    	new_cen->latitude = coords.lat;
    	new_cen->longitude = coords.lng;
    	map_widget_set_current_location(map, new_cen);
    	g_free(new_cen);
    }
   }
    map_widget_draw_current_place(map);

    if (array_filled) {
	check_place();
    }
    DEBUG_FUNC_END
    return TRUE;
}

void set_update_timeout(int seconds)
{

    DEBUG_FUNC_START
    if (timeout_id != 0) {
	g_source_remove(timeout_id);
    }
    timeout_id = g_timeout_add(seconds * 1000, check_place_poll, NULL);
    DEBUG_FUNC_END

}


void add_current_city(void)
{
    DEBUG_FUNC_START

    gchar *your_current_place = NULL;
    your_current_place = parse_place(coords);

    gtk_label_set_text(GTK_LABEL(current_location), your_current_place);
    g_debug("your current location has been updated\n");
    DEBUG_FUNC_END
}

void draw_places_to_screen(gint source)
{
    DEBUG_FUNC_START

    MapPoint *poi;
    poi = g_new0(MapPoint, 1);

    map_widget_draw_current_place(map);

    gint i = 0;
    PlacesInfo *returned_placeinfo = NULL;
    GtkWidget *label_place = NULL;
    GtkWidget *distance_to_place = NULL;

    gchar *ppl = NULL;
    gchar *dist = NULL;

    MapPoint *point;
    point = g_new0(MapPoint, 1);

    gdouble distance = 0.0;

    gtk_widget_destroy(appdata->table_POI);

    GtkWidget *table_POI = NULL;
    table_POI = gtk_table_new(10, 7, TRUE);
    appdata->table_POI = table_POI;

    for (i = 0; i < pop_places->len - 1 && i < 8; i++) {
	returned_placeinfo = g_ptr_array_index(pop_places, i);
	distance =
	    calculate_distance(atof(returned_placeinfo->lat),
			       atof(returned_placeinfo->lng), coords.lat,
			       coords.lng);
	ppl = g_strdup_printf("%s\n", returned_placeinfo->name);
	dist = g_strdup_printf("%.fm\n", distance * 1000);
	label_place = gtk_label_new(ppl);
	distance_to_place = gtk_label_new(dist);

	gtk_misc_set_alignment((GtkMisc *) label_place, 0, 0.5);
	gtk_misc_set_alignment((GtkMisc *) distance_to_place, 0, 0.5);
	gtk_table_attach_defaults(GTK_TABLE(appdata->table_POI),
				  label_place, 0, 4, i, i + 1);
	gtk_table_attach_defaults(GTK_TABLE(appdata->table_POI),
				  distance_to_place, 4, 6, i, i + 1);

	/*draw arc to exact point */


    	point->latitude =  atof(returned_placeinfo->lat);
    	point->longitude = atof(returned_placeinfo->lng);
	map_widget_draw_arc_to_coords(map, point,5, 5, TRUE);

    }
    gtk_container_add(GTK_CONTAINER(appdata->vbox_places),
		      appdata->table_POI);
    gtk_widget_show_all(appdata->vbox_places);

    g_free(ppl);
    g_free(dist);
    DEBUG_FUNC_END


}

gboolean find_places_func(void)
{
    DEBUG_FUNC_START
    gint source;
    GConfClient *gconf = NULL;

    gconf = gconf_client_get_default();

    source = gconf_client_get_int(gconf, GCONF_KEY_SOURCE, NULL);

    previus = -1;

    if (source == 1) {
	pop_places = parse_wikipedia_articles(coords);

    }

    if (source == 2) {
	pop_places = parse_populated_places(coords);

    }

    map_widget_set_static_poi_array(map, pop_places, TRUE);
    array_filled = 1;

    draw_places_to_screen(source);

    is_screen_draw = 1;

    g_object_unref(gconf);
    DEBUG_FUNC_END
    return TRUE;

}

void find_places(GtkWidget * button, gpointer data)
{


    DEBUG_FUNC_START

    gboolean test = 0;;

    if (city_added == 1) 
    {
    test = find_places_func();
    }
    else 
    {
    hildon_banner_show_information(GTK_WIDGET(appdata->window),
				   NULL, "GPS not ready");
    }
    
    add_current_city();
    DEBUG_FUNC_END
}
void check_place(void)
{
    DEBUG_FUNC_START

    GConfClient *gconf = NULL;
    gint source = 0;
    gchar *lang = NULL;
    gconf = gconf_client_get_default();


    source = gconf_client_get_int(gconf, GCONF_KEY_SOURCE, NULL);
    lang = gconf_client_get_string(gconf, GCONF_KEY_COUNTRY, NULL);

    gint hit_number = -1;
    gdouble pop_place_coord_lat = 0.0;
    gdouble pop_place_coord_lng = 0.0;

    PlacesInfo *returned_placeinfo = NULL;
    gint i = 0;
    gchar *ppl = NULL;
    gdouble distance = 0.0;

    for (i = 0; i < pop_places->len - 1; i++) {
	returned_placeinfo = g_ptr_array_index(pop_places, i);

	pop_place_coord_lat = atof(returned_placeinfo->lat);
	pop_place_coord_lng = atof(returned_placeinfo->lng);

	distance =
	    calculate_distance(pop_place_coord_lat, pop_place_coord_lng,
			       coords.lat, coords.lng);
	if (distance < 0.015) 
	{
	    ppl =
		g_strdup_printf("http://%s.wikipedia.org/wiki/%s", lang, 
				returned_placeinfo->name);
	    hit_number = i;

	    if (hit_number != previus) {
		 hildon_uri_open(ppl,NULL,NULL);
	}
	    previus = i;
	}
    }

    draw_places_to_screen(source);
    g_free(ppl);
    g_object_unref(gconf);
    DEBUG_FUNC_END
}

void add_coords(void)
{
    DEBUG_FUNC_START
    gchar *temp = NULL;
    temp = g_strdup_printf("lat: %.3f\nlon: %.3f", coords.lat, coords.lng);
    gtk_label_set_text(GTK_LABEL(appdata->coords_label), temp);
    g_free(temp);
    DEBUG_FUNC_END
}

static void
location_changed (LocationGPSDevice *device,
		  gpointer           userdata)
{
/* loacation changed*/
}

void *fill_gps_coordinate_fields(void *data)
{
    return 0;
}
static
void settings(GtkWidget * widget, gpointer data)
{
   DEBUG_FUNC_START
    GtkWidget *dialog = NULL;
    GtkWidget *table = NULL;
    GtkWidget *label = NULL;
    GtkWidget *entry = NULL;
    GtkWidget *entry_maptype = NULL;
    GtkWidget *r_button1 = NULL;
    GtkWidget *r_button2 = NULL;
    gint source;
    const gchar *buffer = NULL;
    GConfClient *gconf = NULL;


    const gchar *new_maptype = NULL;
    gchar *maptype_new = NULL;


    gconf = gconf_client_get_default();

    dialog = gtk_dialog_new_with_buttons("Settings", NULL,
					 GTK_DIALOG_MODAL |
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT, NULL);

    table = gtk_table_new(6, 6, TRUE);


    label = gtk_label_new("Language (e.g. fi,en,de)");
    gtk_misc_set_alignment((GtkMisc *) label, 0, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 0, 1);

    entry = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(table), entry, 3, 6, 0, 1);


    label = gtk_label_new("Maptype (e.g. Open Street)");
    gtk_misc_set_alignment((GtkMisc *) label, 0, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 4, 5);

    entry_maptype = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(table), entry_maptype, 3, 6, 4, 5);

    label = gtk_label_new("Source:");
    gtk_misc_set_alignment((GtkMisc *) label, 0, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 1, 3);

    r_button1 =
	gtk_radio_button_new_with_label(NULL, "Wikipedia articles");
    gtk_table_attach_defaults(GTK_TABLE(table), r_button1, 3, 6, 1, 2);
    gtk_widget_show(r_button1);


    r_button2 =
	gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON
						    (r_button1),
						    "Places nearby");
    gtk_table_attach_defaults(GTK_TABLE(table), r_button2, 3, 6, 2, 3);
    gtk_widget_show(r_button2);

    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), table);

    //take current values from gconf
    gtk_entry_set_text(GTK_ENTRY(entry),
		       gconf_client_get_string(gconf, GCONF_KEY_COUNTRY,
					       NULL));

    gtk_entry_set_text(GTK_ENTRY(entry_maptype),
		       gconf_client_get_string(gconf, GCONF_KEY_MAPSOURCE,
					       NULL));


    source = gconf_client_get_int(gconf, GCONF_KEY_SOURCE, NULL);

    if (source == 1) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(r_button1), TRUE);

    }

    if (source == 2) {
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(r_button2), TRUE);

    }




    gtk_widget_show_all(dialog);
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {

	buffer = gtk_entry_get_text(GTK_ENTRY(entry));
	gconf_client_set_string(gconf, GCONF_KEY_COUNTRY, buffer, NULL);

	new_maptype = gtk_entry_get_text(GTK_ENTRY(entry_maptype));
	gconf_client_set_string(gconf, GCONF_KEY_MAPSOURCE, new_maptype, NULL);
    
        maptype_new = g_strdup_printf("%s",new_maptype);
	map_widget_change_maptype(map,maptype_new);
	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(r_button1))) {
	    gconf_client_set_int(gconf, GCONF_KEY_SOURCE, 1, NULL); 
	}

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(r_button2))) {
	    gconf_client_set_int(gconf, GCONF_KEY_SOURCE, 2, NULL);
	}
    }
    gtk_widget_destroy(dialog);
    g_object_unref(gconf);
    DEBUG_FUNC_END
}
void do_menu(void)
{

    DEBUG_FUNC_START
    GtkWidget *main_menu;
    GtkWidget *menu_item;

    main_menu = gtk_menu_new();

    menu_item = gtk_menu_item_new_with_label("Settings");
    g_signal_connect(menu_item, "activate", G_CALLBACK(settings), NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);


    menu_item = gtk_menu_item_new_with_label("Exit CityGuide");
    g_signal_connect(menu_item, "activate", G_CALLBACK(gtk_main_quit),
		     NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(main_menu), menu_item);

    hildon_window_set_menu(HILDON_WINDOW(appdata->window),
			   GTK_MENU(main_menu));
    DEBUG_FUNC_END
}




static gboolean auto_center_pressed ( GtkWidget *event_box, 
                         		GdkEventButton *event,
					gpointer data)
{
        DEBUG_FUNC_START
	gboolean auto_center_status;	
	auto_center_status = map_widget_get_auto_center_status(map);

	if (auto_center_status == FALSE)
	{
	hildon_banner_show_information(GTK_WIDGET(appdata->window),
					   NULL, "Auto center on");
	map_widget_set_auto_center(map,TRUE);	
        gtk_image_set_from_file(data,AUTO_CENTER_ON);
	}


	if (auto_center_status == TRUE)
	{
	hildon_banner_show_information(GTK_WIDGET(appdata->window),
					   NULL, "Auto center off");
	map_widget_set_auto_center(map,FALSE);
        gtk_image_set_from_file(data,AUTO_CENTER_OFF);
	}

  DEBUG_FUNC_END
}


int main(int argc, char **argv)
{
    DEBUG_FUNC_START

    int maptileloader;
    maptileloader = system("pidof maptile-loader");
 
    if ( maptileloader != 0)
    {
	system("run-standalone.sh maptile-loader &"); 
	printf("123\n");
    }
    HildonProgram *program;
    GtkWidget *window;
    osso_context_t *osso_context;
    int ret;


    //variables for main view
    GtkWidget *button;
    GtkWidget *coords_label;
    GtkWidget *vbox_places;
    GtkWidget *notebook;
    GtkWidget *table_map;

    GtkWidget *zoombutton;
    GtkWidget *zoomlabel;

    GtkWidget* background_panel;
    GtkWidget* image_place;
    GtkWidget* image_zoom_in;
    GtkWidget* image_zoom_out;
    GtkWidget* image_auto_center;
    GtkWidget *fixed_window;
    //for errors
    gchar *error_msg = NULL;


    GConfClient *gconf = NULL;
    gint source = 0;
    gchar *country = NULL;
    gchar *mapsource = NULL;
    GtkWidget *table_POI = NULL;
    gchar *maptype = NULL;

    //inits
    g_thread_init(NULL);
    gdk_threads_init();
    gdk_threads_enter();

    gnome_vfs_init();
    gtk_init(&argc, &argv);
    gps_closed = 0;
    gconf = gconf_client_get_default();



    if ((osso_context =
	 osso_initialize(PACKAGE, VERSION, TRUE, NULL)) == NULL) {
	fprintf(stderr, "ERROR: osso_initialize failed: %s\n",
		strerror(errno));

	error_msg = g_strdup_printf("ERROR: osso_initialize failed: %s\n",
				    strerror(errno));
	error_message(error_msg);
    }
    if ((ret = osso_rpc_set_cb_f(osso_context, SERVICE_URI, OBJECT_PATH,
				 SERVICE_URI, dbus_req_handler,
				 osso_context)) != OSSO_OK) {
	fprintf(stderr, "ERROR: D-BUS callback setting failed (%d)\n",
		ret);
	error_msg =
	    g_strdup_printf("ERROR: D-BUS callback setting failed (%d)\n",
			    ret);

	error_message(error_msg);
    }
    /*create new thread for getting gps data, first ask if gps is used */
    gboolean test;

    test = gps_start_message();

    if (test == TRUE) 
    {

    cont = g_object_new (LOCATION_TYPE_GPSD_CONTROL, NULL);
    cont = location_gpsd_control_get_default();

    location_gpsd_control_start(cont);	
    device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);
	g_signal_connect (device, "changed",
			  G_CALLBACK (location_changed), NULL); 
    is_gps_used = 1;
    }
    else 
    {
    coords.lat = 48.8583;
    coords.lng = 2.2945;
    add_current_city();
    city_added = 1;
    

    }

    program = HILDON_PROGRAM(hildon_program_get_instance());
    g_set_application_name("GPS City Guide");
    window = hildon_window_new();

    hildon_program_add_window(program, HILDON_WINDOW(window));

    g_signal_connect(G_OBJECT(window), "delete_event",
		     G_CALLBACK(gtk_main_quit), NULL);
    g_signal_connect(G_OBJECT(window), "window_state_event",
		     G_CALLBACK(window_state_event), window);
    g_signal_connect(G_OBJECT(window), "key_press_event",
		     G_CALLBACK(key_press_event), window);


    /* Initialize pango stuff */
    PangoFontDescription *pango_font;
    PangoFontDescription *pango_font_zoom;

    pango_font = pango_font_description_new();
    pango_font_description_set_size(pango_font, 15 * PANGO_SCALE);

    pango_font_zoom = pango_font_description_new();
    pango_font_description_set_size(pango_font_zoom, 12 * PANGO_SCALE);

    /*  UI */
    fixed_window = gtk_fixed_new();
    table_map = gtk_table_new(20, 20, TRUE);
    table_POI = gtk_table_new(10, 1, TRUE);

    /* Set background panel image */

    background_panel = gtk_image_new_from_file(BACKGROUND_PANEL);
    gtk_fixed_put(GTK_FIXED(fixed_window),background_panel,489,8);	

    /* Create a new notebook, place the position of the tabs */

    notebook = gtk_notebook_new();
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);

    gtk_fixed_put(GTK_FIXED(fixed_window),notebook,16,8);
    gtk_widget_set_size_request(notebook, 456, 379);

    gtk_widget_show(notebook);

    /*vbox where ppl goes */
    vbox_places = gtk_vbox_new(TRUE, 1);

    /* notebook pages */


    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table_map,
			     gtk_label_new("Map"));

    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), vbox_places,
			     gtk_label_new("Info"));


    /*Label for coordinates */
    coords_label = gtk_label_new("");
    gtk_widget_modify_font (coords_label, pango_font);
    image_place = gtk_image_new_from_file(COORDS_IMAGE);
    gtk_fixed_put(GTK_FIXED(fixed_window),image_place,488,100);
    gtk_fixed_put(GTK_FIXED(fixed_window),coords_label,505,110);

    /* FIll AppData */

    appdata = g_new0(AppData, 1);
    appdata->program = program;
    appdata->window = window;
    appdata->osso_context = osso_context;
    appdata->vbox_places = vbox_places;
    appdata->table_POI = table_POI;
    appdata->coords_label = coords_label;

 
    gtk_container_add(GTK_CONTAINER(vbox_places), table_POI);


    /* Current location, image and labels */
    image_place = gtk_image_new_from_file(CURRENT_PLACE_IMAGE);
    gtk_fixed_put(GTK_FIXED(fixed_window),image_place,488,8);

    current_location = gtk_label_new("");
    gtk_widget_modify_font (current_location, pango_font);
    gtk_widget_set_size_request(current_location, 160, 75);
    gtk_fixed_put(GTK_FIXED(fixed_window),current_location,488,30);	





    /* Find Places button */


    button = gtk_button_new_with_label("Find Places");
    gtk_widget_set_size_request(button, 150, 75);
    g_signal_connect(G_OBJECT(button), "clicked",
		     G_CALLBACK(find_places), NULL);
    gtk_fixed_put(GTK_FIXED(fixed_window),button,498,268);
    gtk_widget_show_all(button);

/* buttons and label for zoom*/


/* just fo r test */

    zoomlabel = gtk_label_new("~");
    gtk_widget_modify_font (zoomlabel,pango_font_zoom);
    gtk_table_attach_defaults(GTK_TABLE(table_map), zoomlabel, 9, 11, 18, 20);
    gtk_widget_show(zoomlabel);


    image_zoom_in = gtk_image_new_from_file(ZOOM_IN_IMAGE);
    zoombutton = gtk_button_new();
    gtk_button_set_image(GTK_BUTTON(zoombutton),image_zoom_in);
    g_signal_connect(G_OBJECT(zoombutton), "clicked",
		     G_CALLBACK(zoom_in), zoomlabel);

    gtk_table_attach_defaults(GTK_TABLE(table_map), zoombutton, 11, 13, 18, 20);

    image_zoom_out = gtk_image_new_from_file(ZOOM_OUT_IMAGE);
    zoombutton = gtk_button_new();
    gtk_button_set_image(GTK_BUTTON(zoombutton),image_zoom_out);
    g_signal_connect(G_OBJECT(zoombutton), "clicked",
		     G_CALLBACK(zoom_out), zoomlabel);  

    gtk_table_attach_defaults(GTK_TABLE(table_map), zoombutton, 7, 9, 18, 20);
 
    gtk_widget_show_all(table_map);

    gtk_container_add(GTK_CONTAINER(window), fixed_window);

    gtk_widget_show_all(window);


    //first time, set program use wikipedia.

    if ((source =
	 gconf_client_get_int(gconf, GCONF_KEY_SOURCE, NULL)) == 0) {
	source = 1;
	gconf_client_set_int(gconf, GCONF_KEY_SOURCE, source, NULL);
    }

    if ((country =
	 gconf_client_get_string(gconf, GCONF_KEY_COUNTRY,
				 NULL)) == NULL) {
	country = g_strdup("en");
	gconf_client_set_string(gconf, GCONF_KEY_COUNTRY, country, NULL);
    }
    if ((mapsource =
	 gconf_client_get_string(gconf, GCONF_KEY_MAPSOURCE,
				 NULL)) == NULL) {
	mapsource = g_strdup("Open Street");
	gconf_client_set_string(gconf, GCONF_KEY_MAPSOURCE, mapsource, NULL);
    }


    do_menu();

    MapPoint *center;
    center = g_new0(MapPoint, 1);

    center->latitude = coords.lat;
    center->longitude = coords.lng;

    map = map_widget_create();

    gtk_table_attach_defaults(GTK_TABLE(table_map), map, 0, 20, 0, 18);

    gtk_widget_show(table_map);



    gtk_widget_show(notebook);


    map_widget_new_from_center_zoom_type(map, center, 14.0 , 0.0, mapsource, osso_context);


    gtk_widget_show(map);
    map_widget_set_current_location_icon(map,PERSON_RED);

/* current zoom label and auto_center button must do after map widget has created */
    gchar *current_zoom;
    current_zoom = g_strdup_printf("%d",map_widget_get_current_zoom_level(map));
    gtk_label_set_text(GTK_LABEL(zoomlabel),current_zoom);
    g_free(current_zoom);

    GtkWidget *auto_center_event_box;

    auto_center_event_box = gtk_event_box_new();
    image_auto_center = gtk_image_new_from_file(AUTO_CENTER_OFF);
    gtk_container_add(GTK_CONTAINER(auto_center_event_box), image_auto_center);
    g_signal_connect (G_OBJECT (auto_center_event_box),  "button_press_event",
	                      G_CALLBACK (auto_center_pressed),
	                      image_auto_center);
    gtk_table_attach_defaults(GTK_TABLE(table_map), auto_center_event_box, 18, 20, 18, 20);
    gtk_widget_show_all(auto_center_event_box);




    set_update_timeout(UPDATE_SECONDS);
    gtk_main();
    gdk_threads_leave();



    if (is_gps_used)
    {
    location_gpsd_control_stop(cont);
    }

 
    g_ptr_array_free(pop_places, TRUE);

    g_free(appdata);
    g_free(maptype);
    g_free(country);
   
   
    g_object_unref(gconf);
 
    osso_deinitialize(appdata->osso_context);
    DEBUG_FUNC_END
    return 0;
}
