/*
 * garchivereaderunzip.c - routines to manipulate archive files
 * using libunzip code.
 *
 * Copyright (C) 2008 Benoit Goudreault-Emond (bgoudreaultemond@gmail.com).
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <glib.h>
#include <string.h>
#include <stdio.h>

#include "unzip.h"
#include "garchivereaderunzip.h"

extern int debug;

static gboolean extension_handled(const char* filename)
{
    static const char* extensions[] = {
	".jpg", ".jpeg", ".png", ".gif",
	".JPG", ".JPEG", ".PNG", ".GIF"
    };
    static const int nr_extensions = 8;
    
    int i;
    for(i = 0; i < nr_extensions; ++i)
    {
	if(strstr(filename, extensions[i]) != NULL)
	    return TRUE;
    }
    return FALSE;
}

static void set_last_error(GArchiveReaderUnzip* reader, gchar* data)
{
    if(reader->last_error)
	g_free(reader->last_error);
    reader->last_error = data;
}

static void
free_page_list(GArchiveReaderUnzip* reader)
{
    int i;

    if(!reader->page_list)
	return;

    for(i = 0; i < reader->page_list->len; ++i)
	g_free(g_array_index(reader->page_list, gchar*, i));
    g_array_free(reader->page_list, TRUE);
    reader->page_list = NULL;
}

static int
register_path(GArchiveReader* reader_iface,
	      const gchar* path)
{
    unzFile file;
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(reader_iface);
    int check_result = g_archive_reader_check_magic(path, &reader->magic);
    if(debug)
	printf(_("Check result for type %s -> %d\n"), reader->magic.sign, check_result);
    if(check_result <= 0)
	return check_result;

    file = unzOpen(path);
    if(!file)
	return -1;
    if(reader->current_file)
    {
	unzClose(reader->current_file);
	free_page_list(reader);
    } 
    reader->current_file = file;
    return 1;
}

#define BUFSIZE 2049

static GArray*
get_file_list(GArchiveReader* reader_iface)
{
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(reader_iface);
    gchar* duped_filename;
    static char filename[BUFSIZE]; /* should be enough */

    if(reader->page_list)
	return reader->page_list;
    
    if(!reader->current_file)
	return NULL;
    
    unzGoToFirstFile(reader->current_file);
    do
    {
	unz_file_info file_info;
	unzGetCurrentFileInfo(reader->current_file,
			      &file_info,
			      filename,
			      BUFSIZE - 1,
			      NULL,
			      0,
			      NULL,
			      0);
	/* terminate with NUL just in case */
	filename[file_info.size_filename] = '\0';
	if(extension_handled(filename))
	{
	    /* assume 16 elements at first */
	    if(!reader->page_list)
		reader->page_list = g_array_sized_new(FALSE, TRUE, sizeof(gchar*), 16);
	    duped_filename = g_strdup(filename);
	    g_array_append_val(reader->page_list, duped_filename);
	}
    } while(unzGoToNextFile(reader->current_file) == UNZ_OK);

    if(!reader->page_list)
    {
	set_last_error(reader, g_strdup(_("No pages in archive")));
    }
    else
    {
	g_array_sort(reader->page_list, 
		     g_archive_reader_translate_strategy(reader->strategy));
    }
    return reader->page_list;
}

static GdkPixbuf*
get_page(GArchiveReader* reader_iface, 
	 int page_number,
	 gdouble* w,
	 gdouble* h)
{
    GdkPixbuf* retval = NULL;
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(reader_iface);    
    const gchar* filename;
    unz_file_info file_info;
    GdkPixbufLoader* loader = NULL;
    int num_bytes = 0;
    GArray* file_list = get_file_list(reader_iface);

    if(w)
	*w = 0;
    if(h)
	*h = 0;
    if(page_number < 0 || !file_list || page_number >= file_list->len)
    {
	set_last_error(reader, g_strdup_printf(_("Page number %d out of range"), page_number));
	return NULL;
    }
	
    filename = (const gchar*)g_array_index(file_list, gchar*, page_number);
    
    if(unzLocateFile(reader->current_file, (const char*)filename, 1) != UNZ_OK)
    {
	set_last_error(reader, g_strdup_printf(_("File %s not found in archive"), filename));
	return NULL;
    }

    if(unzOpenCurrentFile(reader->current_file) != UNZ_OK)
    {
	set_last_error(reader, g_strdup_printf(_("Error reading %s from archive"), filename));
	return NULL;
    }
    loader = gdk_pixbuf_loader_new();
    while((num_bytes = unzReadCurrentFile(reader->current_file, 
					  reader->buf, 
					  reader->buf_size)) > 0)
	gdk_pixbuf_loader_write(loader, reader->buf, num_bytes, NULL);
    unzCloseCurrentFile(reader->current_file);
    gdk_pixbuf_loader_close(loader, NULL);
    if(num_bytes < 0)
    {
	set_last_error(reader, g_strdup_printf(_("Error reading %s from archive"), filename));
	return NULL;
    }

    retval = gdk_pixbuf_loader_get_pixbuf(loader);
    if(retval)
    {
	g_object_ref(retval);
	if(w)
	    *w = gdk_pixbuf_get_width(retval);
	if(h)
	    *h = gdk_pixbuf_get_height(retval);
    }
    else
    {
	set_last_error(reader, g_strdup_printf(_("Unable to load page %d"), page_number));
    }
    g_object_unref(loader);
    return retval;
}

static gchar*
get_last_error(GArchiveReader* reader_iface)
{
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(reader_iface);
    gchar* last_error = reader->last_error;
    reader->last_error = NULL;
    return last_error;
}

static void
set_sort_strategy(GArchiveReader* reader_iface, SortStrategy strategy)
{
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(reader_iface);
    SortStrategy old_strategy = reader->strategy;
    reader->strategy = strategy;
    if(strategy != old_strategy && reader->page_list && reader->page_list->len > 0)
	g_array_sort(reader->page_list, g_archive_reader_translate_strategy(strategy));
}

/* class instance housekeeping stuff */

static void
g_archive_reader_unzip_interface_init(gpointer g_iface,
				      gpointer iface_data)
{
    GArchiveReaderClass* klass = (GArchiveReaderClass*)g_iface;
    klass->register_path = register_path;
    klass->get_file_list = get_file_list;
    klass->get_page = get_page;
    klass->get_last_error = get_last_error;
    klass->set_sort_strategy = set_sort_strategy;
}

static void
g_archive_reader_unzip_init(GArchiveReaderUnzip* reader)
{
    reader->magic.offs = 0;
    reader->magic.len = 0;
    reader->magic.fn_offs = 0;
    reader->magic.sign = NULL;
    reader->buf = NULL;
    reader->buf_size = 0;
    reader->current_file = NULL;
    reader->last_error = NULL;
    reader->page_list = NULL;
    reader->strategy = SORT_ASCII;
}

G_DEFINE_TYPE_EXTENDED(GArchiveReaderUnzip,
		       g_archive_reader_unzip,
		       G_TYPE_OBJECT,
		       0,
		       G_IMPLEMENT_INTERFACE(G_TYPE_ARCHIVE_READER,
					     g_archive_reader_unzip_interface_init));

static void
g_archive_reader_unzip_finalize (GObject* object)
{
    GArchiveReaderUnzip* reader = G_ARCHIVE_READER_UNZIP(object);
    if(reader->current_file)
	unzClose(reader->current_file);
    if(reader->last_error)
	g_free(reader->last_error);
    free_page_list(reader);
}

static void
g_archive_reader_unzip_class_init(GArchiveReaderUnzipClass* klass)
{
    GObjectClass* object_class = (GObjectClass*)klass;
    object_class->finalize = g_archive_reader_unzip_finalize;
}

/* public interface implementation */
GArchiveReader* g_archive_reader_unzip_new(guchar* buf,
					   int buf_size,
					   magic_sign magic)
{
    gpointer data = NULL;
    GArchiveReaderUnzip* reader = NULL;

    g_return_val_if_fail(buf, NULL);
    g_return_val_if_fail(buf_size > 0, NULL);

    /* copy properties from the parameters */
    data = g_object_new(G_TYPE_ARCHIVE_READER_UNZIP, NULL);
    reader = G_ARCHIVE_READER_UNZIP(data);
    
    reader->magic = magic;
    reader->current_file = NULL;
    reader->last_error = NULL;
    reader->page_list = NULL;
    reader->strategy = SORT_ASCII;

    reader->buf = buf;
    reader->buf_size = buf_size;

    return G_ARCHIVE_READER(data);
}
