/*
 * garchivereadercommandline.c - routines to manipulate archive files
 * using command line programs.
 *
 * Copyright (C) 2008 Benoit Goudreault-Emond (bgoudreaultemond@gmail.com).
 * Inspired by code from John Coppens (john@jcoppens.com).
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <glib.h>
#include <unistd.h>
#include <sys/types.h>	/* open */
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>	/* wait */
#include <string.h>

#include "garchivereadercommandline.h"

extern int debug;

static gint
compare_pages(gconstpointer s1, gconstpointer s2)
{
    return strcmp(*(const char**)s1, *(const char**)s2);
}

static gboolean extension_handled(const char* filename)
{
    static const char* extensions[] = {
	".jpg", ".jpeg", ".png", ".gif",
	".JPG", ".JPEG", ".PNG", ".GIF"
    };
    static const int nr_extensions = 8;
    
    int i;
    for(i = 0; i < nr_extensions; ++i)
    {
	if(strstr(filename, extensions[i]) != NULL)
	    return TRUE;
    }
    return FALSE;
}

static void set_last_error(GArchiveReaderCommandLine* reader, gchar* data)
{
    if(reader->last_error)
	g_free(reader->last_error);
    reader->last_error = data;
}

static void
free_page_list(GArchiveReaderCommandLine* reader)
{
    int i;

    if(!reader->page_list)
	return;

    for(i = 0; i < reader->page_list->len; ++i)
	g_free(g_array_index(reader->page_list, gchar*, i));
    g_array_free(reader->page_list, TRUE);
    reader->page_list = NULL;
}

static int
register_path(GArchiveReader* reader_iface,
	      const gchar* path)
{
    GArchiveReaderCommandLine* reader = G_ARCHIVE_READER_COMMAND_LINE(reader_iface);
    int check_result = g_archive_reader_check_magic(path, &reader->magic);
    if(debug)
	printf(_("Check result for type %s -> %d\n"), reader->magic.sign, check_result);
    if(check_result <= 0)
	return check_result;

    if(reader->current_filename)
    {
	g_free(reader->current_filename);
	free_page_list(reader);
    }
    reader->current_filename = g_strdup(path);
    return 1;
}

static GArray*
get_file_list(GArchiveReader* reader_iface)
{
    static const char* all_extensions = 
	"\\.jpg\\|\\.jpeg\\|\\.png\\|\\.gif\\|"
	"\\.JPG\\|\\.JPEG\\|\\.PNG\\|\\.GIF";

    int pfd[2], pfd2[2];
    int pid_i, pid_j;
    char *bff = NULL, **names = NULL;
    int i;
    gchar* prev_filename = NULL;
    GIOChannel* channel;
    gsize length;
    GArchiveReaderCommandLine* reader = G_ARCHIVE_READER_COMMAND_LINE(reader_iface);

    if(reader->page_list)
	return reader->page_list;
    
    if(!reader->current_filename)
	return NULL;
    
    
    prev_filename = reader->list_command[reader->list_command_archive_index];
    if(prev_filename)
	g_free(prev_filename);
    reader->list_command[reader->list_command_archive_index] = g_strdup(reader->current_filename);

    if(pipe(pfd) == -1)
    {
	printf(_("pipe failed\n"));
	return NULL;
    }
    if(pipe(pfd2) == -1)
    {
	printf(_("pipe failed\n"));
	close(pfd[0]);
	close(pfd[1]);
	return NULL;
    }

    
    pid_i = fork();
    if(pid_i == -1)
    {
	printf(_("fork failed\n"));
	return NULL;
    }
    /* Child 1: do command */
    else if(pid_i == 0)
    {
	close(pfd[0]);
	close(1);
	if(dup(pfd[1]) != 1)
	{
	    printf(_("dup failed\n"));
	    return NULL;
	}
	close(pfd[1]);
	execvp(reader->list_command[0], reader->list_command);
	/* never reached */
	return NULL;
    }
    else
    {
	pid_j = fork();
	if(pid_j == -1)
	{
	    printf(_("fork failed\n"));
	    return NULL;
	}
        /* child 2: do grep */
	else if(pid_j == 0) 
	{ 
	    close(pfd[1]);
	    close(0); /* close stdin */
	    if(dup(pfd[0]) != 0)
	    {
		printf(_("dup failed\n"));
		return NULL;
	    }
	    close(pfd[0]);
	    close(pfd2[0]);
	    close(1);
	    if(dup(pfd2[1]) != 1)
	    {
		printf(_("dup failed\n"));
		return NULL;
	    }
	    close(pfd2[1]);
	    execlp("grep", "grep", all_extensions, NULL);
	    /* never reached */
	    return NULL;
	}
    }
    /* main process */
    /* close all pipe ends */
    close(pfd[0]);
    close(pfd[1]);
    close(pfd2[1]);

    channel = g_io_channel_unix_new(pfd2[0]);
    g_io_channel_read_to_end(channel, &bff, &length, NULL);
    g_io_channel_unref(channel);
    close(pfd2[0]);
    
    /* wait children */
    waitpid(pid_i, 0, 0);
    waitpid(pid_j, 0, 0);
    
    names = g_strsplit(bff, "\n", 0);
    g_free(bff);
    
    for(i = 0; names[i] != NULL; ++i)
    {
	gchar* filename;
	if(!strlen(names[i]))
	    continue;
	if(!extension_handled(names[i] + reader->magic.fn_offs))
	    continue;
	/* assume 16 elements at first */
	if(!reader->page_list)
	    reader->page_list = g_array_sized_new(FALSE, TRUE, sizeof(gchar*), 16);
	filename = g_strdup(names[i] + reader->magic.fn_offs);
	reader->page_list = g_array_append_val(reader->page_list, filename);
    }
    g_strfreev(names);
    if(!reader->page_list)
    {
	set_last_error(reader, g_strdup_printf(_("No pages in archive '%s'"), reader->current_filename));
    }
    else
    {
	g_array_sort(reader->page_list, compare_pages);
    }
    return reader->page_list;
}

static GdkPixbuf*
get_page(GArchiveReader* reader_iface, 
	 int page_number,
	 gdouble* w,
	 gdouble* h)
{
    const gchar* filename;
    gchar* filename_esc = NULL;
    gchar* prev_filename = NULL;
    int sz;
    int i;
    int index = 0;
    int pid;
    GdkPixbuf* retval = NULL;
    GdkPixbufLoader* loader;
    int pfd[2];
    ssize_t num_bytes;
    GArchiveReaderCommandLine* reader = G_ARCHIVE_READER_COMMAND_LINE(reader_iface);    

    if(page_number < 0 || page_number >= get_file_list(reader_iface)->len)
    {
	set_last_error(reader, g_strdup_printf(_("Page number %d out of range"), page_number));
	return NULL;
    }	
    filename = (const gchar*)g_array_index(get_file_list(reader_iface), gchar*, page_number);

    if(reader->decompress_command_file_needs_escape)
    {
	sz = strlen(filename);
	filename_esc = g_malloc(sz * 2 + 1);
	for(i = 0; i < sz; ++i)
	{
	    const char c = filename[i];
	    switch(c)
	    {
	    case '[':
	    case ']':
	    case '{':
	    case '}':
	    case '"':
		filename_esc[index++] = '\\';
	    }
	    filename_esc[index++] = c;
	}
	filename_esc[index] = '\0';
	filename = filename_esc;
    }
    
    prev_filename = reader->decompress_command[reader->decompress_command_archive_index];
    if(prev_filename)
	g_free(prev_filename);
    reader->decompress_command[reader->decompress_command_archive_index] = g_strdup(reader->current_filename);
    
    prev_filename = reader->decompress_command[reader->decompress_command_file_index];
    if(prev_filename)
	g_free(prev_filename);
    reader->decompress_command[reader->decompress_command_file_index] = g_strdup(filename);

    if(pipe(pfd) == -1)
    {
	set_last_error(reader, g_strdup(_("Pipe failed")));
	return NULL;
    }

    /* The following patch was received from Mamoru Tasaka, 2008/05/22
       to solve potential security problems with the system() call */
    pid = fork();
    switch(pid)
    {
    case -1:
	set_last_error(reader, g_strdup(_("Fork failed")));
	close(pfd[0]);
	close(pfd[1]);
	return NULL;
    case 0:
	close(pfd[0]);
	close(1);
	if(dup(pfd[1]) != 1)
	{
	    return NULL;
	}
	close(pfd[1]);
	execvp(reader->decompress_command[0],
	       reader->decompress_command);
	/* never reached */
	return NULL;
    default:
	close(pfd[1]);
    }
    /* if we are here, we're in the parent process */
    /* create a pixbuf loader */
    loader = gdk_pixbuf_loader_new();
    /* read from buffer in 1KB chunks until we get to the end */
    while((num_bytes = read(pfd[0], reader->buf, reader->buf_size)) > 0)
	gdk_pixbuf_loader_write(loader, reader->buf, num_bytes, NULL);
    close(pfd[0]);
    gdk_pixbuf_loader_close(loader, NULL);
    waitpid(pid, NULL, 0);
    if(filename_esc)
    {
	g_free(filename_esc);
    }
    
    retval = gdk_pixbuf_loader_get_pixbuf(loader);
    if(retval)
    {
	g_object_ref(retval);
	if(w)
	    *w = gdk_pixbuf_get_width(retval);
	if(h)
	    *h = gdk_pixbuf_get_height(retval);
    }
    else
    {
	set_last_error(reader, g_strdup_printf(_("Unable to load page %d"), page_number));
    }
    g_object_unref(loader);
    return retval;
}

static gchar*
get_last_error(GArchiveReader* reader_iface)
{
    GArchiveReaderCommandLine* reader = G_ARCHIVE_READER_COMMAND_LINE(reader_iface);
    gchar* last_error = reader->last_error;
    reader->last_error = NULL;
    return last_error;
}

/* class instance housekeeping stuff */

static void
g_archive_reader_command_line_interface_init(gpointer g_iface,
					     gpointer iface_data)
{
    GArchiveReaderClass* klass = (GArchiveReaderClass*)g_iface;
    klass->register_path = register_path;
    klass->get_file_list = get_file_list;
    klass->get_page = get_page;
    klass->get_last_error = get_last_error;
}

static void
g_archive_reader_command_line_init(GArchiveReaderCommandLine* reader)
{
    reader->list_command = NULL;
    reader->list_command_archive_index = 0;
    reader->decompress_command = NULL;
    reader->decompress_command_archive_index = 0;
    reader->decompress_command_file_index = 0;
    reader->decompress_command_file_needs_escape = FALSE;
    reader->buf = NULL;
    reader->buf_size = 0;
    reader->magic.offs = 0;
    reader->magic.len = 0;
    reader->magic.fn_offs = 0;
    reader->magic.sign = NULL;
    reader->current_filename = NULL;
    reader->last_error = NULL;
    reader->page_list = NULL;
}

G_DEFINE_TYPE_EXTENDED(GArchiveReaderCommandLine,
		       g_archive_reader_command_line,
		       G_TYPE_OBJECT,
		       0,
		       G_IMPLEMENT_INTERFACE(G_TYPE_ARCHIVE_READER,
					     g_archive_reader_command_line_interface_init));

static void
g_archive_reader_command_line_finalize (GObject* object)
{
    GArchiveReaderCommandLine* reader = G_ARCHIVE_READER_COMMAND_LINE(object);
    g_strfreev(reader->list_command);
    g_strfreev(reader->decompress_command);
    if(reader->current_filename)
	g_free(reader->current_filename);
    if(reader->last_error)
	g_free(reader->last_error);
    free_page_list(reader);
}

static void
g_archive_reader_command_line_class_init(GArchiveReaderCommandLineClass* klass)
{
    GObjectClass* object_class = (GObjectClass*)klass;
    object_class->finalize = g_archive_reader_command_line_finalize;
}

/* public interface implementation */
GArchiveReader* g_archive_reader_command_line_new(guchar* buf,
						  int buf_size,
						  magic_sign magic,
						  gchar** list_command,
						  gchar** decompress_command)
{
    gpointer data = NULL;
    GArchiveReaderCommandLine* reader = NULL;
    gchar** iterator;
    int index;

    g_return_val_if_fail(list_command, NULL);
    g_return_val_if_fail(decompress_command, NULL);
    g_return_val_if_fail(buf, NULL);
    g_return_val_if_fail(buf_size > 0, NULL);
    /* copy properties from the parameters */
    data = g_object_new(G_TYPE_ARCHIVE_READER_COMMAND_LINE, NULL);
    reader = G_ARCHIVE_READER_COMMAND_LINE(data);
    
    reader->magic = magic;
    reader->current_filename = NULL;
    reader->last_error = NULL;
    reader->page_list = NULL;

    reader->buf = buf;
    reader->buf_size = buf_size;
    
    reader->list_command = g_strdupv(list_command);
    reader->decompress_command = g_strdupv(decompress_command);
    
    for(iterator = reader->list_command, index = 0; *iterator; ++iterator, ++index)
    {
	if(!strcmp("%a", *iterator))
	{
	    reader->list_command_archive_index = index;
	    break;
	}
    }

    for(iterator = reader->decompress_command, index = 0; *iterator; ++iterator, ++index)
    {
	if(!strcmp("%a", *iterator))
	    reader->decompress_command_archive_index = index;
	else if(!strcmp("%f", *iterator))
	{
	    reader->decompress_command_file_index = index;
	    reader->decompress_command_file_needs_escape = FALSE;
	}
	else if(!strcmp("%F", *iterator))
	{
	    reader->decompress_command_file_index = index;
	    reader->decompress_command_file_needs_escape = TRUE;
	}
    }

    return G_ARCHIVE_READER(data);
}
