#include "support.h"
#include "conf.h"
#include "main.h"
#include "global.h"
#include <glib/gstring.h>
#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>
#include <string.h>
#include <stdlib.h>
#ifdef USE_HILDON
#    include <hildon/hildon-program.h>
#    define DEFAULT_WIDTH 700
#    define DEFAULT_HEIGHT 480
#    define DEFAULT_BUFFER_SIZE (10*1024)
#else
#    define DEFAULT_WIDTH 480
#    define DEFAULT_HEIGHT 640
#    define DEFAULT_BUFFER_SIZE (1024*1024)
#endif

GtkListStore	*accel_store = NULL;
GtkListStore    *special_accel_store = NULL;

int		debug = 0;
double          zoom_factor = 1.0;

typedef struct {
  char	*legend,
	*widget;
  gint	 mods;
  guint	 accel;
} key_assignment;

#define DEFAULT_ZONE_SIZE 60

#ifdef USE_HILDON
key_assignment key_asg0[] = {
  {N_("Zoom in"),	"zoom_in_btn",    0, HILDON_HARDKEY_INCREASE},
  {N_("Zoom out"),	"zoom_out_btn",   0, HILDON_HARDKEY_DECREASE},
  {N_("Fit width"),	"zoom_width_btn", 0, 0},
  {N_("Fit page"),	"zoom_fit_btn",   0, 0},
  {N_("Zoom 1:1"),	"zoom_orig_btn",  0, GDK_z},
  {N_("Next page"),	"next_btn",       0, HILDON_HARDKEY_DOWN},
  {N_("Previous page"),	"back_btn",       0, HILDON_HARDKEY_UP},
  {N_("Last page"),	"last_btn",       0, GDK_l},
  {N_("First page"),	"begin_btn",      0, GDK_f},
  {N_("Open file"),	"open_btn",       0, 0},
  {N_("Quit cbrPager"),	"quit_btn",       0, 0},
  {N_("Rotate"),	"rotate_btn",     0, HILDON_HARDKEY_ESC},
  {N_("Full screen"),   "fullscreen_btn", 0, HILDON_HARDKEY_FULLSCREEN},
  {N_("Toggle mode"),   "toggle_keymode", 0, HILDON_HARDKEY_SELECT},
  {N_("Scroll down"),   "scroll_down",    0, 0},
  {N_("Scroll up"),     "scroll_up",      0, 0},
  {N_("Scroll right"),  "scroll_right",   0, 0},
  {N_("Scroll left"),   "scroll_left",    0, 0},
  {N_("Page down"),     "page_down",      0, HILDON_HARDKEY_RIGHT},
  {N_("Page up"),       "page_up",        0, HILDON_HARDKEY_LEFT},
};

key_assignment key_asg1[] = {
  {N_("Zoom in"),	"zoom_in_btn",    0, 0},
  {N_("Zoom out"),	"zoom_out_btn",   0, 0},
  {N_("Fit width"),	"zoom_width_btn", 0, 0},
  {N_("Fit page"),	"zoom_fit_btn",   0, 0},
  {N_("Zoom 1:1"),	"zoom_orig_btn",  0, GDK_z},
  {N_("Next page"),	"next_btn",       0, HILDON_HARDKEY_INCREASE},
  {N_("Previous page"),	"back_btn",       0, HILDON_HARDKEY_DECREASE},
  {N_("Last page"),	"last_btn",       0, GDK_l},
  {N_("First page"),	"begin_btn",      0, GDK_f},
  {N_("Open file"),	"open_btn",       0, 0},
  {N_("Quit cbrPager"),	"quit_btn",       0, 0},
  {N_("Rotate"),	"rotate_btn",     0, HILDON_HARDKEY_ESC},
  {N_("Full screen"),   "fullscreen_btn", 0, HILDON_HARDKEY_FULLSCREEN},
  {N_("Toggle mode"),   "toggle_keymode", 0, HILDON_HARDKEY_SELECT},
  {N_("Scroll down"),   "scroll_down",    0, HILDON_HARDKEY_DOWN},
  {N_("Scroll up"),     "scroll_up",      0, HILDON_HARDKEY_UP},
  {N_("Scroll right"),  "scroll_right",   0, HILDON_HARDKEY_RIGHT},
  {N_("Scroll left"),   "scroll_left",    0, HILDON_HARDKEY_LEFT},
  {N_("Page down"),     "page_down",      0, 0},
  {N_("Page up"),       "page_up",        0, 0},
};

key_assignment* key_asg = key_asg0;
   
#else
key_assignment key_asg0[] = {
  {N_("Zoom in"),	"zoom_in_btn",	  0, GDK_KP_Add},
  {N_("Zoom out"),	"zoom_out_btn",	  0, GDK_KP_Subtract},
  {N_("Fit width"),	"zoom_width_btn", 0, GDK_w},
  {N_("Fit page"),	"zoom_fit_btn",   0, GDK_p},
  {N_("Zoom 1:1"),	"zoom_orig_btn",  0, GDK_1},
  {N_("Next page"),	"next_btn",       0, GDK_Page_Down},
  {N_("Previous page"),	"back_btn",       0, GDK_Page_Up},
  {N_("Last page"),	"last_btn",       0, GDK_End},
  {N_("First page"),	"begin_btn",      0, GDK_Home},
  {N_("Open file"),	"open_btn",       0, GDK_o},
  {N_("Quit cbrPager"),	"quit_btn",       0, GDK_q},
  {N_("Rotate"),	"rotate_btn",     0, GDK_r},
  {N_("Full screen"),   "fullscreen_btn", 0, GDK_F11},
  
  {N_("Scroll down"),   "scroll_down",    0, GDK_Down},
  {N_("Scroll up"),     "scroll_up",      0, GDK_Up},
  {N_("Scroll right"),  "scroll_right",   0, GDK_Right},
  {N_("Scroll left"),   "scroll_left",    0, GDK_Left},
  {N_("Page down"),     "page_down",      GDK_SHIFT_MASK, GDK_Down},
  {N_("Page up"),       "page_up",        GDK_SHIFT_MASK, GDK_Up},
};

key_assignment* key_asg = key_asg0;
#endif

#define nr_keys_assigned (sizeof(key_asg0)/sizeof(key_assignment))

preferences pref = {
  ""
};

#define KEY_FMT "/%s/accel/%s_key"
#define MODS_FMT "/%s/accel/%s_mods"
#define KEY_FMT_ALT "/%s/accel/%s_key_alt"
#define MODS_FMT_ALT "/%s/accel/%s_mods_alt"

static void
save_accels(GConfClient* client, 
	    const char* key_fmt, 
	    const char* mod_fmt, 
	    key_assignment* keys)
{
  gchar path[120];
  int ka;
  
  for (ka = 0; ka < nr_keys_assigned; ka++) {
    g_sprintf(path, mod_fmt, prog_name, keys[ka].widget);
    gconf_client_set_int(client, path, keys[ka].mods, NULL);
    g_sprintf(path, key_fmt, prog_name, keys[ka].widget);
    gconf_client_set_int(client, path, keys[ka].accel, NULL);
  }
}

void
save_config(void)
{
  gchar path[120];
  int ka;
  GConfClient* client;

  client = gconf_client_get_default();

  g_sprintf(path, "/%s/global/lastbook", prog_name);
  gconf_client_set_string(client, path, pref.lastbook, NULL);

  g_sprintf(path, "/%s/global/fixed_dir", prog_name);
  gconf_client_set_string(client, path, pref.fixeddir, NULL);

  g_sprintf(path, "/%s/global/nextpage", prog_name);
  gconf_client_set_int(client, path, pref.nextpage, NULL);

  g_sprintf(path, "/%s/global/prevpage", prog_name);
  gconf_client_set_int(client, path, pref.prevpage, NULL);

  g_sprintf(path, "/%s/global/nrpages", prog_name);
  gconf_client_set_int(client, path, pref.nrpages, NULL);

  g_sprintf(path, "/%s/startup/width", prog_name);
  gconf_client_set_int(client, path, pref.initwidth, NULL);

  g_sprintf(path, "/%s/startup/height", prog_name);
  gconf_client_set_int(client, path, pref.initheight, NULL);

  g_sprintf(path, "/%s/global/nav_pos", prog_name);
  gconf_client_set_int(client, path, pref.nav_pos, NULL);

  g_sprintf(path, "/%s/global/nav_hor", prog_name);
  gconf_client_set_int(client, path, pref.nav_hor, NULL);

  g_sprintf(path, "/%s/global/dirmode", prog_name);
  gconf_client_set_int(client, path, pref.dirmode, NULL);

  g_sprintf(path, "/%s/global/nav_hidden", prog_name);
  gconf_client_set_int(client, path, pref.nav_hidden, NULL);

  g_sprintf(path, "/%s/global/pagenumber", prog_name);
  gconf_client_set_int(client, path, pref.pagenumber, NULL);

  g_sprintf(path, "/%s/global/manga_mode", prog_name);
  gconf_client_set_int(client, path, pref.manga_mode, NULL);

  g_sprintf(path, "/%s/global/auto_rotate", prog_name);
  gconf_client_set_int(client, path, pref.auto_rotate, NULL);

  g_sprintf(path, "/%s/global/zone_size", prog_name);
  gconf_client_set_int(client, path, pref.zone_size, NULL);

  g_sprintf(path, "/%s/global/buffer_size", prog_name);
  gconf_client_set_int(client, path, pref.buffer_size, NULL);

  g_sprintf(path, "/%s/global/scrollbars_fullscreen", prog_name);
  gconf_client_set_int(client, path, pref.scrollbars_fullscreen, NULL);

  g_sprintf(path, "/%s/global/rotate_ccw", prog_name);
  gconf_client_set_int(client, path, pref.rotate_ccw, NULL);

  save_accels(client, KEY_FMT, MODS_FMT, key_asg0);
#if USE_HILDON
  save_accels(client, KEY_FMT_ALT, MODS_FMT_ALT, key_asg1);
#endif

  gconf_client_suggest_sync(client, NULL);
  g_object_unref(G_OBJECT(client));
}

gchar* gconf_client_get_string_with_default(GConfClient* client, const gchar* key, const gchar* def)
{
  gchar* retval;
  retval = gconf_client_get_string(client, key, NULL);
  if(!retval) {
    /* dup to ensure caller always needs to free */
    retval = g_strdup(def);
  }
  return retval;
}

int gconf_client_get_int_with_default(GConfClient* client, const gchar* key, int def)
{
  int retval;
  retval = gconf_client_get_int(client, key, NULL);
  if(!retval)
    retval = def;
  return retval;
}

static void
load_accels(GConfClient* client, const char* key_fmt, const char* mod_fmt, key_assignment* keys)
{
  gchar path[120];
  int ka;
  for (ka = 0; ka < nr_keys_assigned; ka++) {
    g_sprintf(path, mod_fmt, prog_name, keys[ka].widget);
    keys[ka].mods = gconf_client_get_int_with_default(client, path, keys[ka].mods);
    g_sprintf(path, key_fmt, prog_name, keys[ka].widget);
    keys[ka].accel = gconf_client_get_int_with_default(client, path, keys[ka].accel);
  }
}

void
load_config(void)
{
  gchar path[120], *p;
  int ka;
  GConfClient* client;

  client = gconf_client_get_default();
  gconf_client_suggest_sync(client, NULL);

  g_sprintf(path, "/%s/global/lastbook", prog_name);
  strcpy(pref.lastbook, p = gconf_client_get_string_with_default(client, path, ""));
  g_free(p);

  g_sprintf(path, "/%s/global/fixed_dir", prog_name);
  strcpy(pref.fixeddir, p = gconf_client_get_string_with_default(client, path, ""));
  g_free(p);

  g_sprintf(path, "/%s/global/nextpage", prog_name);
  pref.nextpage = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/prevpage", prog_name);
  pref.prevpage = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/nrpages", prog_name);
  pref.nrpages = gconf_client_get_int_with_default(client, path, 1);
    
  g_sprintf(path, "/%s/startup/width", prog_name);
  pref.initwidth = gconf_client_get_int_with_default(client, path, DEFAULT_WIDTH);

  g_sprintf(path, "/%s/startup/height", prog_name);
  pref.initheight = gconf_client_get_int_with_default(client, path, DEFAULT_HEIGHT);

  g_sprintf(path, "/%s/global/nav_pos", prog_name);
  pref.nav_pos = gconf_client_get_int_with_default(client, path, NAV_SE);

  g_sprintf(path, "/%s/global/nav_hor", prog_name);
  pref.nav_hor = gconf_client_get_int_with_default(client, path, FALSE);

  g_sprintf(path, "/%s/global/dirmode", prog_name);
  pref.dirmode = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/nav_hidden", prog_name);
  pref.nav_hidden = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/pagenumber", prog_name);
  pref.pagenumber = gconf_client_get_int_with_default(client, path, 0);
  
  g_sprintf(path, "/%s/global/manga_mode", prog_name);
  pref.manga_mode = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/auto_rotate", prog_name);
  pref.auto_rotate = gconf_client_get_int_with_default(client, path, 0);

  g_sprintf(path, "/%s/global/zone_size", prog_name);
  pref.zone_size = gconf_client_get_int_with_default(client, path, DEFAULT_ZONE_SIZE);
  
  g_sprintf(path, "/%s/global/buffer_size", prog_name);
  pref.buffer_size = gconf_client_get_int_with_default(client, path, DEFAULT_BUFFER_SIZE);

  g_sprintf(path, "/%s/global/scrollbars_fullscreen", prog_name);
  pref.scrollbars_fullscreen = gconf_client_get_int_with_default(client, path, 1);

  g_sprintf(path, "/%s/global/rotate_ccw", prog_name);
  pref.rotate_ccw = gconf_client_get_int_with_default(client, path, 0);


  load_accels(client, KEY_FMT, MODS_FMT, key_asg0);
#if USE_HILDON
  load_accels(client, KEY_FMT_ALT, MODS_FMT_ALT, key_asg1);
#endif

  g_object_unref(G_OBJECT(client));
}


void
show_config(GtkWidget *ref)
{
  GtkWidget *w;
  char bff[20];

  w = lookup_widget(ref, "conf_goto_entry");
  gtk_entry_set_text(GTK_ENTRY(w), pref.fixeddir);

  switch (pref.nextpage) {
    case 0: w = lookup_widget(ref, "fwd_top_rbtn"); break;
    case 1: w = lookup_widget(ref, "fwd_btm_rbtn"); break;
    case 2: w = lookup_widget(ref, "fwd_stay_rbtn"); break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  switch (pref.prevpage) {
    case 0: w = lookup_widget(ref, "back_top_rbtn"); break;
    case 1: w = lookup_widget(ref, "back_btm_rbtn"); break;
    case 2: w = lookup_widget(ref, "back_stay_rbtn"); break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  switch (pref.nrpages) {
    case 1: w = lookup_widget(ref, "conf_1page_rbtn"); break;
    case 2: w = lookup_widget(ref, "conf_2page_rbtn"); break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  sprintf(bff, "%d", pref.initwidth);
  w = lookup_widget(ref, "conf_w_entry");
  gtk_entry_set_text(GTK_ENTRY(w), bff);

  sprintf(bff, "%d", pref.initheight);
  w = lookup_widget(ref, "conf_h_entry");
  gtk_entry_set_text(GTK_ENTRY(w), bff);

  switch (pref.dirmode) {
    case 0: w = lookup_widget(ref, "conf_prevdir_rbtn"); break;
    case 1: w = lookup_widget(ref, "conf_goto_rbtn"); break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  w = lookup_widget(ref, "manga_mode");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), pref.manga_mode ? TRUE : FALSE);
  w = lookup_widget(ref, "auto_rotate");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), pref.auto_rotate ? TRUE : FALSE);
  w = lookup_widget(ref, "scrollbars_fullscreen");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), pref.scrollbars_fullscreen ? TRUE : FALSE);
  w = lookup_widget(ref, "rotate_ccw");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), pref.rotate_ccw ? TRUE : FALSE);


#if !USE_HILDON
  switch (pref.nav_pos) {
    case NAV_NE: w = lookup_widget(ref, "conf_nav_ne_rbtn"); break;
    case NAV_SE: w = lookup_widget(ref, "conf_nav_se_rbtn"); break;
    case NAV_NW: w = lookup_widget(ref, "conf_nav_nw_rbtn"); break;
    case NAV_SW: w = lookup_widget(ref, "conf_nav_sw_rbtn"); break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

  w = lookup_widget(ref, "conf_nav_hide_cbtn");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), pref.nav_hidden);

  switch (pref.nav_hor) {
    case FALSE:
      w = lookup_widget(ref, "conf_nav_vert_rbtn"); 
      break;
    case TRUE:
      w = lookup_widget(ref, "conf_nav_hor_rbtn");
      break;
  }
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
#endif
}


static void
save_accelerators(GtkListStore* store, key_assignment* keys) 
{
  guint accel;
  gint mods;
  int ka;
  GtkTreeIter iter;

  /* ensure we don't zap configuration with b0rked model values */
  if(GTK_IS_TREE_MODEL(store)) {
    gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
    for (ka = 0; ka < nr_keys_assigned; ka++) {
      gtk_tree_model_get(GTK_TREE_MODEL(store), &iter,
			 ACC_MODS, &mods,
			 ACC_ACCEL, &accel,
			 -1);
      keys[ka].mods = mods;
      keys[ka].accel = accel;
      gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter);
    }
  }
}

void
parse_config(GtkWidget *ref)
{
  GtkWidget* w;

  w = lookup_widget(ref, "conf_goto_entry");
  strncpy(pref.fixeddir,
		gtk_entry_get_text(GTK_ENTRY(w)),
		sizeof(pref.fixeddir));

  w = lookup_widget(ref, "fwd_top_rbtn");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    pref.nextpage = 0;
  else {
    w = lookup_widget(ref, "fwd_btm_rbtn");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
      pref.nextpage = 1;
    else
      pref.nextpage = 2;
  }

  w = lookup_widget(ref, "back_top_rbtn");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    pref.prevpage = 0;
  else {
    w = lookup_widget(ref, "back_btm_rbtn");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
      pref.prevpage = 1;
    else
      pref.prevpage = 2;
  }

  w = lookup_widget(ref, "conf_1page_rbtn");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    pref.nrpages = 1;
  else
    pref.nrpages = 2;

  w = lookup_widget(ref, "conf_w_entry");
  pref.initwidth = atol(gtk_entry_get_text(GTK_ENTRY(w)));

  w = lookup_widget(ref, "conf_h_entry");
  pref.initheight = atol(gtk_entry_get_text(GTK_ENTRY(w)));

  w = lookup_widget(ref, "conf_prevdir_rbtn");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    pref.dirmode = DIRM_LAST;
  else
    pref.dirmode = DIRM_FIXED;

  w = lookup_widget(ref, "manga_mode");
  pref.manga_mode = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = lookup_widget(ref, "auto_rotate");
  pref.auto_rotate = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = lookup_widget(ref, "scrollbars_fullscreen");
  pref.scrollbars_fullscreen = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

  w = lookup_widget(ref, "rotate_ccw");
  pref.rotate_ccw = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));

#if !USE_HILDON
  w = lookup_widget(ref, "conf_nav_nw_rbtn");
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    pref.nav_pos = NAV_NW;
  else {
    w = lookup_widget(ref, "conf_nav_ne_rbtn");
    if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
      pref.nav_pos = NAV_NE;
    else {
      w = lookup_widget(ref, "conf_nav_se_rbtn");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
        pref.nav_pos = NAV_SE;
      else
        pref.nav_pos = NAV_SW;
    }
  }
  
  w = lookup_widget(ref, "conf_nav_hide_cbtn");
  pref.nav_hidden = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  if (pref.nav_hidden)
    gtk_widget_hide(nav_window);
  else
    gtk_widget_show(nav_window);
  w = lookup_widget(ref, "conf_nav_hor_rbtn");
  pref.nav_hor = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  w = lookup_widget(nav_window, "nav_toolbar");
  gtk_toolbar_set_orientation(GTK_TOOLBAR(w),
	pref.nav_hor ? GTK_ORIENTATION_HORIZONTAL
		     : GTK_ORIENTATION_VERTICAL);
#endif

  save_accelerators(accel_store, key_asg0);
#if USE_HILDON
  save_accelerators(special_accel_store, key_asg1);
#endif
}


void
conf_accel_tview_cursor_changed(void)
{
}

/* methods used for tree view model management */
static void
store_accel_model(GtkListStore* store,
		  GtkTreeIter* iter,
		  const gchar* legend,
		  guint accel_key,
		  gint mods)
{
  if(legend)
    gtk_list_store_set(store, iter,
		       ACC_FUNCTION, legend,
		       -1);
  gtk_list_store_set(store, iter,
		     ACC_MODS, mods,
		     ACC_ACCEL, accel_key,
		     ACC_DISABLE, (mods == 0 && accel_key == 0 ? TRUE : FALSE),
		     ACC_ESCAPE, (mods == 0 && accel_key == GDK_Escape ? TRUE : FALSE),
		     -1);
}
		  

static void
update_accel_model(GtkTreeView *treeview, gchar *path,
		   guint accel_key,
		   GdkModifierType mask)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreePath *tpath;

  model = gtk_tree_view_get_model(treeview);
  if (gtk_tree_model_get_iter_from_string(model, &iter, path)) {
    store_accel_model(GTK_LIST_STORE(model), &iter,
		      NULL, 
		      accel_key, 
		      (gint)mask);
    tpath = gtk_tree_path_new_from_string(path);
    gtk_tree_model_row_changed(model, tpath, &iter);
    gtk_tree_path_free(tpath);
  }
}

static void
accel_edited(GtkCellRendererAccel *renderer, gchar *path, guint accel_key,
		GdkModifierType mask, guint hardware_keycode,
		gpointer treeview)
{
  update_accel_model(GTK_TREE_VIEW(treeview), path, accel_key, mask);
}


static void
disable_accel_toggle(GtkCellRendererToggle *renderer,
		     gchar *path,
		     gpointer treeview)
{
  update_accel_model(GTK_TREE_VIEW(treeview), path, 0, (GdkModifierType)0);
}

static void
escape_accel_toggle(GtkCellRendererToggle *renderer,
		    gchar *path,
		    gpointer treeview)
{
  update_accel_model(GTK_TREE_VIEW(treeview), path, GDK_Escape, (GdkModifierType)0);
}


static void
conf_accel_tview_do_realize(GtkTreeView *tview,
			    GtkListStore **store,
			    key_assignment* keys) 
{
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *col;
  GtkTreeIter iter;
  int ka;

  *store = gtk_list_store_new(ACC_COLS,
			      G_TYPE_STRING,
			      G_TYPE_INT,
			      G_TYPE_UINT,
			      G_TYPE_BOOLEAN,
			      G_TYPE_BOOLEAN);
  
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tview), FALSE);
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tview), TRUE);
  gtk_tree_view_set_model(GTK_TREE_VIEW(tview),
			  GTK_TREE_MODEL(*store));
  g_object_unref(G_OBJECT(*store));

  /* Accelerator function */
  renderer = gtk_cell_renderer_text_new();
  gtk_object_set(GTK_OBJECT(renderer),
		"ypad", 0, NULL);
  col = gtk_tree_view_column_new_with_attributes(_("Function"), 
						 renderer,
						 "text", 
						 0, 
						 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tview), col);

  /* Accelerator assignment */
  renderer = gtk_cell_renderer_accel_new();
  g_object_set(renderer, "accel-mode", GTK_CELL_RENDERER_ACCEL_MODE_OTHER,
		"editable", TRUE, NULL);
  g_signal_connect(G_OBJECT(renderer), "accel_edited",
		   G_CALLBACK(accel_edited),
		   tview);
  col = gtk_tree_view_column_new_with_attributes(_("Accelerator"), 
						 renderer,
						 "accel-mods", 
						 ACC_MODS,
						 "accel-key", 
						 ACC_ACCEL,
						 NULL);
  g_object_set(col, "expand", TRUE, NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tview), col);

  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_signal_connect(G_OBJECT(renderer), "toggled",
		   G_CALLBACK(disable_accel_toggle),
		   tview);
  col = gtk_tree_view_column_new_with_attributes(_("Disable"),
						 renderer,
						 "active",
						 ACC_DISABLE,
						 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tview), col);
  
  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_signal_connect(G_OBJECT(renderer), "toggled",
		   G_CALLBACK(escape_accel_toggle),
		   tview);
  col = gtk_tree_view_column_new_with_attributes(_("Escape"),
						 renderer,
						 "active",
						 ACC_ESCAPE,
						 NULL);
  gtk_tree_view_append_column(GTK_TREE_VIEW(tview), col);

  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tview), TRUE);

  for (ka = 0; ka < nr_keys_assigned; ka++) {
    gtk_list_store_append(*store, &iter);
    store_accel_model(*store, &iter,
		      keys[ka].legend,
		      keys[ka].accel,
		      keys[ka].mods);
  }  
}

void
conf_accel_tview_realize(GtkTreeView *tview, gboolean main_keys)
{
#if USE_HILDON
  if(main_keys)
    conf_accel_tview_do_realize(tview, &accel_store, key_asg0);
  else
    conf_accel_tview_do_realize(tview, &special_accel_store, key_asg1);
#else
  conf_accel_tview_do_realize(tview, &accel_store, key_asg0);
#endif
}

const char*
find_accelerator(int keyval, int state)
{
  const char* widget_name = NULL;
  int i;
  for(i = 0; i < nr_keys_assigned; ++i) {
    if(key_asg[i].accel == keyval && ((state == 0 && !key_asg[i].mods) || (state & key_asg[i].mods) != 0))
      return key_asg[i].widget;
  }
  return NULL;
}

gboolean
toggle_accelerator_mode(void)
{
#if USE_HILDON
  if(key_asg == key_asg0) {
    key_asg = key_asg1;
    return TRUE;
  } else {
    key_asg = key_asg0;
    return FALSE;
  }
#else
  return FALSE;
#endif
}
