#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os
from common.carmanconfig import CarmanConfig, METRIC
from models.tripsmodel import TripsModel, SPEED, RPM, ALTITUDE
from models.kbmodel import KeyBoardModel
from main.container import Controller
from main.mainview import MainView
from main.messagedlgctrl import MessageDlgController
from maps.maptilectrl import MapTileController
from maps.trackctrl import TrackController
from trips.optionctrl import OptionController
from trips.graphctrl import GraphController
from trips.tripsview import TripsView
from trips.statsview import StatsView
from trips.pointerevent import PointerEvent

class TripsController(Controller):

    def __init__(self, parent, canvas, theme):
        self.canvas = canvas
        self.screen = None
        self.splited = False
        self._parent = parent
        self.loaded_trip = None
        self.data_type = SPEED
        self.main_view = MainView()
        self.metric = CarmanConfig().get_unit() == METRIC
        self.track_ctrl = TrackController(canvas, trip=True)
        self.graph_ctrl = GraphController(canvas, self)
        self.map_ctrl = MapTileController(self, canvas, self.track_ctrl)
        self.map_ctrl.set_zoom_changed_cb(self.update_zoom_buttons)
        self.model =  TripsModel()
        self.model.add_data_update_cb(self._data_update_cb)
        self.pointer_event = PointerEvent(canvas, self.graph_ctrl)
        self.set_theme(theme)
        lat, lon, zoom = CarmanConfig().get_last_trip_fix()
        self.map_ctrl.set_position(lat, lon, zoom)
        self.kb_model = KeyBoardModel()
        self.option = OptionController(self)
        self.load_current_trip()

    def get_view(self):
        return self.screen

    def get_icon(self):
        return (self.theme_file, "images/icon-trips")

    def set_center_free(self):
        pass

    def show_options(self):
        self.option.show()

    def activate(self):
        self.screen.show()
        self.map_ctrl.activate()
        self.kb_model.add_key_down_cb(self.key_down_cb)

    def deactivate(self):
        self.kb_model.del_key_down_cb(self.key_down_cb)

    def key_down_cb(self, obj, event):
        if self.main_view.get_transiting():
            return False

        if event.key == "Left":
            if self.splited:
                self.map_ctrl.move(-32, 0)
            else:
                self.graph_ctrl.move(32)
        elif event.key == "Right":
            if self.splited:
                self.map_ctrl.move(32, 0)
            else:
                self.graph_ctrl.move(-32)
        elif event.key == "Up":
            if self.splited:
                self.map_ctrl.move(0, -32)
        elif event.key == "Down":
            if self.splited:
                self.map_ctrl.move(0, 32)
        elif event.key == "F4":
            self.show_options()
        elif event.key == "F7":
            self.on_zoom_up()
        elif event.key == "F8":
            self.on_zoom_down()
        else:
            return False

        return True

    def set_theme(self, theme):
        if self.screen:
            self.screen.delete()
        self.theme_file = theme
        first_time = self.screen is None
        self.screen = TripsView(self.canvas, self.theme_file)

        self.graph_ctrl.set_view(theme, self.screen)
        self.track_ctrl.set_view(theme, self.screen)
        self.map_ctrl.set_view(theme, self.screen)
        self.pointer_event.set_view(theme, self.screen)
        if self.splited:
            self.screen.signal_emit("split-screen-update", "")
        if not first_time:
            self.update_zoom_buttons()
            self.map_ctrl.update_position()

        self.screen.signal_callback_add("zoom-up-pressed", "",
            self.on_zoom_up)
        self.screen.signal_callback_add("zoom-down-pressed", "",
            self.on_zoom_down)
        self.screen.signal_callback_add("split-pressed", "",
            self.split_screen)
        self.screen.signal_callback_add("status-pressed", "",
            self.show_stats)
        self.screen.signal_callback_add("split-anim-end", "",
            self._split_end)

        field = self.graph_ctrl.get_field()
        if field == SPEED:
            self.screen.value_set_speed(self.metric)
        elif field == RPM:
            self.screen.value_set_rpm()
        elif field == ALTITUDE:
            self.screen.value_set_altitude()

    def finalize(self):
        lat, lon, zoom = self.map_ctrl.get_position()
        CarmanConfig().set_last_trip_fix(lat,lon, zoom)
        self.map_ctrl.finalize()

    def load_trip(self, path):
        trip = os.path.join(path, "trip.dat")
        if self.graph_ctrl.load_file_trip(trip):
            self.model.del_trip_reseted_cb(self.load_current_trip)
            locations = self.model.get_all_locations()
            if locations:
                self.track_ctrl.add_track(locations)
                latlon = self.model.get_location(self.graph_ctrl.total_time)
                if latlon:
                    self.map_ctrl.set_position(latlon[0], latlon[1])
            else:
                self.track_ctrl.clear()
        else:
            msg = MessageDlgController()
            msg.show_message("Invalid trip file")

    def load_current_trip(self):
        if self.graph_ctrl.load_current_trip():
            self.model.add_trip_reseted_cb(self.load_current_trip)
            locations = self.model.get_all_locations()
            if locations:
                self.track_ctrl.add_track(locations)
                latlon = self.model.get_location(self.graph_ctrl.total_time)
                if latlon:
                    self.map_ctrl.set_position(latlon[0], latlon[1])
            else:
                self.track_ctrl.clear()

    def set_map_position(self, lat, lon):
        if self.splited:
            self.map_ctrl.set_position(lat, lon)
            self.track_ctrl.set_trip_position(lat, lon)

    def update_zoom_buttons(self, *params):
        if self.splited:
            up, down = self.map_ctrl.get_zoom_state()
        else:
            up, down = self.graph_ctrl.get_zoom_state()

        self.screen.update_zoom_buttons(up, down)

    def on_zoom_up(self, *params):
        if self.splited:
            self.map_ctrl.zoom_up()
        else:
            self.graph_ctrl.set_zoom_up()
            self.update_zoom_buttons()

    def on_zoom_down(self, *params):
        if self.splited:
            self.map_ctrl.zoom_down()
        else:
            self.graph_ctrl.set_zoom_down()
            self.update_zoom_buttons()

    def split_screen(self, *params):
        if self.splited:
            self.screen.signal_emit("unsplit-screen", "")
            self.pointer_event.deactivate()
        else:
            self.screen.signal_emit("split-screen", "")
        self.splited = not self.splited
        self.graph_ctrl.split_screen(self.splited)
        self.update_zoom_buttons()

    def _split_end(self, *params):
        self.graph_ctrl.update()
        if self.splited:
            self.pointer_event.activate()

    def scale_set(self, scale):
        min, max = scale
        if not max or not self.screen:
            return
        field = self.graph_ctrl.get_field()
        if field == SPEED and not self.metric:
            max = (max * 62) / 100
        if field == RPM:
            max /= 100
        self.screen.scale_set(min, max)

    def set_unit_system(self, metric):
        self.metric = metric
        if self.graph_ctrl.get_field() == SPEED:
            self.screen.value_set_speed(self.metric)
            self.scale_set(self.graph_ctrl.scale)

    def change_graph_speed(self, *params):
        self.screen.value_set_speed(self.metric)
        self.graph_ctrl.set_data_speed()

    def change_graph_rpm(self, *params):
        self.screen.value_set_rpm()
        self.graph_ctrl.set_data_rpm()

    def change_graph_altitude(self, *params):
        self.screen.value_set_altitude()
        self.graph_ctrl.set_data_altitude()

    def show_stats(self, *params):
        stats_view = StatsView()
        stats_view.show()

    def get_loaded_trip(self):
        return self.loaded_trip

    def set_loaded_trip(self, trip):
        self.loaded_trip = trip

    def _data_update_cb(self, speed, rpm, gps_data):
        if gps_data[0] is not None:
            self.track_ctrl.add_point((gps_data[0], gps_data[1]))

    def get_graph_type(self):
        return self.graph_ctrl.get_field()
