#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os, dbus, time, ecore
from main import trip
from common.singleton import Singleton
from common.carlog import DEBUG, WARNING, ERROR
from ConfigParser import ConfigParser, DEFAULTSECT
from models.obdmodel import OBDModel
from models.gpsmodel import GPSModel
from models.dbusmodel import CarmandDbusModel, DBUS_BUS_NAME, \
    DBUS_OBJECT_PATH,  DBUS_IFACE_TRIP

ALTITUDE, RPM, SPEED = range(3)

class TripsModel(Singleton, dbus.Interface):

    def __init__(self):
        Singleton.__init__(self)
        dbus.Interface.__init__(self, CarmandDbusModel().get_bus_object(),
            DBUS_IFACE_TRIP)
        self.initalized = False
        self.update = False
        self.obdmodel = OBDModel()
        self.gpsmodel = GPSModel()
        self._data_update_cbs = []
        self._trip_reseted_cbs = []

        bus = CarmandDbusModel().get_bus()
        bus.add_signal_receiver(self._trip_reseted,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_TRIP, signal_name = 'TripReseted')
        bus.add_signal_receiver(self._data_flushed,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_TRIP, signal_name = 'DataFlushed')

    def load_file(self, file):
        if self.initalized:
            self.initalized = False
        try:
            trip.load_file(file)
        except Exception, err:
            ERROR("Error loading trip file: %s" % err)
            return

        self.update = False
        self.initalized = True
        return self.initalized

    def load_current_trip(self):
        try:
            folder = self.ActualTripFolder()
        except dbus.exceptions.DBusException, err:
            ERROR("Error in get trip folders: %s" % err)
            return False

        file = os.path.join(folder, "trip.dat")
        self.FlushData()
        up = self.update
        if self.load_file(file):
            if not up:
                try:
                    prec = CarmandDbusModel().GetTimePrecision()
                except dbus.exceptions.DBusException, err:
                    ERROR("Error in get time precision: %s" % err)
                    self.update = False
                    return False

                ecore.timer_add(prec, self.__data_update_cb)
            self.update = True
            return True

    def get_times(self):
        if self.initalized:
            try:
                return trip.get_period()
            except Exception, err:
                DEBUG("Error in get trip times: %s" % err)
        return (0,0)

    def init(self, image):
        try:
            trip.init(image)
        except Exception, err:
            DEBUG("Error in init graph: %s" % err)


    def set_graph(self, field, width, scale, zoom):
        try:
            trip.set_graph(field, width, scale, zoom)
        except Exception, err:
            DEBUG("Error in set graph: %s" % err)

    def set_theme(self, color, line_width, height):
        try:
            trip.set_theme(color, line_width, height)
        except Exception, err:
            DEBUG("Error in set theme: %s" % err)

    def update_graph(self):
        try:
            trip.update_graph()
        except Exception, err:
            DEBUG("Error in update graph: %s" % err)

    def get_speed_stats(self):
        if self.initalized:
            try:
                return trip.get_speed_stats()
            except Exception, err:
                DEBUG("Error in get trip speed avg: %s" % err)

    def get_rpm_stats(self):
        if self.initalized:
            try:
                return trip.get_rpm_stats()
            except Exception, err:
                DEBUG("Error in get trip RPM avg: %s" % err)

    def get_altitude_stats(self):
        if self.initalized:
            try:
                return trip.get_altitude_stats()
            except Exception, err:
                DEBUG("Error in get trip RPM avg: %s" % err)

    def get_max_speed(self):
        stat = self.get_speed_stats()
        if stat:
            return stat[1]

    def get_max_rpm(self):
        stat = self.get_rpm_stats()
        if stat:
            return stat[1]

    def get_max_altitude(self):
        stat = self.get_altitude_stats()
        if stat:
            return stat[1]

    def get_location(self, time):
        if self.initalized:
            try:
                return trip.get_latlon(time)
            except Exception, err:
                DEBUG("Error in get trip location: %s" % err)

    def get_all_locations(self):
        if self.initalized:
            try:
                return trip.get_track()
            except Exception, err:
                DEBUG("Error in get trip track: %s" % err)

    def get_trip_distance(self):
        if self.initalized:
            try:
                return trip.get_trip_distance()
            except Exception, err:
                DEBUG("Error in get trip distance: %s" % err)

    def is_initalized(self):
        return self.initalized

    def is_current(self):
        return self.initalized and self.update

    def add_data_update_cb(self, cb):
        if callable(cb) and cb not in self._data_update_cbs:
            self._data_update_cbs.append(cb)

    def del_data_update_cb(self, cb):
        if cb in self._data_update_cbs:
            self._data_update_cbs.remove(cb)

    def add_trip_reseted_cb(self, cb):
        if callable(cb) and cb not in self._trip_reseted_cbs:
            self._trip_reseted_cbs.append(cb)

    def del_trip_reseted_cb(self, cb):
        if cb in self._trip_reseted_cbs:
            self._trip_reseted_cbs.remove(cb)


    def _data_flushed(self):
        DEBUG("Data flushed received")

    def _trip_reseted(self):
        DEBUG("Trip Reseted received")
        for cb in self._trip_reseted_cbs:
            cb()

    def __data_update_cb(self):
        if not self.update:
            return False
        gps_data = self.gpsmodel.get_last_data()
        speed = self.obdmodel.get_last_data("0D")
        rpm = self.obdmodel.get_last_data("0C")

        if speed:
            speed = int(speed[0])
        elif gps_data:
            speed = int(gps_data[3])

        if rpm:
            rpm = int(rpm[0])

        if not gps_data:
            gps_data = (None, None, None)

        try:
            trip.add_data(gps_data[0], gps_data[1], gps_data[2], rpm, speed)
        except Exception, err:
            DEBUG("Error in add trip data %s" % err)

        for cb in self._data_update_cbs:
            cb(speed, rpm, gps_data)

        return True

    def change_trip_name(self, path, name):
        info = os.path.join(path, "info.txt")
        if os.path.exists(info):
            config = ConfigParser()
            config.read(info)
            config.set(DEFAULTSECT, 'tag', name)
            file = open(info, "w")
            config.write(file)
            file.close()

    def get_trip_info(self, path):
        info = os.path.join(path, "info.txt")
        if os.path.exists(info):
            config = ConfigParser()
            config.read(info)
            epoch = int(config.get(DEFAULTSECT, 'start'))
            date_time = time.strftime("%d %b, %Y <hour>%H:%M",
                    time.localtime(epoch))

            tag = ""
            if config.has_option(DEFAULTSECT, 'tag'):
                tag = config.get(DEFAULTSECT, 'tag').strip()

            return [tag, date_time, path]
