# reposet.py - Set of repositories
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os, sys, math, glob
from common.carlog import ERROR
from common.carmanconfig import CarmanConfig

class RepoBase(object):

    NAME = "Dummy"
    CACHE_NAME = ".dummy"
    TILE_WIDTH = 256
    TILE_HEIGHT = 256
    URI = ""
    ZOOM = range(0, 18)
    DIMENSION_OFFSET = 0
    WORLD_SIZE = (1 << (ZOOM[-1] + DIMENSION_OFFSET)) * TILE_WIDTH
    WORLD_SIZE_360 = WORLD_SIZE / 360.0
    EXTENSION = '.png'
    URBAN_ZOOM = 2
    CITY_ZOOM = 6
    STATE_ZOOM = 12
    AVERAGE_SIZE = 10

    def build_uri(self, tile_x, tile_y, zoom):
        pass
    # build_uri

    def get_downloads_zoom(self):
        return self.URBAN_ZOOM, self.CITY_ZOOM, self.STATE_ZOOM
    # get_downloads_zoom

    def get_name(self):
        return self.NAME
    # get_name

    def get_cache_name(self):
        return self.CACHE_NAME
    # get_cache_name

    def get_extension(self):
        return self.EXTENSION
    # get_extension

    def get_tile_width(self):
        return self.TILE_WIDTH
    # get_tile_width

    def get_tile_height(self):
        return self.TILE_HEIGHT
    # get_tile_height

    def get_zoom(self):
        return self.ZOOM
    # get_zoom

    def get_dimension_offset(self):
        return self.DIMENSION_OFFSET
    # get_dimension_offset

    def get_average_size(self):
        return self.AVERAGE_SIZE
    # get_average_size

    def xy_to_latlon(self, x, y):
        lon = x / self.WORLD_SIZE_360 - 180
        y = y - self.WORLD_SIZE / 2
        radius = self.WORLD_SIZE / (2 * math.pi)
        lat = (math.pi / 2) - 2 * math.atan(math.exp(-1.0 * y / radius))
        return -math.degrees(lat), lon
    # xy_to_latlon

    def latlon_to_xy(self, lat, lon):
        """ Return map position from latitude and longitude """
        x = int(round(self.WORLD_SIZE_360 * (lon + 180)))
        if lat >= 90.0:
            y = 0
        elif lat <= -90.0:
            y = self.WORLD_SIZE
        else:
            tmp = math.sin(math.radians(lat))
            tmp = 0.5 * math.log((1 + tmp) / (1 - tmp))
            tmp = (1 - tmp / math.pi) / 2.0
            y = int(round(tmp * self.WORLD_SIZE))
        return x, y
    # latlon_to_xy


class RepositoriesSet(object):

    def __init__(self):
        self.path = CarmanConfig().get_repository_plugins_path()
    # __init__

    def load_repository(self, name):
        try:
            module = __import__(os.path.join(self.path, name))
            repo = module.get_repository()
            if isinstance(repo, RepoBase):
                return repo
            else:
                DEBUG("repository %s invalid for maps" % name)
        except Exception, e:
            ERROR("error opening repository %s: %s" % (name, e))
    # load_repository

    def load_plugins_description(self):
        description = []
        list_aux = ["__init__"]

        for plugin in glob.glob(os.path.join(self.path, "*.py*")):
            name = os.path.basename(plugin).rsplit('.', 1)[0]

            if name in list_aux:
                continue
            list_aux.append(name)

            repo = self.load_repository(name)
            if repo is not None:
                description.append((repo.get_name(), name))

        description.sort()
        return description
    # load_plugins_description
