#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import dbus
from main.soundctrl import SoundController
from common.carmanconfig import CarmanConfig
from common.carlog import DEBUG
from models.btmodel import BtModel
from common.singleton import Singleton
from main.pairingdlgctrl import PairingDlgController
from models.dbusmodel import CarmandDbusModel, DBUS_BUS_NAME, \
    DBUS_OBJECT_PATH,  DBUS_IFACE_OBD

class HandlePIDs(object):

    def __init__(self, pids, handles, cb):
        self.cb = cb
        self.pids = pids
        self.handles = handles


class OBDModel(Singleton, dbus.Interface):

    DISCONNECTED = "Disconnected"
    CONNECTING   = "Connecting"
    CONNECTED    = "Connected"

    def __init__(self):
        """ OBD Model constructor """
        Singleton.__init__(self)
        carmandbusmodel = CarmandDbusModel()
        dbus.Interface.__init__(self, carmandbusmodel.get_bus_object(),
                        DBUS_IFACE_OBD)

        self._cache = {}
        self._dtc_list_cbs = []
        self._status_changed_cbs = []
        self._data_available_cbs = []
        self._connection_lost_cbs = []
        self._disconnecting = False
        self._add_carmand_listeners(carmandbusmodel.get_bus())

        self.sound_player = None
        self._speed_alert_sound = CarmanConfig().get_speed_alert()
        if (self._speed_alert_sound == "ON"):
            self.play_sound_flag = True
            self.sound_player = SoundController()
            self.enable_speed_alert(CarmanConfig().get_max_speed())
    # __init__

    def _on_dtc_list(self, dtcs):
        for cb in self._dtc_list_cbs:
            cb(dtcs)
    # _on_dtc_list

    def _on_status_changed(self, status):
        """ Called when the carmand change the OBD status connection """
        DEBUG("OBD status changed to %s" % status)
        for cb in self._status_changed_cbs:
            cb(status)
        if status == self.DISCONNECTED:
            self._cache = {}
            if not self._disconnecting:
                for cb in self._connection_lost_cbs:
                    cb()
            self._disconnecting = False
    # _on_connected

    def _on_data_available(self, pid, *data):
        """
        Called when the carmand have information about a sensor to report.
        """
        self._cache["%02X" % pid] = data

        for handle in self._data_available_cbs:
            if pid in handle.pids:
                handle.cb(self, "%02X" % pid, *data)
    # _on_data_available

    def _add_carmand_listeners(self, bus):
        """
        Add interesting listeners
        """
        bus.add_signal_receiver(self._on_dtc_list,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'DTCList')

        bus.add_signal_receiver(self._on_status_changed,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'StatusChanged')

        bus.add_signal_receiver(self._on_data_available,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'DataAvailable')

    def add_dtc_list_cb(self, cb):
        if callable(cb) and cb not in self._dtc_list_cbs:
            self._dtc_list_cbs.append(cb)

    def del_dtc_list_cb(self, cb):
        if cb in self._dtc_list_cbs:
            self._dtc_list_cbs.remove(cb)

    def add_status_changed_cb(self, cb):
        if callable(cb) and cb not in self._status_changed_cbs:
            self._status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        if cb in self._status_changed_cbs:
            self._status_changed_cbs.remove(cb)

    def add_connection_lost_cb(self, cb):
        if callable(cb) and cb not in self._connection_lost_cbs:
            self._connection_lost_cbs.append(cb)

    def del_connection_lost_cb(self, cb):
        if cb in self._connection_lost_cbs:
            self._connection_lost_cbs.remove(cb)

    def add_data_available_cb(self, cmds, cb):
        """
        Aks carmand to get a set of pids from the obd device.
        @param pids: list of OBD pids
        @param callback: a reply function called when data arived.
        """
        if not cmds or not callable(cb) or cb in self._data_available_cbs:
            return

        pids = []
        handles = []
        for cmd in cmds:
            if len(cmd) == 2:
                handle = self.AddTimeSensor(int(cmd[0], 16), cmd[1])
            elif len(cmd) == 3:
                handle = self.AddRoundSensor(int(cmd[0], 16), cmd[1], cmd[2])
            else:
                continue
            pids.append(int(cmd[0], 16))
            handles.append(handle)

        if handles:
            handlepid = HandlePIDs(pids, handles, cb)
            self._data_available_cbs.append(handlepid)
            return handlepid

    def del_data_available_cb(self, handlepid):
        if handlepid in self._data_available_cbs:
            self._data_available_cbs.remove(handlepid)
            for handle in handlepid.handles:
                self.DelSensor(handle)

    def get_last_data(self, pid):
        return self._cache.get(pid)

    def speed_alert_cb(self, model, pid, *data):
        current_speed = data[0]
        if ((self._max_speed - 5) <= 0):
            self.play_sound_flag = True
        else:
            if (current_speed < (self._max_speed - 5)):
                self.play_sound_flag = True

        if (current_speed > self._max_speed) and self.play_sound_flag:
            self.sound_player._play_alert()
            self.play_sound_flag = False

    def enable_speed_alert(self, speed):
        """
        Add speed_alert_cb in callback lists for data_available_cb
        @param speed: speed limit to play sound alert
        """
        if not self.sound_player:
            self.sound_player = SoundController()
        self._max_speed = speed;
        self.speed_handle = self.add_data_available_cb((("0D", 1, 0), ),
            self.speed_alert_cb)

    def disable_speed_alert(self):
        self.del_data_available_cb(self.speed_handle)

    def connect(self, address):
        bt_model = BtModel()
        if not bt_model.has_bonding(address):
            paringctrl = PairingDlgController()
            bt_model.passkey_register(address, paringctrl.show)
        self.ConnectBT()

    def disconnect(self):
        self._disconnecting = True
        self.Disconnect()
