#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import ecore
from common.carmanconfig import CarmanConfig, METRIC
from common.bluetoothadapter import BluetoothAdapter
from main.transition import Transition
from main.statusctrl import StatusController
from main.container import ContainerController
from main.messagedlgctrl import MessageDlgController
from models.connmodel import ConnectionModel
from models.obdmodel import OBDModel
from models.gpsmodel import GPSModel
from models.kbmodel import KeyBoardModel

class MainController(object):

    def __init__(self, view):
        self.view = view
        self.current = 0
        self.transiting = None

        self.bt_adapter = BluetoothAdapter()
        self.bt_adapter.turn_on()

        self.config = CarmanConfig()
        theme = self.config.get_current_theme()
        self.view.setup(self, theme)

        self.canvas = view.get_evas_canvas()
        container = ContainerController(self, self.canvas, theme)
        self.modules = container.load_containers(
            self.config.get_viewer_plugins_path())
        if self.modules:
            self.current_module = self.modules[0]
            self.current_module.activate()
            self.update_icon()
        else:
            self.current_module = None
            msg = MessageDlgController()
            msg.show_message("No plugins found")

        self.gps_model = GPSModel()
        self.gps_model.add_connection_lost_cb(self.gps_connection_lost_cb)
        self.obd_model = OBDModel()
        self.obd_model.add_connection_lost_cb(self.obd_connection_lost_cb)
        self.conn_model = ConnectionModel()
        self.conn_model.add_connection_lost_cb(self.wlan_connection_lost_cb)
        KeyBoardModel().set_default_key_down_cb("Escape", self.exit_button)
    # __init__

    def finalize(self):
        for module in self.modules:
            module.finalize()
        self.bt_adapter.turn_off()
    # finalize

    def set_unit_system(self, unit):
        self.config.set_unit(unit)
        for module in self.modules:
            module.set_unit_system(unit == METRIC)

    def set_theme(self):
        theme = self.config.get_current_theme()
        self.view.setup(self, theme)

        if self.current_module:
            self.current_module.deactivate()
        for module in self.modules:
            module.set_theme(theme)
        if self.current_module:
            self.update_icon()
            self.current_module.activate()

    def exit_button(self, *params):
        dialog = MessageDlgController(confirm_cb=self._exit_cb)
        dialog.show_message("Do you want to exit Carman?", buttons=2)
    # exit_button

    def _exit_cb(self):
        ecore.main_loop_quit()

    def __module_changed(self):
        self.transiting = False
        self.last_module.deactivate()
        self.current_module.activate()
    # __module_changed

    def update_icon(self):
        next_module = self.modules[(self.current + 1) % len(self.modules)]
        self.view.set_icon(next_module.get_icon())
    # update_icon

    def change_button(self, *params):
        if not self.modules or self.transiting:
            return

        self.current = (self.current + 1) % len(self.modules)
        curr = self.modules[self.current]
        if self.current_module != curr:
            self.transiting = True
            transition = Transition(self.canvas,
                                    self.current_module.get_view(),
                                    curr.get_view(), 0.5)
            transition.fade(self.__module_changed)
            self.last_module = self.current_module
            self.current_module = curr
            self.update_icon()
    # change_button

    def option_button(self, *params):
        if self.current_module:
            self.current_module.show_options()
    # option_button

    def status_button(self, *params):
        status = StatusController()
        status.show()
    # status_button

    def gps_connection_lost_cb(self):
        msg = MessageDlgController()
        msg.show_message("GPS Disconnected")

    def obd_connection_lost_cb(self):
        msg = MessageDlgController()
        msg.show_message("OBD2 Disconnected")

    def wlan_connection_lost_cb(self):
        msg = MessageDlgController()
        msg.show_message("Wlan Disconnected")

# MainController
