#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import evas, edje, ecore, time
from common.carlog import DEBUG
from main.mainview import MainView
from models.kbmodel import KeyBoardModel
from common.carmanconfig import CarmanConfig

__TIME_ANIM__ = 0.35

class ListItemView(edje.Edje):

    def __init__(self, controller, canvas, theme, group, label, cb, param, toggled):
        edje.Edje.__init__(self, canvas, file=theme, group=group)
        self.canvas = canvas
        self.checked = False
        self.toggled = toggled
        self.thumbnail = None
        self.controller = controller
        self.set_label(label)
        if callable(cb):
            self.cb = cb
            self.param = param
            self.signal_callback_add("item-pressed", "", self.__item_pressed)
            self.signal_callback_add("item-selected", "", self.__item_selected)

    def __del__(self):
        DEBUG("deleting list item view %s" % self)
        if self.thumbnail is not None:
            self.thumbnail.delete()

    def __item_pressed(self, *params):
        if self.checked and self.toggled:
            self.checked = False
            self.signal_emit("unchecked-button-select", "")
            return

        if not self.checked:
            self.controller.uncheck_items()
        self.controller.signal_emit("show-click-off", "")
        self.signal_emit("start-anime", "")

    def __item_selected(self, *params):
        self.controller.signal_emit("hide-click-off", "")
        self.cb(self.controller, self, self.param)

    def set_label(self, label):
        self.part_text_set("label", label.encode("utf-8"))

    def is_checked(self):
        return self.checked

    def set_checkbox(self, value):
        if self.checked == value:
            return
        self.checked = value
        if value:
            self.signal_emit("checked-button", "")
        else:
            self.signal_emit("unchecked-button", "")

    def set_thumbnail(self, image):
        self.thumbnail = self.canvas.Image(file=image)
        w, h = self.thumbnail.image_size_get()
        self.thumbnail.size_set(w, h)
        self.thumbnail.fill_set(0, 0, w, h)
        self.part_swallow("thumbnail", self.thumbnail)

    def set_tag(self, tag):
        self.part_text_set("tag", tag.encode("utf-8"))


class ListContainer(evas.SmartObject):

    def __init__(self, canvas, parent):
        evas.SmartObject.__init__(self, canvas)
        self.items = []
        self.posx = self.posy = 0
        self.__pos = 0
        self.anim = 0
        self.moving = False
        self._parent = parent
        self.clipper = self.Rectangle()

    def __del__(self):
        DEBUG("deleting list container %s" % self)

    def destroy(self):
        self._parent = None
        self.clear()
        self.delete()

    def clip_set(self, obj):
        self.clipper.clip_set(obj)

    def clip_unset(self):
        self.clipper.clip_unset()

    def resize(self, w, h):
        self.clipper.resize(w, h)

    def show(self):
        if self.items:
            self.clipper.show()

    def hide(self):
        self.clipper.hide()

    def move(self, x, y):
        self.posx = x
        self.posy = y
        evas.SmartObject.move(self, x, y)

    def clear(self):
        for item in self.items:
            self.member_del(item)
            item.delete()
        self.items = []
        self.clipper.hide()

    def add_item(self, item):
        if self.items:
            pos = self.items[-1].pos[1] + self.items[-1].size[1]
        else:
            pos = self.posy
        self.clipper.show()
        self.member_add(item)
        self.items.append(item)
        item.clip_set(self.clipper)
        item.move(self.posx, pos)
        item.show()

    def remove_item(self, item):
        if item not in self.items:
            return False

        height = item.size[1]
        idx = self.items.index(item)
        self.items.remove(item)
        self.member_del(item)
        item.delete()

        if self.items:
            for item in self.items[idx:]:
                posx, posy = item.pos
                item.move(posx, posy - height)
        else:
            self.clipper.hide()

        return True

    def is_moving(self):
        return self.moving

    def uncheck_items(self):
        for item in self.items:
            if item.is_checked:
                item.set_checkbox(False)

    def move_items(self, pos, acce=0):
        self.anim = 0
        self.moving = True
        acce = __TIME_ANIM__ - acce
        self.__pos = pos - self.__pos
        ecore.animator_add(self.anim_move, (time.time(), acce))

    def anim_move(self, data):
        init_anim, acce = data
        part_time = time.time() - init_anim
        anim_current = int(self.__pos * (part_time / acce))

        if abs(anim_current) >= abs(self.__pos):
            self.moving = False
            if self._parent._move is None:
                anim_current = self.__pos
                self.__pos = 0
            else:
                self.__pos = anim_current - self.__pos

        offset = abs(anim_current) - self.anim
        if anim_current < 0:
            offset *= -1

        for item in self.items:
            posx, posy = item.pos
            item.move(posx, posy + offset)
        self.anim = abs(anim_current)

        if not self.moving:
            self._parent.end_anim()
            return False

        return self.moving

    def has_items(self):
        return len(self.items) > 0


class ListView(edje.Edje):

    def __init__(self, screen, item, title):
        self.canvas = MainView().get_evas_canvas()
        self.theme = CarmanConfig().get_current_theme()
        edje.Edje.__init__(self, self.canvas, file=self.theme, group=screen)
        self.group = item
        self.item_pos = 0
        self.item_height = 0
        self.items_size = 0
        self.acce_anin = 0
        self.part_text_set("title", title)

        self.layer_set(20)
        self.container = ListContainer(self.canvas, self)
        self.part_swallow("container", self.container)
        self.container_h = self.container.size[1]

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

        self.signal_callback_add("scroll-up-pressed", "",
                                    self.__mouse_move_up)
        self.signal_callback_add("scroll-down-pressed", "",
                                    self.__mouse_move_down)
        self.signal_callback_add("list-hidden", "", self.destroy)
        self.on_mouse_up_add(self.__on_mouse_up)
        self._move = None

    def __del__(self):
        DEBUG("deleting list view %s" % self)

    def __update_scroll(self):
        if self.item_pos < 0:
            self.signal_emit("enable-scroll-up", "")
        else:
            self.signal_emit("disable-scroll-up", "")
        if self.item_pos + self.items_size > self.container_h:
            self.signal_emit("enable-scroll-down", "")
        else:
            self.signal_emit("disable-scroll-down", "")

    def __on_mouse_up(self, *params):
        self._move = None
        self.acce_anin = 0

    def __mouse_move_up(self, *params):
        self._move = self.__move_up
        self.__move_up()

    def __mouse_move_down(self, *params):
        self._move = self.__move_down
        self.__move_down()

    def __move_up(self, *params):
        if not self.container.is_moving() and self.item_pos < 0:
            self.item_pos += self.item_height
            self.container.move_items(self.item_height, self.acce_anin)
            self.__update_scroll()
            if not self.item_pos < 0:
                self._move = None

    def __move_down(self, *params):
        if not self.container.is_moving() and\
                self.item_pos + self.items_size > self.container_h:
            self.item_pos -= self.item_height
            self.container.move_items(-self.item_height, self.acce_anin)
            self.__update_scroll()
            if not self.item_pos + self.items_size > self.container_h:
                self._move = None

    def end_anim(self):
        if callable(self._move):
            if self.acce_anin < 0.25:
                self.acce_anin += 0.05
            self._move()

    def destroy(self, *params):
        self.container.destroy()
        self.delete()
        self.kb_model.del_key_down_cb(self.__key_down_cb)

    def __key_down_cb(self, obj, event):
        if event.key == "Up":
            self.__move_up()
            return True
        elif event.key == "Down":
            self.__move_down()
            return True
        elif event.key == "Escape":
            self.signal_emit("button-back-pressed", "")
            self.hide()
            return True

        return False

    def hide(self, hide_now=False):
        if hide_now:
            self.signal_emit("list-hidden", "")
        else:
            self.signal_emit("hide-list", "")

    def uncheck_items(self):
        self.container.uncheck_items()

    def add_item(self, label, cb, param=None, check=None, thumbnail=None,
            tag=None, toggled=False):
        item = ListItemView(self, self.canvas, self.theme, self.group, label,
                            cb, param, toggled)
        if check:
            item.set_checkbox(True)
        if thumbnail is not None:
            item.set_thumbnail(thumbnail)
        if tag is not None:
            item.set_tag(tag)

        self.container.add_item(item)

        self.item_height = item.size[1]
        self.items_size += self.item_height
        if self.item_pos + self.items_size > self.container_h:
            self.signal_emit("enable-scroll-down", "")
        return item

    def remove_item(self, item):
        if self.container.remove_item(item):
            self.items_size -= self.item_height
            self.__update_scroll()

    def has_items(self):
        return self.container.has_items()
