#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

from models.obdmodel import OBDModel
from common.carlog import DEBUG
from gauges.troublelist import TroubleListView
from main.messagedlgctrl import MessageDlgController

class ScreenController(object):

    def __init__(self, view):
        """
        Screen constructor
        @param view: Edje screen object
        """
        self.ctrls = {}
        self.view = view
        self.handle = None
        self.signal_added = False
        self.model = OBDModel()

    def __del__(self):
        DEBUG("deleting screen %s" % self)
        for ctrls in self.ctrls.values():
           for ctrl in ctrls:
               ctrl.destroy()
        self.view.delete()

    def reset_values(self):
        for ctrls in self.ctrls.values():
            for ctrl in ctrls:
                ctrl.set_value((0, 0, 0, 0))
        self.set_dtc_button_status(False)

    def __dtc_pressed(self, *param):
        if self.model.Status() == OBDModel.CONNECTED:
            dtc = TroubleListView()
            dtc.show()
        else:
            dlg = MessageDlgController()
            dlg.show_message("OBD2 not connected")

    def set_dtc_button_status(self, visible):
        if visible:
            self.view.signal_emit("enable-dtc-button", "")
        else:
            self.view.signal_emit("disable-dtc-button", "")

    def get_view(self):
        """ Return screen view """
        return self.view

    def add_controller(self, ctrl):
        """
        Add controller in the screen
        @param ctrl: SmartController object
        """
        if ctrl.pid[:2] in self.ctrls:
            self.ctrls[ctrl.pid[:2]].append(ctrl)
        else:
            self.ctrls[ctrl.pid[:2]] = [ctrl]

    def set_unit_system(self, metric):
        """
        Set unit system to metric or imperial
        @param value: True for metric or False for imperial
        """
        for ctrls in self.ctrls.values():
            for ctrl in ctrls:
                ctrl.set_unit_system(metric)
                data = self.model.get_last_data(ctrl.pid[:2])
                ctrl.set_value(data)

    def sensor_data_cb(self, model, pid, *data):
        """
        Change sensor value from OBD
        @param model: OBD model
        @param pid: sensor PID
        @param data: metric and imperial values
        """
        for ctrl in self.ctrls.get(pid, []):
            ctrl.set_value(data)

    # FIXME Check for pids repeated
    def activate(self):
        """ Return all PID commands """
        cmds = []
        for ctrls in self.ctrls.values():
            for ctrl in ctrls:
                data = self.model.get_last_data(ctrl.pid[:2])
                ctrl.set_value(data)
                cmd = ctrl.get_obd_command()
                if cmd not in cmds:
                    cmds.append(cmd)
        if cmds:
            self.handle = self.model.add_data_available_cb(cmds,
                self.sensor_data_cb)

        self.signal_added = True
        self.view.signal_callback_add("dtc-button-pressed", "",
            self.__dtc_pressed)
        self.view.show()

    def deactivate(self):
        if self.signal_added:
            self.signal_added = False
            self.view.signal_callback_del("dtc-button-pressed", "",
                self.__dtc_pressed)
        self.model.del_data_available_cb(self.handle)
        self.handle = None
