#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import math
from gauges.gaugeview import GaugeView
from common.carlog import DEBUG

class ArcBarView(GaugeView):

    def __init__(self, canvas, image=None, pieces=19, initial_angle=225,
      total_angle=270, radius_v=88, radius_h=88, clock_wise="yes"):
        """
        Gauge ArcBar viewer constructor
        @param canvas: Evas canvas object
        @param image: light image key
        @param pieces: number of the images
        @param initial_angle: initial angle
        @param total_angle: total angle
        @param radius_v: vertical radius
        @param radius_h: horizontal radius
        @param clock_wise: orientation
        """
        GaugeView.__init__(self, canvas)
        self.dx = self.dy = 0
        self.images = []
        self.pieces = pieces >= 2 and pieces or 2
        self.initial_angle = initial_angle
        self.total_angle = total_angle
        self.radius_v = radius_v
        self.radius_h = radius_h
        self.clock_wise = clock_wise.lower() == "yes"
        if image is None:
            raise ValueError("Image not specified")
        for i in range(pieces):
            filename = list(image)
            filename[1] += str(i)
            img = self.Image(file=filename)
            sx, sy = img.image_size_get()
            img.size_set(sx, sy)
            img.fill_set(0, 0, sx, sy)
            img.clip_set(self.clipper)
            self.images.append(img)

    def __del__(self):
        DEBUG("deleting arcbar gauge %s" % self)

    def move(self, x, y):
        """ Set images position """
        if self.dx != x or self.dy != y:
            self.dx = x
            self.dy = y
            for i in range(self.pieces):
                degree = i * self.total_angle / (self.pieces - 1)
                if self.clock_wise:
                    degree = self.initial_angle - degree
                else:
                    degree = self.initial_angle + degree
                degree = degree * math.pi / 180
                w, h = self.images[i].image_size_get()
                self.images[i].move(x + int(round(self.radius_h *
                    math.cos(degree))) - w / 2, y - int(round(self.radius_v *
                    math.sin(degree))) - h / 2)

    def set_value(self, value_abs, value_rel):
        """
        Change current value
        @param value_abs: absolute value (not used)
        @param value_rel: relative value from 0.0 to 1.0
        """
        pieces = round(value_rel * self.pieces)
        for i in range(self.pieces):
            if i < pieces:
                self.images[i].show()
            else:
                self.images[i].hide()

