#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os, ecore, shutil
from common.carlog import WARNING
from common.singleton import Singleton
from common.carmanconfig import CarmanConfig
from models.dbusmodel import CarmandDbusModel
from models.tile_downloader import TileDownloader

class MapModel(Singleton):

    def __init__(self):
        Singleton.__init__(self)
        config = CarmanConfig()
        self.repo = None
        self.__map_service_updated_cbs = []
        self.downloader = TileDownloader()
        self.choose_repository(config.get_repository_default_name())
        fd = self.downloader.initialize()
        self.fdh = ecore.FdHandler(fd, ecore.ECORE_FD_READ,
            self.__process_download)
        self.fdh.active_set(ecore.ECORE_FD_READ)

    def __process_download(self, obj, *args, **kargs):
        self.downloader.process_results()
        return True

    def add_map_service_updated(self, cb):
        if callable(cb) and cb not in self.__map_service_updated_cbs:
            self.__map_service_updated_cbs.append(cb)

    def del_map_service_updated(self, cb):
        if cb in self.__map_service_updated_cbs:
            self.__map_service_updated_cbs.remove(cb)

    def cancel_fg_downloads(self):
        self.downloader.clear_fg_queue()

    def cancel_bg_downloads(self):
        self.downloader.clear_bg_queue()

    def cancel_all_downloads(self):
        self.downloader.clear_queue()

    def cancel_download(self, x, y, zoom):
        self.downloader.cancel_download(x, y, zoom)

    def clear_repository_files(self):
        self.cancel_all_downloads()
        cache_dir = self.downloader.get_cache_dir()
        if os.path.exists(cache_dir):
            shutil.rmtree(cache_dir)
            os.makedirs(cache_dir)
        for cb in self.__map_service_updated_cbs:
            cb(self.repo)

    def get_tile_filename(self, x, y, zoom):
        return self.downloader.get_tile_filename(x, y, zoom)

    def request_tile(self, x, y, zoom, priority, cb):
        return self.downloader.request_tile(x, y, zoom, priority, cb)

    def has_fg_downloads(self):
        return self.downloader.has_fg_downloads()

    def has_bg_downloads(self):
        return self.downloader.has_bg_downloads()

    def has_downloads(self):
        return self.downloader.has_downloads()

    def get_repository(self):
        return self.repo

    def choose_repository(self, repository):
        if self.repo:
            self.downloader.clear_queue()
        self.repo = self.downloader.choose_repo(repository)
        self.downloader.set_cache_dir(CarmandDbusModel().GetMapsFolder())
        self.zoom_range = self.repo.get_zoom()
        self.dim_offset = self.repo.get_dimension_offset()
        for cb in self.__map_service_updated_cbs:
            cb(self.repo)

    def get_tile_file_size_avg(self):
        return self.repo.get_average_size()

    def get_tile_size(self):
        return self.repo.get_tile_width(), self.repo.get_tile_height()

    def get_downloads_zoom(self):
        return self.repo.get_downloads_zoom()

    def is_valid_tile(self, x, y, zoom):
        if x < 0 or y < 0 or zoom < self.zoom_range[0] or \
          zoom > self.zoom_range[-1]:
            return False
        dim = 1 << (self.zoom_range[-1] - zoom + self.dim_offset)
        return x < dim and y < dim

    def get_world_size(self, zoom):
        return 256 << (self.zoom_range[-1] - zoom + self.dim_offset)

    def latlon_to_xy(self, lat, lon):
        return self.repo.latlon_to_xy(lat, lon)

    def xy_to_latlon(self, x, y, zoom):
        return self.repo.xy_to_latlon(x << zoom, y << zoom)

    def finalize(self):
        self.downloader.close()
