#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import edje
from common.carmanconfig import CarmanConfig
from main.container import Controller
from main.mainview import MainView
from models.gpsmodel import GPSModel
from models.kbmodel import KeyBoardModel
from models.mapmodel import MapModel
from maps.optionctrl import OptionController
from maps.maptilectrl import MapTileController
from maps.progressbarctrl import ProgressBarController
from maps.trackctrl import TrackController, TrackAuxController
from maps.infosharingctrl import InfoSharingController

class MapsController(Controller):

    def __init__(self, parent, canvas, theme):
        self.gps_ready = False
        self.buddy_connected = False
        self.timer = None
        self.canvas = canvas
        self._parent = parent
        self.screen = None
        self.download_ctrl = None
        self.toggle_fs = False
        self.main_view = MainView()
        self.map_model = MapModel()
        self.map_model.add_map_service_updated(self.check_map_service_zoom)
        self.gps_model = GPSModel()
        self.gps_model.add_status_changed_cb(self.set_gps_status)

        self.track_ctrl = TrackController(canvas, self.gps_model)
        self.track_aux_ctrl = TrackAuxController(canvas)
        self.is_ctrl = InfoSharingController(self, canvas)
        self.is_ctrl.set_buddy_status_changed_cb(self.set_buddy_status)
        self.map_ctrl = MapTileController(self, canvas, self.track_ctrl,
            self.is_ctrl, self.track_aux_ctrl)
        self.map_ctrl.set_zoom_changed_cb(self.update_zoom_buttons)
        self.progressbar_ctrl = ProgressBarController(canvas)

        self.set_theme(theme)
        lat, lon, self.zoom = CarmanConfig().get_last_fix()
        self.map_ctrl.set_position(lat, lon, self.zoom)

        self.kb_model = KeyBoardModel()
        self.option = OptionController(self)

    def check_map_service_zoom(self, repo):
        zoom = self.map_ctrl.zoom
        if zoom is None:
            zoom = self.zoom
        zoom_range = repo.get_zoom()
        if zoom < zoom_range[0]:
            for x in xrange(zoom, zoom_range[0]):
                self.map_ctrl.zoom_down()

        if zoom > zoom_range[-1]:
            for x in xrange(zoom_range[-1], zoom):
                self.map_ctrl.zoom_up()

    def get_view(self):
        return self.screen

    def get_icon(self):
        return (self.theme_file, "images/icon-maps")

    def show_options(self):
        self.option.show()

    def activate(self):
        self.screen.show()
        self.map_ctrl.activate()
        self.kb_model.add_key_down_cb(self.key_down_cb)

    def deactivate(self):
        self.kb_model.del_key_down_cb(self.key_down_cb)

    def key_down_cb(self, obj, event):
        if self.main_view.get_transiting():
            return False

        if event.key == "Left":
            self.map_ctrl.move(-32, 0)
        elif event.key == "Right":
            self.map_ctrl.move(32, 0)
        elif event.key == "Up":
            self.map_ctrl.move(0, -32)
        elif event.key == "Down":
            self.map_ctrl.move(0, 32)
        elif event.key == "F4":
            self.show_options()
        elif event.key == "F6":
            self.fullscreen()
        elif event.key == "F7":
            self.on_zoom_up()
        elif event.key == "F8":
            self.on_zoom_down()
        else:
            return False

        return True

    def set_unit_system(self, value):
        self.is_ctrl.set_unit_system(value)

    def set_theme(self, theme):
        if self.screen:
            self.screen.delete()
        self.theme_file = theme
        first_time = self.screen is None
        self.screen = edje.Edje(self.canvas, file=self.theme_file,
                                group="maps")
        self.screen.size = self.canvas.size

        self.track_ctrl.set_view(theme, self.screen)
        self.track_aux_ctrl.set_view(theme, self.screen)
        self.is_ctrl.set_view(theme, self.screen)
        self.map_ctrl.set_view(theme, self.screen)
        self.progressbar_ctrl.set_view(theme, self.screen)

        if not first_time:
            self.update_zoom_buttons(*self.map_ctrl.get_zoom_state())
            self.map_ctrl.update_position()
        self.set_gps_status(self.gps_model.Status())

        self.screen.signal_callback_add("zoom-up-pressed", "",
                                        self.on_zoom_up)
        self.screen.signal_callback_add("zoom-down-pressed", "",
                                        self.on_zoom_down)
        self.screen.signal_callback_add("goto-gps-pressed", "",
                                        self.set_center_gps)
        self.screen.signal_callback_add("goto-buddy-pressed", "",
                                        self.set_center_buddy)

    def set_center_free(self):
        self.track_ctrl.set_move_cb(None)
        self.is_ctrl.set_move_cb(None)

    def set_center_gps(self, *data):
        if self.gps_ready:
            self.is_ctrl.set_move_cb(None)
            self.track_ctrl.set_move_cb(self.map_ctrl.set_position_xy)

    def set_center_buddy(self, *data):
        if self.buddy_connected:
            self.track_ctrl.set_move_cb(None)
            self.is_ctrl.set_move_cb(self.map_ctrl.set_position_xy)

    def set_gps_status(self, status):
        if status == GPSModel.FIXED:
            self.gps_ready = True
            self.screen.signal_emit("enable-goto-gps", "")
        else:
            self.gps_ready = False
            self.screen.signal_emit("disable-goto-gps", "")

    def set_buddy_status(self, enabled):
        self.buddy_connected = enabled
        if enabled:
            self.screen.signal_emit("enable-goto-buddy", "")
        else:
            self.screen.signal_emit("disable-goto-buddy", "")

    def finalize(self):
        self.is_ctrl.finalize()
        self.track_ctrl.finalize()
        self.track_aux_ctrl.finalize()
        lat, lon, zoom = self.map_ctrl.get_position()
        CarmanConfig().set_last_fix(lat, lon, zoom)
        self.map_ctrl.finalize()

    def update_zoom_buttons(self, up, down):
        if up:
            self.screen.signal_emit("enable-zoom-up", "")
        else:
            self.screen.signal_emit("disable-zoom-up", "")
        if down:
            self.screen.signal_emit("enable-zoom-down", "")
        else:
            self.screen.signal_emit("disable-zoom-down", "")

    def fullscreen(self):
        controller = self.main_view.controller
        view = self.main_view.view

        if self.toggle_fs:
            view.signal_callback_add("exit-button-pressed", "*",
                                     controller.exit_button)
            view.signal_callback_add("change-button-pressed", "*",
                                     controller.change_button)
            view.signal_callback_add("option-button-pressed", "*",
                                     controller.option_button)
            view.signal_callback_add("status-button-pressed", "*",
                                     controller.status_button)
            self.screen.signal_callback_add("zoom-up-pressed", "",
                                            self.on_zoom_up)
            self.screen.signal_callback_add("zoom-down-pressed", "",
                                            self.on_zoom_down)
            self.screen.signal_callback_add("goto-gps-pressed", "",
                                            self.set_center_gps)
            self.screen.signal_callback_add("goto-buddy-pressed", "",
                                            self.set_center_buddy)
            view.signal_emit("default", "")
            self.screen.signal_emit("default", "")
            self.toggle_fs = False
        else:
            view.signal_callback_del("exit-button-pressed", "*",
                                     controller.exit_button)
            view.signal_callback_del("change-button-pressed", "*",
                                     controller.change_button)
            view.signal_callback_del("option-button-pressed", "*",
                                     controller.option_button)
            view.signal_callback_del("status-button-pressed", "*",
                                     controller.status_button)
            self.screen.signal_callback_del("zoom-up-pressed", "",
                                            self.on_zoom_up)
            self.screen.signal_callback_del("zoom-down-pressed", "",
                                            self.on_zoom_down)
            self.screen.signal_callback_del("goto-gps-pressed", "",
                                            self.set_center_gps)
            self.screen.signal_callback_del("goto-buddy-pressed", "",
                                            self.set_center_buddy)
            view.signal_emit("fullscreen", "")
            self.screen.signal_emit("fullscreen", "")
            self.toggle_fs = True
        self.is_ctrl.set_fullscreen(self.toggle_fs)

    def on_zoom_up(self, *params):
        self.map_ctrl.zoom_up()

    def on_zoom_down(self, *params):
        self.map_ctrl.zoom_down()

    def request_authorization(self, name, email):
        self.is_ctrl.request_authorization(name, email)

    def disconnect_buddy(self, path, show_msg):
        self.is_ctrl.disconnect_buddy(path, show_msg)

    def calculate_zoom_and_position(self):
        self.map_ctrl.calculate_zoom_and_position()

    def set_track_aux(self, models, track_file):
        self.track_aux_ctrl.set_track_model(models, track_file)
        # FIXME: Point screen to loaded track
        self.map_ctrl.update_position()

    def get_selected_track(self):
        return self.track_aux_ctrl.get_selected_track()

    def unload_track_aux(self):
         self.track_aux_ctrl.clear()

    def get_map_view_area(self):
        return self.map_ctrl.get_view_area()

    def show_progressbar(self, ctrl):
        self.download_ctrl = ctrl
        self.progressbar_ctrl.show()

    def hide_progressbar(self):
        self.download_ctrl = None
        self.progressbar_ctrl.hide()

    def update_progressbar(self, pos):
        self.progressbar_ctrl.update(pos)

    def is_downloading(self):
        return self.download_ctrl is not None

    def stop_download(self):
        if self.download_ctrl:
            self.download_ctrl.stop_download()
            self.hide_progressbar()
