#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os, threading, fcntl, ecore
from common.carlog import DEBUG
from models.mapmodel import MapModel
from maps.loadingctrl import LoadingController
from maps.mapzoomview import MapZoomDlgView
from common.xmlparser import get_xml_parser

DOWNLOAD_MIN_ITEMS = 10
DOWNLOAD_MAX_ITEMS = 40

class DownloadAreaController(object):

    def __init__(self, parent):
        self.timer = None
        self._parent = parent
        self.map_model = MapModel()

        self.ranges = []
        self.totals = []
        self.zooms = self.map_model.get_downloads_zoom()
        tile_w, tile_h = self.map_model.get_tile_size()
        area = parent._parent._parent.get_map_view_area()
        for zoom in self.zooms:
            range = self.__get_tiles_range(area, tile_w, tile_h, zoom)
            total = (range[2] - range[0] + 1) * (range[3] - range[1] + 1)
            self.ranges.append(range)
            self.totals.append(total)

    def __del__(self):
        DEBUG("deleting %s" % self)

    def __get_tiles_range(self, area, tile_w, tile_h, zoom):
        return (area[0] >> zoom) / tile_w, (area[1] >> zoom) / tile_h, \
            (area[2] >> zoom) / tile_w, (area[3] >> zoom) / tile_h

    def show(self):
        avg_size = self.map_model.get_tile_file_size_avg()
        estimated_size = [count * avg_size for count in self.totals]
        dlg = MapZoomDlgView(estimated_size, self.__download_tiles)
        dlg.show()

    def __download_tiles(self, dlg, indexes):
        dlg.hide()
        self._parent.hide(True)
        self._parent._parent.hide(True)
        self._parent = self._parent._parent._parent
        self._parent.show_progressbar(self)

        self.total = 0.0
        self.count = DOWNLOAD_MAX_ITEMS
        self.total_downloaded = 0
        self.indexes = indexes
        for idx in indexes:
            self.total += self.totals[idx]

        self.generator = self.__request_tiles()
        if self.generator.next():
            self.timer = ecore.timer_add(0.5, self.__continue_request_tiles)
        else:
            self.generator.close()

    def __continue_request_tiles(self):
        if self.count < DOWNLOAD_MIN_ITEMS:
            self.count += DOWNLOAD_MAX_ITEMS
            if not self.generator.next():
                self.timer = None
                self.generator.close()
                return False
        return True

    def __request_tiles(self):
        count = DOWNLOAD_MAX_ITEMS
        for idx in self.indexes:
            for j in xrange(self.ranges[idx][1], self.ranges[idx][3] + 1):
                for i in xrange(self.ranges[idx][0], self.ranges[idx][2] + 1):
                    if self.map_model.request_tile(i, j, self.zooms[idx],
                            False, self.__download_complete_cb):
                        self.__download_complete_cb(None, 0, 0, 0)
                    count -= 1
                    if not count:
                        count = DOWNLOAD_MAX_ITEMS
                        yield True
        yield False

    def __download_complete_cb(self, *param):
        self.count -= 1
        self.total_downloaded += 1
        self._parent.update_progressbar(self.total_downloaded / self.total)
        if self.total_downloaded == self.total:
            self._parent.hide_progressbar()

    def stop_download(self):
        if self.timer:
            self.timer.delete()
            self.timer = None
        self.generator.close()
        self.map_model.cancel_bg_downloads()


class DownloadTrackController(threading.Thread):

    def __init__(self, parent, track):
        threading.Thread.__init__(self)

        self.timer = None
        self._parent = parent
        self.track = track
        self.running = True
        self.map_model = MapModel()
        self.zooms = self.map_model.get_downloads_zoom()

        self.in_fd, self.out_fd = os.pipe()
        arg = fcntl.fcntl(self.out_fd, fcntl.F_GETFL)
        fcntl.fcntl(self.out_fd, fcntl.F_SETFL, arg | os.O_NONBLOCK)
        self.fdh = ecore.FdHandler(self.in_fd, ecore.ECORE_FD_READ,
            self.__load_complete)
        self.fdh.active_set(ecore.ECORE_FD_READ)

    def __del__(self):
        DEBUG("deleting %s" % self)
        os.close(self.in_fd)
        os.close(self.out_fd)

    def show(self):
        self.loading = LoadingController(self.__cancel_loading)
        self.loading.show()
        self.start()

    def __cancel_loading(self):
        self.running = False
        self.join()
        self.fdh.delete()
        self.fdh = None
        self.loading = None

    def run(self):
        namespace = ""
        zoom_count = len(self.zooms)
        points = [[None] for zoom in self.zooms]
        tile_w, tile_h = self.map_model.get_tile_size()

        track = get_xml_parser().parse(self.track)
        root_tag = track.getroot().tag
        if root_tag[0] == "{":
            namespace = root_tag.split("}")[0] + "}"

        repo = MapModel().get_repository()
        trk = track.find("%strk" % namespace)
        if trk:
            for trkseg in trk.findall("%strkseg" % namespace):
                for trkpt in trkseg.findall("%strkpt" % namespace):
                    point = repo.latlon_to_xy(float(trkpt.attrib["lat"]),
                        float(trkpt.attrib["lon"]))
                    for i in range(zoom_count):
                        x = (point[0] / tile_w) >> self.zooms[i]
                        y = (point[1] / tile_h) >> self.zooms[i]
                        if points[i][-1] != (x, y):
                            points[i].append((x, y))
                    if not self.running:
                        return

        self.points = points
        os.write(self.out_fd, " ")

    def __load_complete(self, *param):
        self.join()
        self.fdh.delete()
        self.fdh = None
        self.loading.hide()
        self.loading = None
        avg_size = self.map_model.get_tile_file_size_avg() * 3
        estimated_size = [len(pts) * avg_size for pts in self.points]
        dlg = MapZoomDlgView(estimated_size, self.__download_tiles)
        dlg.show()

    def __download_tiles(self, dlg, indexes):
        dlg.hide()
        self._parent.hide(True)
        self._parent._parent.hide(True)
        self._parent._parent._parent.hide(True)
        self._parent = self._parent._parent._parent._parent
        self._parent.show_progressbar(self)

        self.total = 0.0
        self.count = DOWNLOAD_MAX_ITEMS
        self.total_downloaded = 0
        self.indexes = indexes
        for idx in indexes:
            self.total += (len(self.points[idx]) - 1) * 9

        self.generator = self.__request_tiles()
        if self.generator.next():
            self.timer = ecore.timer_add(0.5, self.__continue_request_tiles)
        else:
            self.generator.close()

    def __continue_request_tiles(self):
        if self.count < DOWNLOAD_MIN_ITEMS:
            self.count += DOWNLOAD_MAX_ITEMS
            if not self.generator.next():
                self.timer = None
                self.generator.close()
                return False
        return True

    def __request_tiles(self):
        count = DOWNLOAD_MAX_ITEMS
        for idx in self.indexes:
            for point in self.points[idx][1:]:
                for j in range(-1, 2):
                    for i in range(-1, 2):
                        if self.map_model.request_tile(point[0] + i,
                                point[1] + j, self.zooms[idx], False,
                                self.__download_complete_cb):
                            self.__download_complete_cb(None, 0, 0, 0)
                        count -= 1
                        if not count:
                            count = DOWNLOAD_MAX_ITEMS
                            yield True
        yield False

    def __download_complete_cb(self, *param):
        self.count -= 1
        self.total_downloaded += 1
        self._parent.update_progressbar(self.total_downloaded / self.total)
        if self.total_downloaded == self.total:
            self._parent.hide_progressbar()

    def stop_download(self):
        if self.timer:
            self.timer.delete()
            self.timer = None
        self.generator.close()
        self.map_model.cancel_bg_downloads()
