#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#


from models.connmodel import ConnectionModel
from models.obdmodel import OBDModel
from models.gpsmodel import GPSModel
from models.btmodel import BtModel
from models.dbusmodel import CarmandDbusModel
from models.kbmodel import KeyBoardModel
from main.configurebt import ConfigureOBD, ConfigureGPS
from main.statusview import StatusView
from common.carlog import DEBUG

class StatusController(object):

    def __init__(self):
        self.view = StatusView(self.gps_bt_cb, self.obd_bt_cb,
                self.wlan_bt_cb)
        self.view.signal_callback_add("hide", "", self.__destroy)

        self.gps_model = GPSModel()
        self.gps_model.add_status_changed_cb(self.view.set_gps_status)

        self.obd_model = OBDModel()
        self.obd_model.add_status_changed_cb(self.view.set_obd_status)

        self.con_model = ConnectionModel()
        self.con_model.add_status_changed_cb(self.view.set_wlan_status)

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

        self.carmand_model = CarmandDbusModel()

    def show(self):
        self.view.set_gps_status(self.gps_model.Status())
        self.view.set_obd_status(self.obd_model.Status())
        self.view.set_wlan_status(self.con_model.Status())
        self.view.show()

    def __del__(self):
        DEBUG("deleting Status Controller %s" % self)

    def __key_down_cb(self, obj, event):
        if event.key == "Escape":
            self.view.signal_emit("hide-anime", "")
            return True

        return False

    def __destroy(self, *data):
        self.gps_model.del_status_changed_cb(self.view.set_gps_status)
        self.obd_model.del_status_changed_cb(self.view.set_obd_status)
        self.con_model.del_status_changed_cb(self.view.set_wlan_status)
        self.kb_model.del_key_down_cb(self.__key_down_cb)

    def __connect_gps(self, obj, item, param):
        obj.hide()
        self.gps_model.connect(param)

    def gps_bt_cb(self, *data):
        status = self.gps_model.Status()
        if status == GPSModel.DISCONNECTED:
            addr = self.carmand_model.GetGPSDevice()
            if addr != "none" or self.carmand_model.GetMode() != "normal":
                self.gps_model.connect(addr)
            else:
                confgps = ConfigureGPS(self.__connect_gps)
                confgps.show()
        elif status == GPSModel.FIXING or status == GPSModel.FIXED:
            self.gps_model.disconnect()

    def __connect_obd(self, obj, item, param):
        obj.hide()
        self.obd_model.connect(param)

    def obd_bt_cb(self, *data):
        status = self.obd_model.Status()
        if status == OBDModel.DISCONNECTED:
            addr = self.carmand_model.GetOBDDevice()
            if addr != "none" or self.carmand_model.GetMode() != "normal":
                self.obd_model.connect(addr)
            else:
                confobd = ConfigureOBD(self.__connect_obd)
                confobd.show()
        elif status == OBDModel.CONNECTED:
            self.obd_model.disconnect()

    def wlan_bt_cb(self, *data):
        if self.con_model.Status() != ConnectionModel.CONNECTED:
            self.con_model.try_connect()
