/*
 *  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
 *
 *  This file is part of carman-pyton
 *  
 *  carman-python is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  carman-python is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#include <math.h>
#include <Python.h>
#include <Evas.h>
#include <evas/evas.c_evas.h>

#define Uint8 unsigned char
#define Uint16 unsigned short
#define Uint32 unsigned int
#define max(a, b) ((a) >= (b) ? (a) : (b))

static PyObject *_rotate(PyObject *self, PyObject *args);

PyMethodDef evas_util_methods[] = {
	{ "rotate", _rotate, METH_VARARGS, "rotate a evas image object" },
	{ NULL, NULL, 0, NULL }
};

DL_EXPORT(void)
initevasutil(void)
{
	Py_InitModule ("evasutil", evas_util_methods);
	if (PyErr_Occurred ())
		Py_FatalError ("can't initialise module evas util");
}

static void
___rotate(void *dst_data, int dst_w, int dst_h, int dst_pitch,
	void *src_data, int src_w, int src_h, int src_pitch, double cangle,
	double sangle, Evas_Colorspace colorspace)
{
	int x, y, dx, dy;
	int cy = dst_h / 2;
	int xd = ((src_w - dst_w) << 15);
	int yd = ((src_h - dst_h) << 15);
	int isin = (int)(sangle * 65536);
	int icos = (int)(cangle * 65536);
	int ax = ((dst_w) << 15) - (int)(cangle * ((dst_w - 1) << 15));
	int ay = ((dst_h) << 15) - (int)(sangle * ((dst_w - 1) << 15));
	int xmaxval = ((src_w) << 16) - 1;
	int ymaxval = ((src_h) << 16) - 1;

	dst_pitch -= dst_w;

	if (colorspace == EVAS_COLORSPACE_ARGB8888) {
		Uint32 *dst = dst_data;
		for (y = 0; y < dst_h; y++) {
			dx = (ax + (isin * (cy - y))) + xd;
			dy = (ay - (icos * (cy - y))) + yd;
			for (x = 0; x < dst_w; x++) {
				if (dx < 0 || dy < 0 || dx > xmaxval || dy > ymaxval)
					*dst++ = 0;
				else
					*dst++ = *(((Uint32*)src_data) + ((dy >> 16) *
						src_pitch) + (dx >> 16));
				dx += icos; dy += isin;
			}
			dst += dst_pitch;
		}

	} else if (colorspace == EVAS_COLORSPACE_RGB565_A5P) {
		Uint16 *dst = dst_data;
		Uint8 *dsta = dst_data + (dst_w + dst_pitch) * dst_h * 2;
		Uint8 *srca = src_data + src_pitch * src_h * 2;
		for (y = 0; y < dst_h; y++) {
			dx = (ax + (isin * (cy - y))) + xd;
			dy = (ay - (icos * (cy - y))) + yd;
			for (x = 0; x < dst_w; x++) {
				if (dx < 0 || dy < 0 || dx > xmaxval || dy > ymaxval) {
					*dst++ = 0;
					*dsta++ = 0;
				} else {
					*dst++ = *(((Uint16*)src_data) + ((dy >> 16) *
						src_pitch) + (dx >> 16));
					*dsta++ = *(((Uint8*)srca) + ((dy >> 16) *
						src_pitch) + (dx >> 16));
				}
				dx += icos; dy += isin;
			}
			dst += dst_pitch;
			dsta += dst_pitch;
		}
	}
}

static void
__rotate(Evas_Object *dst, Evas_Object *src, double angle)
{
	void *src_data, *dst_data;
	Evas_Colorspace colorspace;
	double rangle, cangle, sangle, cx, cy, sx, sy;
	int src_w, src_h, src_pitch, dst_w, dst_h, dst_pitch, aux_w, aux_h;

	evas_object_image_size_get(src, &src_w, &src_h);
	src_data = evas_object_image_data_get(src, 0);
	src_pitch = evas_object_image_stride_get(src);
	colorspace = evas_object_image_colorspace_get(src);

	rangle = angle * M_PI / 180.0;
	sangle = sin(rangle);
	cangle = cos(rangle);
	cx = cangle * src_w;
	cy = cangle * src_h;
	sx = sangle * src_w;
	sy = sangle * src_h;
	dst_w = (int)(max(max(max(fabs(cx + sy), fabs(cx - sy)),
		fabs(-cx + sy)), fabs(-cx - sy)));
	dst_h = (int)(max(max(max(fabs(sx + cy), fabs(sx - cy)),
		fabs(-sx + cy)), fabs(-sx - cy)));

	evas_object_image_size_get(dst, &aux_w, &aux_h);
	if (aux_w != dst_w || aux_h != dst_h) {
		evas_object_image_size_set(dst, dst_w, dst_h);
		evas_object_resize(dst, dst_w, dst_h);
		evas_object_image_fill_set(dst, 0, 0, dst_w, dst_h);
	}
	dst_data = evas_object_image_data_get(dst, 1);
	dst_pitch = evas_object_image_stride_get(dst);
	if (!dst_pitch)
		dst_pitch = (dst_w & 3 ? dst_w + 2 : dst_w);

	___rotate(dst_data, dst_w, dst_h, dst_pitch, src_data, src_w, src_h,
		src_pitch, cangle, sangle, colorspace);

	evas_object_image_data_set(dst, dst_data);
	evas_object_image_data_update_add(dst, 0, 0, dst_w, dst_h);
}

static PyObject *
_rotate(PyObject *self, PyObject *args)
{
	double angle;
	Evas_Object *dst, *src;
	PyObject *py_dst, *py_src;

	if (!PyArg_ParseTuple(args, "OOd", &py_dst, &py_src, &angle))
		return NULL;

	dst = ((struct PyEvasObject *)py_dst)->obj;
	src = ((struct PyEvasObject *)py_src)->obj;
	__rotate(dst, src, angle);

	Py_RETURN_NONE;
}

