#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import evas, edje, math, os
from datetime import datetime
from common.carlog import DEBUG
from main.mainview import MainView
from common.carmanconfig import CarmanConfig
from models.kbmodel import KeyBoardModel

class AccelerationRankView(edje.Edje):

    def __init__(self):
        self.max = 0
        self.nguides = 0

        canvas = MainView().get_evas_canvas()
        theme = CarmanConfig().get_current_theme()
        edje.Edje.__init__(self, canvas, file=theme, group="accelerationrank")
        self.signal_callback_add("screen-hidden", "", self.__destroy)
        self.layer_set(25)

        self.screen = edje.Edje(canvas, file=theme,
                                group="accelerationrank-screen")
        self.part_swallow("screen", self.screen)

        self.kb_model = KeyBoardModel()
        self.kb_model.add_key_down_cb(self.__key_down_cb)

    def __del__(self):
        DEBUG("deleting AccelerationRankView %s" % self)

    def __key_down_cb(self, obj, event):
        if event.key == "Escape":
            self.signal_emit("hide-screen", "")
            return True

        return False

    def __calc_nun_guides(self, max):
        for nguides in [5, 3, 4]:
            mod = max % nguides
            if mod == 0:
                return (max, nguides)
        return self.__calc_nun_guides(max+1)

    def __change_nun_guide(self):
        if self.nguides == 3:
            self.signal_emit("three-guides", "");
        elif self.nguides == 4:
            self.signal_emit("four-guides", "");

    def set_max_range(self, max):
        max = max + (max / 10)
        max = int(math.ceil(max))
        self.max, self.nguides = self.__calc_nun_guides(max)
        step = self.max / self.nguides
        self.__change_nun_guide()
        self.part_text_set("guide_max_label", str(self.max)[:4])
        for i in range(1, self.nguides):
            self.part_text_set("guide%d_label" % i, str(step*i)[:4])

    def set_bar(self, values):
        i = 1
        v_max = 0
        for value in values:
            if not self.screen.part_exists("bar%d" % i):
                break

            v_max = max(value[1], v_max)
            self.part_text_set("bar%d_status_date" % i,
                datetime.fromtimestamp(value[0]).strftime("%d/%m/%Y"))
            self.part_text_set("bar%d_status_time" % i,
                datetime.fromtimestamp(value[0]).strftime("%H:%M:%S"))
            self.screen.part_text_set("num%d" % i, str(value[1]))
            i += 1
        self.set_max_range(v_max)

        for i in range(1, 6):
            if not self.screen.part_exists("bar%d" % i):
                break

            bar = self.screen.part_object_get("bar%d" % i)
            num = self.screen.part_object_get("num%d" % i)
            if i <= len(values):
                bar.resize(bar.size[0], int(round(values[i - 1][1] * \
                    bar.size[1] / self.max)))
                bar.move(bar.pos[0], self.screen.pos[1] +
                    self.screen.size[1] - bar.size[1])
                num.move(num.pos[0], bar.pos[1] - num.size[1])
            else:
                bar.hide()
                num.hide()

    def __destroy(self, *params):
        self.screen.delete()
        self.delete()
        self.kb_model.del_key_down_cb(self.__key_down_cb)
