# profilereader.py - Reader for profile files
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from common.carlog import INFO, WARNING, ERROR
from common.singleton import Singleton
from common.carmanconfig import CarmanConfig
from common.xmlparser import get_xml_parser
from common.sensors import Sensor

class SensorReader(Singleton):
    """
    This class reads the sensor XML file associated with a profile.
    """

    def __init__(self, sensors_file_name):
        """
        SensorReader constructor.
        @param sensors_file_name: sensor XML file name
        """
        Singleton.__init__(self)
        self._sensors_xml = get_xml_parser().parse(sensors_file_name)
        self._sensors_functions = None
        self._filename = sensors_file_name
        self._sensors = {}

        self._process_sensors_xml()
    # __init__

    def _process_sensors_xml(self):
        """
        Process the sensor XML file and create a dictionary of sensors.
        """
        self._load_sensor_functions()

        for sensor in self._sensors_xml.findall("sensor"):
            items = dict(sensor.items())
            if 'pid' not in items or 'description' not in items or \
              'short_description' not in items:
                WARNING('%s conatains an invalid line: %s. Discarting it.'\
                        % (self._filename, items))
                continue
            pid = items['pid']
            desc = items['description']
            short_dec = items['short_description'] 
            (calc_func, conv_func) = self._get_sensor_functions(pid)
            (unit, mmin, mmax) = self._load_unit_metric(sensor)
            (iunit, imin, imax) = self._load_unit_imperial(sensor)
            try:
                sensor_inst = Sensor(pid, desc, short_dec, [unit, iunit],
                                     [mmin, imin], [mmax, imax], calc_func,
                                     conv_func)
            except ValueError, err:
                WARNING('Ignoring sensor %s - %s' % (pid, err))
                continue
            self._sensors[pid] = sensor_inst

    # _process_sensors_xml

    def _load_sensor_functions(self):
        """
        Load an external module with custom OBD-II data conversion functions.
        """
        items = dict(self._sensors_xml.getroot().items())
        if "code" in items:
            try:
                module = __import__(os.path.join("modules", items["code"]))
                self._sensors_functions = module.get_sensor_functions
            except ImportError, err:
                WARNING('Unable to read sensors %s module %s.'\
                        ' The error was: %s' % (os.path.join("modules",\
                         items["code"], self._filename,err)))
            except AttributeError:
                WARNING('%s from %s has no get_sensor_functions function.' \
                        % (os.path.join("modules", items["code"])),
                        self._filename)
            if not callable(self._sensors_functions):
                WARNING('s from %s provides no get_sensor_functions'\
                        'callable attribute'\
                        % (os.path.join("modules", items["code"])),\
                        self._filename)
                self._sensors_functions = None
            # TODO: add further module verificartions
    # __load_sensor_functions

    def _get_sensor_functions(self, pid):
        """
        Collect the data conversion function for a given PID.
        @param pid: the OBD-II pid (e.g. 0C)
        """
        if self._sensors_functions is not None:
            functions = self._sensors_functions(pid)
            if isinstance(functions, tuple) or \
              isinstance(functions, list) and len(functions) == 2:
                return (functions[0], functions[1])
        return (None, None)

    @staticmethod
    def _load_unit_metric(sensor):
        """
        Load the metric unit data for the given sensor.
        @param sensor: the XML root of the sensor tag
        """
        metric = sensor.find("metric")

        if metric is None:
            return (None, None, None)

        metric_items = dict(metric.items())
        return (metric_items.get("unit"), metric_items.get("min"),
            metric_items.get("max"))
    # __load_unit_metric

    @staticmethod
    def _load_unit_imperial(sensor):
        """
        Load the imperial unit data for the given sensor.
        @param sensor: the XML root of the sensor tag
        """
        imperial = sensor.find("imperial")

        if imperial is None:
            return (None, None, None)

        imperial_items = dict(imperial.items())
        return (imperial_items.get('unit'), imperial_items.get('min'),
                imperial_items.get('max'))
    # __load_unit_imperial

    def get_sensor(self, pid):
        """
        Return the Sensor class associated with{pid}.
        @param pid: a sting OBD-II pid (e.g. 0C)
        """
        if self._sensors.has_key(pid):
            return self._sensors[pid]
        else:
            return None
    # get_sensor

    def get_pid_list(self):
        """
        Return the list of pids available in the associated sensor XML file.
        """
        return self._sensors.keys()
    # get_pid_list
# SensorReader
