# dtcs.py - DTCdb helper
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gdbm
from common.carlog import WARNING, INFO

class DTCdb(object) :
    """
    DTCdb helps reading the DTC codes from a gdbm database. DTCs codes have to
    be stored at a database file as all the codes is not usually needed and
    we should spare around 300 KB of memory by not loading them into memory.
    """

    _systems = {
        'P' : 'Powertrain',
        'B' : 'Body',
        'C' : 'Chassis',
        'U' : 'Network',
    }

    _hex_to_chars = [
        'P0', 'P1', 'P2', 'P3',
        'C0', 'C1', 'C2', 'C3',
        'B0', 'B1', 'B2', 'B3',
        'U0', 'U1', 'U2', 'U3',
    ]

    def __init__(self, filename) :
        """
        DTCdb constructor.

        @param filename: the file name of the database archive.
        """
        try:
            self._dtcs_db = gdbm.open(filename, 'r')
        except gdbm.error, err:
            WARNING('Unable to load %s dtc database: %s' % (filename, err))
            self._dtcs_db = None 
    # __init__()

    @classmethod
    def get_dtc_from_data(cls, data):
        """
        Convert the data to a string representation.

        @param data: the data.
        """
        if not isinstance(data, list) or len(data) != 2:
            raise ValueError('get_dtc_from_data: Expecting a list with 2 bytes')

        head = cls._hex_to_chars[int(data[0][:1], 16)]
        return ''.join([head, data[0][1:], data[1]])
    # get_dtc_from_data

    @classmethod
    def get_dtc_list_from_data(cls, data):
        """
        Return a list of DTCs from the OBD-II data. The data should not contain
        the reply header (e.g. 43).
        @param data: the OBD-II data
        """
        ret = []
        if not isinstance(data, list):
            raise ValueError('get_dtc_list_from_data: Expecting a list')

        for obd_dtc in data:
            try:
                dtc_str = cls.get_dtc_from_data(obd_dtc)
                if dtc_str != 'P0000':
                    ret.append(cls.get_dtc_from_data(obd_dtc))
            except (ValueError, IndexError):
                INFO('ignoring %s' % obd_dtc)

        return ret
    # get_dtc_list_from_data

    def has_meaning(self, codes):
        '''
        Return if this dtc file has a meaning for C{codes}.
        @param code: the error codes (e.g. P0001) - could be a list or a
        single element.
        '''
        if self._dtcs_db is not None:
            if isinstance(codes, list):
                ret = []
                for code in codes:
                    ret.append(self._dtcs_db.has_key(code))
                return ret
            else:
                return self._dtcs_db.has_key(codes)
        else:
            if isinstance(codes, list):
                return [False for code in codes]
            else:
                return False
    # has_meaning

    def get_dtc_meaning(self, codes) :
        """
        Returns the descripttion of the error code.

        @param codes: the error code (e.g. P0001) - can be a list.
        """
        if isinstance(codes, list):
            ret = []
            for dtc in codes:
                if self._dtcs_db is None or not self.has_meaning(dtc):
                    ret.append('')
                else:
                    ret.append(self._dtcs_db[dtc])
            return ret
        else:
            if self._dtcs_db is None or not self.has_meaning(codes):
                return ''
            else:
                return self._dtcs_db[codes]
    # get_dtc_meaning()
# DTCdb
