#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os
from models.mapmodel import MapModel
from maps.eventview import EventView
from maps.maptileview import MapTileView
from models.gpsmodel import GPSModel

class MapTileController(object):

    def __init__(self, controller, canvas, track_ctrl=None,
                    im_friend_ctrl=None, track_aux_ctrl=None):
        self.zoom = None
        self.map_view = None
        self.track_ctrl = track_ctrl
        self.track_aux_ctrl = track_aux_ctrl
        self.controller = controller
        self.im_friend_ctrl = im_friend_ctrl
        self.__zoom_changed_cb = None

        self.map_model = MapModel()
        self.__map_service_updated(self.map_model.get_repository())
        self.map_model.add_map_service_updated(self.__map_service_updated)

        self.map_view = MapTileView(canvas, self, self.tile_sizex,
            self.tile_sizey)
        self.event_view = EventView(canvas, self);

    def __map_service_updated(self, repo):
        self.zoom_range = repo.get_zoom()
        self.tile_sizex = repo.get_tile_width()
        self.tile_sizey = repo.get_tile_height()
        if self.map_view:
            self.map_view.reset(self.tile_sizex, self.tile_sizey)
            self.map_view.set_position(self.centerx, self.centery, self.zoom)
        if self.__zoom_changed_cb:
            self.__zoom_changed_cb(*self.get_zoom_state())

    def set_zoom_changed_cb(self, cb):
        self.__zoom_changed_cb = callable(cb) and cb or None

    def set_view(self, theme, view):
        view.part_swallow("map_view", self.map_view)
        view.part_swallow("event_view", self.event_view)
        self.area = view.part_geometry_get("area_view")
        self.map_view.set_area(view.part_geometry_get("map_view"),
            self.area)
        self.map_view.set_theme(theme)

    def update_position(self):
        world_size = self.map_model.get_world_size(self.zoom)
        if self.centerx < 0:
            self.centerx = 0
        elif self.centerx > world_size:
            self.centerx = world_size
        if self.centery < 0:
            self.centery = 0
        elif self.centery > world_size:
            self.centery = world_size

        self.map_view.set_position(self.centerx, self.centery, self.zoom)
        if self.track_ctrl:
            self.track_ctrl.set_position(self.centerx, self.centery,
                self.zoom)
        if self.track_aux_ctrl:
            self.track_aux_ctrl.set_position(self.centerx, self.centery,
                self.zoom)
        if self.im_friend_ctrl:
            self.im_friend_ctrl.set_position(self.centerx, self.centery,
                self.zoom)

    def set_position(self, lat, lon, zoom=None):
        if zoom is None:
            zoom = self.zoom
        elif zoom != self.zoom:
            self.zoom = zoom
            if self.__zoom_changed_cb:
                self.__zoom_changed_cb(*self.get_zoom_state())
        x, y = self.map_model.latlon_to_xy(lat, lon)
        self.centerx = x >> self.zoom
        self.centery = y >> self.zoom
        self.update_position()

    def get_position(self):
        lat, lon = self.map_model.xy_to_latlon(self.centerx, self.centery,
            self.zoom)
        return lat, lon, self.zoom

    def set_position_xy(self, x, y, check_coord=True):
        if not check_coord or self.map_view.is_coord_out(x, y):
            self.centerx = x
            self.centery = y
            self.update_position()

    def is_valid_tile(self, x, y, zoom):
        return self.map_model.is_valid_tile(x, y, zoom)

    def request_last_tile(self, x, y, zoom):
        if self.map_model.is_valid_tile(x, y, zoom):
            count = 3
            offsetx = offsety = 0
            sizex = self.tile_sizex
            sizey = self.tile_sizey
            while count > 0 and zoom <= self.zoom_range[-1]:
                count -= 1
                x, rx = divmod(x, 2)
                y, ry = divmod(y, 2)
                offsetx = offsetx - rx * sizex
                offsety = offsety - ry * sizey
                sizex = sizex * 2
                sizey = sizey * 2
                zoom = zoom + 1
                filename = self.map_model.get_tile_filename(x, y, zoom)
                if os.path.exists(filename):
                    return filename, (offsetx, offsety, sizex, sizey)
        return None, None

    def clear_download_queue(self):
        self.map_model.cancel_fg_downloads()

    def cancel_download(self, x, y, zoom):
        self.map_model.cancel_download(x, y, zoom)

    def request_tile(self, x, y, zoom, priority, cb):
        return self.map_model.request_tile(x, y, zoom, priority, cb)

    def move(self, dx, dy):
        if dx != 0 or dy != 0:
            self.centerx += dx
            self.centery += dy
            self.controller.set_center_free()
            self.update_position()

    def zoom_up(self):
        if self.zoom > self.zoom_range[0]:
            self.map_model.cancel_fg_downloads()
            self.zoom -= 1
            if self.track_ctrl and self.track_ctrl.get_lock():
                data = GPSModel().get_last_data()
                x, y = self.map_model.latlon_to_xy(data[0], data[1])
                self.centerx = x >> self.zoom
                self.centery = y >> self.zoom
            elif self.im_friend_ctrl and self.im_friend_ctrl.get_lock():
                data = self.im_friend_ctrl.get_last_data()
                x, y = self.map_model.latlon_to_xy(data[0], data[1])
                self.centerx = x >> self.zoom
                self.centery = y >> self.zoom
            else:
                self.centerx *= 2
                self.centery *= 2
            self.update_position()
            if self.__zoom_changed_cb:
                self.__zoom_changed_cb(*self.get_zoom_state())

    def zoom_down(self):
        if self.zoom < self.zoom_range[-1]:
            self.map_model.cancel_fg_downloads()
            self.zoom += 1
            if self.track_ctrl and self.track_ctrl.get_lock():
                data = GPSModel().get_last_data()
                x, y = self.map_model.latlon_to_xy(data[0], data[1])
                self.centerx = x >> self.zoom
                self.centery = y >> self.zoom
            elif self.im_friend_ctrl and self.im_friend_ctrl.get_lock():
                data = self.im_friend_ctrl.get_last_data()
                x, y = self.map_model.latlon_to_xy(data[0], data[1])
                self.centerx = x >> self.zoom
                self.centery = y >> self.zoom
            else:
                self.centerx /= 2
                self.centery /= 2
            self.update_position()
            if self.__zoom_changed_cb:
                self.__zoom_changed_cb(*self.get_zoom_state())

    def get_zoom_state(self):
        return self.zoom > self.zoom_range[0], self.zoom < self.zoom_range[-1]

    def get_center_position(self):
        return self.centerx, self.centery, self.zoom

    def get_view_area(self):
        view_size = self.map_view.size
        width = (view_size[0] / 2) << self.zoom
        height = (view_size[1] / 2) << self.zoom
        x1 = (self.centerx << self.zoom) - width
        y1 = (self.centery << self.zoom) - height
        x2 = (self.centerx << self.zoom) + width
        y2 = (self.centery << self.zoom) + height
        return (x1, y1, x2, y2)

    def calculate_zoom_and_position(self):
        gps_x, gps_y = self.track_ctrl.get_gps_position()
        buddy_x, buddy_y = self.im_friend_ctrl.get_buddy_position()

        dx = abs(gps_x - buddy_x)
        dy = abs(gps_y - buddy_y)
        self.centerx = (gps_x + buddy_x) / 2
        self.centery = (gps_y + buddy_y) / 2

        if dx - self.area[2] > 0 or dy - self.area[3] > 0:
            while dx >= self.area[2] or dy >= self.area[3]:
                dx /= 2
                dy /= 2
                self.zoom += 1
                self.centerx /= 2
                self.centery /= 2
            if self.__zoom_changed_cb:
                self.__zoom_changed_cb(*self.get_zoom_state())

        self.update_position()

    def activate(self):
        self.map_view.set_position(self.centerx, self.centery, self.zoom)

    def finalize(self):
        self.map_model.finalize()
