#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import evas, edje

class BuddyView(evas.SmartObject):

    def __init__(self, controller, canvas):
        evas.SmartObject.__init__(self, canvas)
        self.active = False
        self.controller = controller
        self.clipper = self.Rectangle()

        self.buddy = []
        for i in range(9):
            buddy = self.Image()
            buddy.clip_set(self.clipper)
            buddy.on_mouse_down_add(self.__buddy_mouse_down)
            self.buddy.append([buddy, -1, -1])
        self.last_buddy = None

        self.x = self.y = 0
        self.buddy_x = self.buddy_y = 0
        self.center_x = self.center_y = 0

    def __load_image(self, buddy, key):
        buddy[0].file_set(*key)
        w, h = buddy[0].image_size_get()
        buddy[0].size_set(w, h)
        buddy[0].fill_set(0, 0, w, h)
        buddy[1], buddy[2] = w / 2, h / 2

    def set_theme(self, theme):
        for i in range(9):
            self.__load_image(self.buddy[i], (theme, "images/buddy%d" % i))

    def set_buddy_area(self, area):
        self.buddy_area = (area[0], area[1], area[0] + area[2],
            area[1] + area[3])
        self.__area = self.buddy_area

    def set_fullscreen(self, fullscreen):
        self.buddy_area = (30, 30, 770, 450) if fullscreen else self.__area
        if self.active:
            self.changed()

    def calculate(self):
        x = self.x + self.w / 2 - self.center_x + self.buddy_x
        y = self.y + self.h / 2 - self.center_y + self.buddy_y

        if y < self.buddy_area[1]:
            if x < self.buddy_area[0]:
                buddy = self.buddy[8]
            elif x > self.buddy_area[2]:
                buddy = self.buddy[2]
            else:
                buddy = self.buddy[1]

        elif y > self.buddy_area[3]:
            if x < self.buddy_area[0]:
                buddy = self.buddy[6]
            elif x > self.buddy_area[2]:
                buddy = self.buddy[4]
            else:
                buddy = self.buddy[5]

        elif x < self.buddy_area[0]:
            buddy = self.buddy[7]
        elif x > self.buddy_area[2]:
            buddy = self.buddy[3]
        else:
            buddy = self.buddy[0]

        if self.last_buddy != buddy[0]:
            if self.last_buddy:
                self.last_buddy.hide()
            buddy[0].show()
        self.last_buddy = buddy[0]

        x = min(max(x, self.buddy_area[0]), self.buddy_area[2])
        y = min(max(y, self.buddy_area[1]), self.buddy_area[3])
        buddy[0].move(x - buddy[1], y - buddy[2])

    def set_position(self, x, y):
        self.center_x = x
        self.center_y = y
        if self.active:
            self.changed()

    def get_buddy_position(self):
        return self.buddy_x, self.buddy_y

    def set_buddy_position(self, x, y):
        self.buddy_x = x
        self.buddy_y = y
        if self.active:
            self.changed()

    def __buddy_mouse_down(self, *param):
        self.controller.calculate_zoom_and_position()

    def activate(self):
        self.active = True

    def deactivate(self):
        self.active = False
        self.last_buddy = None
        for buddy in self.buddy:
            buddy[0].hide()

    def clip_set(self, obj):
        self.clipper.clip_set(obj)

    def clip_unset(self):
        self.clipper.clip_unset()

    def show(self):
        self.clipper.show()

    def move(self, x, y):
        self.x = x
        self.y = y
        self.clipper.move(x, y)

    def resize(self, w, h):
        self.w = w
        self.h = h
        self.clipper.resize(w, h)


class BuddyInformationView(object):

    def __init__(self, canvas, metric):
        self.view = None
        self.canvas = canvas
        self.metric = metric
        self.visible = False
        self.message = ""
        self.buddy_name = ""
        self.speed = self.rpm = -1

    def set_view(self, theme, view):
        self.parent_view = view
        if self.view:
            self.view.delete()
        self.view = edje.Edje(self.canvas, file=theme,
            group="buddy_information")
        view.part_swallow("buddy_information", self.view)

        if self.visible:
            self.show()
            self.set_buddy_name(self.buddy_name)
            if self.message:
                self.show_message(self.message)
            else:
                self.set_speed_value(self.speed)
                self.set_rpm_value(self.rpm)

    def set_unit_system(self, value):
        self.metric = value
        self.set_speed_value(self.speed)

    def set_buddy_name(self, name):
        self.buddy_name = name
        self.view.part_text_set("buddy_name", name.encode("utf-8"))

    def set_speed_value(self, value):
        self.speed = value
        if value < 0:
            self.view.part_text_set("speed-value", "--")
        else:
            if self.metric:
                self.view.part_text_set("speed-value", "%dKm/h" % value)
            else:
                value = round(value * 62 / 100)
                self.view.part_text_set("speed-value", "%dMPH" % value)

    def set_rpm_value(self, value):
        self.rpm = value
        if value < 0:
            self.view.part_text_set("RPM-value", "--")
        else:
            self.view.part_text_set("RPM-value", "%d" % value)

    def show_message(self, message):
        self.message = message
        self.view.signal_emit("hide-data", "")
        self.view.part_text_set("message", message.encode("utf-8"))

    def show_data_values(self):
        self.message = ""
        self.view.signal_emit("show-data", "")

    def show(self):
        self.visible = True
        self.parent_view.signal_emit("show-buddy-info", "")

    def hide(self):
        self.visible = False
        self.parent_view.signal_emit("hide-buddy-info", "")

    def is_visible(self):
        return self.visible
