# _passkeyagent.py - Makes it easier to create a custom passkey agent
#
# Copyright (C) 2008 Instituto Nokia de Tecnologia - INdT
#
# This file is part of carman
#
# carman is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# carman is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import ecore
import e_dbus
import dbus
import dbus.service

PASSKEY_AGENT_INTERFACE = 'org.bluez.PasskeyAgent'
PASSKEY_AGENT_PATH = '/carman_passkeyagent/dev'

class CarmanPasskeyAgent(dbus.service.Object):
    """ """
    def __init__(self, main_loop, address, request_cb=None):
        if not address:
            raise ValueError("Error must be a bluetooth address.")
        self._address = address
        self._path = PASSKEY_AGENT_PATH + str(self._address).replace(":","")
        self._bus = dbus.SystemBus(mainloop=main_loop, private=True)
        try:
            dbus.service.Object.__init__(self, self._bus, self._path)
        except KeyError:
            pass # This path is already handled
        self._return_cb = None
        self._err_cb = None
        self._sec = None
        self._request_cb = callable(request_cb) and request_cb or None
    # __init__

    def success(self, value):
        """
        This is the asynchonous reply to the Request function.

        @param value: this is the *returned* string of the function.
        """
        self._return_cb(value)
        self._request_cb = None
    # success

    def error(self, e):
        """
        This is the asynchronous error to the Request function.

        @param e: Exception instance
        """
        self._err_cb(e)
        self._request_cb = None
    # error

    @dbus.service.method(dbus_interface=PASSKEY_AGENT_INTERFACE,
                         in_signature='ssb', out_signature='s',
                         async_callbacks=('return_cb', 'err_cb'))
    def Request(self, agent_path, remote_address, use_default, return_cb,
                err_cb):
        """ """
        self._return_cb = return_cb
        self._err_cb = err_cb
        if self._request_cb is not None:
            self._request_cb(remote_address)
        self._request_cb = None
        return
    # Request

    @dbus.service.method(dbus_interface=PASSKEY_AGENT_INTERFACE,
                         in_signature='', out_signature='')
    def Release(self):
        """ """
        self.unregister()
    # Releaseaddress):

    def register(self):
        """ """
        if self._sec:
            return
        self._sec = dbus.Interface(self._bus.get_object('org.bluez',
                                                        '/org/bluez/hci0'),
                                                        'org.bluez.Security')
        self._sec.RegisterPasskeyAgent(self._path, self._address)
    # register

    def unregister(self):
        if not self._sec:
            return
        self._sec.UnregisterPasskeyAgent(self._path, self._address)
        self._sec = None
    # unregister

# CarmanPassKeyAgent
