#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{HandlePIDs} and L{OBDModel}.
"""

import dbus
from common.carlog import DEBUG
from common.singleton import Singleton
from main.pairingdlgctrl import PairingDlgController
from models.btmodel import BtModel
from models.dbusmodel import CarmandDbusModel, DBUS_BUS_NAME, \
    DBUS_OBJECT_PATH,  DBUS_IFACE_OBD

class HandlePIDs(object):
    """
    Abstraction for OBD PID's properties.

    @type   pids:   list
    @param  pids:   List of OBD PID's.
    @type   handles:    list
    @param  handles:    List of OBD D-Bus handles.
    @type   cb: callback
    @param  cb: Callback to be called when new data is available.
    """

    def __init__(self, pids, handles, cb):
        self.cb = cb
        self.pids = pids
        self.handles = handles


class OBDModel(Singleton, dbus.Interface):
    """
    Implements OBD-related features (handles OBD data, status and connection).

    @cvar DISABLED: OBD status flag (OBD disabled).
    @cvar DISCONNECTED: OBD status flag (OBD disconnected).
    @cvar CONNECTING: OBD status flag (OBD connecting).
    @cvar CONNECTED: OBD status flag (OBD connected).
    """

    DISABLED     = "Disabled"
    DISCONNECTED = "Disconnected"
    CONNECTING   = "Connecting"
    CONNECTED    = "Connected"

    def __init__(self):
        Singleton.__init__(self)

        self._cache = {}
        self._dtc_list_cbs = []
        self._status_changed_cbs = []
        self._data_available_cbs = []
        self._connection_lost_cbs = []
        self._disconnecting = False

        carmandbusmodel = CarmandDbusModel()
        carmandbus = carmandbusmodel.get_bus()

        dbus.Interface.__init__(self, carmandbusmodel.get_bus_object(),
                        DBUS_IFACE_OBD)

        carmandbus.add_signal_receiver(self.__on_dtc_list,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'DTCList')

        carmandbus.add_signal_receiver(self.__on_status_changed,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'StatusChanged')

        carmandbus.add_signal_receiver(self.__on_data_available,
            bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH,
            dbus_interface = DBUS_IFACE_OBD, signal_name = 'DataAvailable')

    def __on_dtc_list(self, dtcs):
        """
        Calls all callbacks registered to be called when a new DTC list is
        available.

        When a DTC list is requested from carmand (using D-Bus),
        it sends a 'DTCList' signal. This signal is captured by OBDModel
        which calls this callback.

        @type   dtcs:   list
        @param  dtcs:   List of DTC's.
        """
        for cb in self._dtc_list_cbs:
            cb(dtcs)

    def __on_status_changed(self, status):
        """
        Called when carmand changes the OBD status connection.

        @type   status: string
        @param  status: OBD status.
        """
        DEBUG("OBD status changed to %s" % status)
        for cb in self._status_changed_cbs:
            cb(status)
        if status == self.DISCONNECTED:
            self._cache = {}
            if not self._disconnecting:
                for cb in self._connection_lost_cbs:
                    cb()
            self._disconnecting = False
            if self.pairingctrl:
                self.pairingctrl.hide()
            self.pairingctrl = None
        elif status == self.CONNECTED:
            self.pairingctrl = None

    def __on_data_available(self, pid, *data):
        """
        Called when the carmand have information about a sensor to report.

        @type   pid:    number
        @param  pid:    PID from OBD data that is available.
        @type   data:   list
        @param  data:   List of last given OBD PID data (eg. use C{data[0]} to
                        pick up last data).
        """
        self._cache["%02X" % pid] = data

        for handle in self._data_available_cbs:
            if pid in handle.pids:
                handle.cb(self, "%02X" % pid, *data)

    def add_dtc_list_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when a new DTC
        list is available.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self._dtc_list_cbs:
            self._dtc_list_cbs.append(cb)

    def del_dtc_list_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when a new
        DTC list is available.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self._dtc_list_cbs:
            self._dtc_list_cbs.remove(cb)

    def add_status_changed_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when OBD status
        has changed.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self._status_changed_cbs:
            self._status_changed_cbs.append(cb)

    def del_status_changed_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when OBD
        status has changed.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self._status_changed_cbs:
            self._status_changed_cbs.remove(cb)

    def add_connection_lost_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when OBD
        connection is lost.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self._connection_lost_cbs:
            self._connection_lost_cbs.append(cb)

    def del_connection_lost_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when OBD
        connection is lost.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self._connection_lost_cbs:
            self._connection_lost_cbs.remove(cb)

    def add_data_available_cb(self, cmds, cb):
        """
        Adds a callback to the list of callbacks to be called when new OBD
        data from given PID is available.

        @type   cmds:   list
        @param  cmds:   List of OBD PID's.
        @type   cb: callback
        @param  cb: Callback to be added.
        @rtype: class
        @return:    Instance of C{HandlePIDs} class.
        """
        if not cmds or not callable(cb) or cb in self._data_available_cbs:
            return

        pids = []
        handles = []
        for cmd in cmds:
            if len(cmd) == 2:
                handle = self.AddTimeSensor(int(cmd[0], 16), cmd[1])
            elif len(cmd) == 3:
                handle = self.AddRoundSensor(int(cmd[0], 16), cmd[1], cmd[2])
            else:
                continue
            pids.append(int(cmd[0], 16))
            handles.append(handle)

        if handles:
            handlepid = HandlePIDs(pids, handles, cb)
            self._data_available_cbs.append(handlepid)
            return handlepid

    def del_data_available_cb(self, handlepid):
        """
        Removes a callback from the list of callbacks to be called when new OBD
        data from given PID is available.

        @type   handlepid:  class
        @param  handlepid:  Instance of C{HandlePIDs} class to be removed.
        """
        if handlepid in self._data_available_cbs:
            self._data_available_cbs.remove(handlepid)
            for handle in handlepid.handles:
                self.DelSensor(handle)

    def get_last_data(self, pid):
        """
        Return last data from given OBD PID.

        @type   pid:    number
        @param  pid:    OBD PID which last data is needed.
        @rtype: number
        @return:    OBD PID's last data.
        """
        return self._cache.get(pid)

    def connect(self, address):
        """
        Connects to an OBD device.

        @type   address:    string
        @param  address:    OBD device bluetooth address (eg.
                            C{01:23:45:67:89:ab}).
        """
        bt_model = BtModel()
        if address != "none" and not bt_model.has_bonding(address):
            self.pairingctrl = PairingDlgController()
            bt_model.passkey_register(address, self.pairingctrl.show)
        self.ConnectBT()

    def disconnect(self):
        """
        Disconnects from a connected OBD device, if any.
        """
        self._disconnecting = True
        self.Disconnect()
