#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{InfoSharingService}.
"""

class InfoSharingService(object):
    """
    InfoSharing service class. This class implements abstract methods
    used by infoSharing service.

    @cvar   _cbs: Dictionary of callbacks to be called when a DBus signal is
                  emitted from InfoShare daemon.
    """
    _cbs = {}

    def __init__(self):
        pass

    def _on_buddy_signed_on(self, name, alias):
        """
        Called when given buddy signs on.

        @type   name: string
        @param  name: Buddy name.
        @type   alias: string
        @param  alias: Buddy alias.
        """
        if 'buddy-signed-on' in self._cbs:
            self._cbs['buddy-signed-on'](name, alias)

    def _on_buddy_signed_off(self, name, alias):
        """
        Called when given buddy signs off.

        @type   name: string
        @param  name: Buddy name.
        @type   alias: string
        @param  alias: Buddy alias.
        """
        if 'buddy-signed-off' in self._cbs:
            self._cbs['buddy-signed-off'](name, alias)

    def _on_connection_error(self, username, protocol_id, short_desc, desc):
        """
        Called when a connection error occurs.

        @type   username: string
        @param  username: Account username.
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        @type   short_desc: string
        @param  short_desc: Short description of connection error reason.
        @type   desc: string
        @param  desc: Description of connection error reason.
        """
        if 'connection-error' in self._cbs:
            self._cbs['connection-error'] \
                (username, protocol_id, short_desc, desc)

    def _on_received_msg(self, name, alias, stripped):
        """
        Called when a message is received.

        @type   name: string
        @param  name: Buddy name.
        @type   alias: string
        @param  alias: Buddy alias.
        @type   stripped: string
        @param  stripped: HTML-stripped message.
        """
        if 'received-msg' in self._cbs:
            self._cbs['received-msg'](name, alias, stripped)

    def _on_request_add_buddy(self, name, alias, message):
        """
        Called when a request to add buddy is emitted.

        @type   name: string
        @param  name: Buddy name.
        @type   alias: string
        @param  alias: Buddy alias.
        @type   message: string
        @param  message: Buddy request message, if any.
        """
        if 'request-add-buddy' in self._cbs:
            self._cbs['request-add-buddy'](name, alias, message)

    def _on_request_authorize_buddy(self, name, alias, message, on_list):
        """
        Called when a request to authorize buddy is emitted.

        @type   name:   string
        @param  name:   Buddy name (eg. C{average.joe@mail.com}).
        @type   alias:  string
        @param  alias:  Buddy alias (eg. C{Joe, the average guy}).
        @type   message: string
        @param  message: Buddy request message, if any.
        @type   on_list: string
        @param  on_list: C{"True"} if buddy appears on buddy list,
                         C{"False"} otherwise.
        """
        if 'request-authorize-buddy' in self._cbs:
            stob = lambda string: False if string == 'False' else True
            self._cbs['request-authorize-buddy'](name, alias, stob(on_list))

    def _on_signed_on(self, username, protocol_id):
        """
        Called when the given account has signed on.

        @type   username: string
        @param  username: Account username.
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        """
        if 'signed-on' in self._cbs:
            self._cbs['signed-on'](username, protocol_id)

    def _on_signed_off(self, username, protocol_id):
        """
        Called when the given account has signed off.

        @type   username: string
        @param  username: Account username.
        @type   protocol_id: string
        @param  protocol_id: Account protocol ID (eg. C{prpl-jabber}).
        """
        if 'signed-off' in self._cbs:
            self._cbs['signed-off'](username, protocol_id)

    def add_callback(self, cb_name, cb):
        """
        Adds a callback to the callbacks dictionary L{_cbs} which is used to
        store InfoShare model callbacks.

        @type   cb_name: string
        @param  cb_name: Callback identifier (eg. C{'signed-on'}).
        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb):
            self._cbs[cb_name] = cb

    def send_message(self, name, message):
        """
        Sends a message to a specified buddy.

        @type name: string
        @param name: Buddy name (eg. C{average.joe@mail.com}).
        @type message: string
        @param message: Message.
        """
        pass

    def add_account(self, acc_id):
        """
        Adds a new account to InfoShare daemon.

        @type   acc_id: tuple
        @param  acc_id: Account information (C{username}, C{protocol_id}).
        """
        pass

    def remove_account(self):
        """
        Removes the current account.
        """
        pass

    def set_account_info(self, info):
        """
        Sets the account information.

        @type info: string
        @param info: Tuple of informations (eg:
                     {username, protocol_id, password, alias,
                     remember_password, server, port}).
        """
        pass

    def set_protocol_options(self, info):
        """
        Sets the account protocol options.

        @type   info: dictionary
        @param  info: Dictionary of protocol options.
        """

    def get_account_info(self):
        """
        Returns the account information.
        """
        pass

    def get_account(self):
        """
        Returns the current infoshare account.
        """
        pass

    def account_connect(self):
        """
        Connects to the current account.
        """
        pass

    def account_disconnect(self):
        """
        Disconnects from the current account.
        """
        pass

    def account_is_enabled(self):
        """
        Verifies if account is enabled.
        """
        pass

    def account_is_connected(self):
        """
        Verifies if account is connected.
        """
        pass

    def account_is_connecting(self):
        """
        Verifies if account is connecting.
        """
        pass

    def account_is_disconnected(self):
        """
        Verifies if account is disconnected.
        """
        pass

    def account_exists(self):
        """
        Verifies if account exists.
        """
        pass

    def set_account_password(self, password=None, remember=None):
        """
        Sets the account password.

        @type password: string
        @param password: Account password.
        @type remember: string
        @param remember: Check if account's password will be remembered.
        """
        pass

    def get_account_password(self):
        """
        Returns the account password.
        """
        pass

    def get_accounts_active(self):
        """
        Returns a list of active accounts.
        """
        pass

    def get_protocol_options(self, protocol_id):
        """
        Returns a dictionary of protocol options.

        @type protocol_id: string
        @param protocol_id: Protocol's ID for retrieve options
                            (eg. C{prpl-jabber}).
        """
        pass

    def get_protocols(self):
        """
        Returns the protocols IDs list.
        """
        pass

    def authorize_buddy(self, name, authorize):
        """
        Notifies InfoShare daemon to authorize (or not) the given buddy.

        @type   name: string
        @param  name: Buddy name.
        @type   authorize: boolean
        @param  authorize: C{True} if buddy is authorized, C{False} if not.
        """
        pass

    def add_buddy(self, name, alias=None, group=None):
        """
        Adds a buddy to the current account.

        @type name: string
        @param name: Buddy name (eg. C{average.joe@mail.com}).
        @type alias: string
        @param alias: Buddy alias (eg. C{Joe, the average guy}) (optional).
        @type group: string
        @param group: Buddy group (optional).
        """
        pass

    def remove_buddy(self, name):
        """
        Removes a buddy from the current account.

        @type name: string
        @param name: Buddy name (eg. C{average.joe@mail.com}).
        """
        pass

    def get_buddies_online(self):
        """
        Returns a list of online buddies using Carman.
        """
        pass

    def get_buddy_alias(self, name):
        """
        Returns the buddy alias.

        @type name: string
        @param name: Buddy name (eg. C{average.joe@mail.com}).
        """
        pass

    def set_buddy_alias(self, name, alias):
        """
        Sets the buddy alias.

        @type name: string
        @param name: Buddy name (eg. C{average.joe@mail.com}).
        @type alias: string
        @param alias  Buddy alias (eg. C{Joe, the average guy}).
        """
        pass
