# download_mng.py - Download manager
#
#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{_FileDownload} and L{DownloadManager}.

@note: This module tries to import C{signal} module (C{ImportError} raises when
not found).
"""

import pycurl
import threading
import os
import fcntl

try:
    import signal
    signal.signal(signal.SIGPIPE, signal.SIG_IGN)
except ImportError:
    pass

class _FileDownload(object):
    """
    File Download class (stores the filename download information).

    @type   url: string
    @param  url: URL name.
    @type   path: string
    @param  path: Filename path.
    @type   data: tuple
    @param  data: Tile data C{(tile_x, tile_y, zoom, [callback])}.
    """

    def __init__(self, url, path, data):
        self.url = url
        self.path = path
        self.downloading = False
        self.handle = None
        self.data = data

    def __str__(self):
        """
        Returns the string representation of the class object.

        @rtype: string
        @return: Object string representation 'name (url, filename, data)'.
        """
        return ('%s (url=%r, filename=%r, data=%s)' %
                (type(self).__name__, self.url, self.path, self.data))

class DownloadManager(threading.Thread):
    """
    Download Manager thread class (handles downloads of L{_FileDownload} objects).

    @type   EVT_CALLBACK: string
    @cvar   EVT_CALLBACK: Event Callback.
    @type   PART_EXT: string
    @cvar   PART_EXT: Part extension.
    """

    EVT_CALLBACK = 'K'
    PART_EXT = '.part'

    def __init__(self):
        self._queue = []
        self._bg_queue = []
        self._freelist = []
        self._multi = pycurl.CurlMulti()
        self._multi.handles = []
        self._in_fd, self._out_fd = os.pipe()
        self._down_ev = threading.Event()
        self._queue_lock = threading.Lock()
        self._multi_lock = threading.Lock()
        self._result_lock = threading.Lock()
        self._down_results = []
        self._errcb = None
        self._complete_cb = None
        self._running = False
        self._num_conn = 3 # Number of downloads
        threading.Thread.__init__(self)

    def _setup_poll(self):
        """
        Pre-allocate a list of curl objects.
        """
        for i in range(self._num_conn):
            c = pycurl.Curl()
            c.fp = None
            c.setopt(pycurl.FOLLOWLOCATION, 1)
            c.setopt(pycurl.MAXREDIRS, 5)
            c.setopt(pycurl.CONNECTTIMEOUT, 30)
            c.setopt(pycurl.TIMEOUT, 300)
            c.setopt(pycurl.NOSIGNAL, 1)
            self._multi.handles.append(c)
        self._freelist = self._multi.handles[:]

    def initialize(self, complete_cb, errcb):
        """
        Initializes the thread.

        @type   complete_cb: callback
        @param  complete_cb: Download completion callback.
        @type   errcb: callback
        @param  errcb: Error callback.

        @rtype: file descriptor
        @return: File descriptor for reading.
        """
        if callable(errcb):
            self._errcb = errcb

        if callable(complete_cb):
            self._complete_cb = complete_cb

        pycurl.global_init(pycurl.GLOBAL_ALL)
        self._setup_poll()
        arg = fcntl.fcntl(self._out_fd, fcntl.F_GETFL)
        fcntl.fcntl(self._out_fd, fcntl.F_SETFL, arg | os.O_NONBLOCK)

        self._running = True
        self.start()

        return self._in_fd

    def _append_download(self, path, data, errmsg=None):
        """
        Appends a download to the list of downloads.

        @type   path: string
        @param  path: Filename path.
        @type   data: tuple
        @param  data: Tile data C{(tile_x, tile_y, zoom, [callback])}.
        @type   errmsg: callback
        @param  errmsg: Error message callback.
        """
        self._result_lock.acquire()
        self._down_results.append((path, data, errmsg))
        len_results = len(self._down_results)
        self._result_lock.release()

        if len_results == 1:
            os.write(self._out_fd, self.EVT_CALLBACK)

    def _get_results(self):
        """
        Returns the list of downloads.
        This list is cleared right after.

        @rtype: list
        @return: List of Downloads C{[(filename, data, errmsg)]}.
        """
        os.read(self._in_fd, 1)
        self._result_lock.acquire()
        results = self._down_results
        self._down_results = []
        self._result_lock.release()

        return results

    def process_results(self):
        """
        Process the results from the list of downloads.
        """
        results = self._get_results()
        if results is None:
            return
        for path, data, errmsg in results:
            if errmsg is not None:
                self._errcb(path, data, errmsg)
            else:
                self._complete_cb(path, data)

    def add_download(self, url, path, data, foreground):
        """
        Add a file download (L{_FileDownload}) to the download queue.

        @type   url: string
        @param  url: URL name.
        @type   path: string
        @param  path: Filename path.
        @type   data: tuple
        @param  data: Tile data C{(tile_x, tile_y, zoom, [callback])}.
        @type   foreground: boolean
        @param  foreground: C{True} if the download is set to foreground,
                            C{False} if is set to background.

        @rtype: boolean
        @return: C{True} if the download is added to the download queue,
                C{False} if the URL is not given.
        """
        if url is None:
            return False

        # has some download in the queue with the same url?
        if self.change_state(url, foreground, data):
            return True

        _new_download = _FileDownload(url, path, data)

        self._queue_lock.acquire()
        if foreground:
            self._queue.append(_new_download)
        else:
            self._bg_queue.append(_new_download)
        self._queue_lock.release()
        self._down_ev.set()
        return True

    def change_state(self, url, foreground, data):
        """
        Returns whether a download with the given I{url} is in
        the download queue.

        @type   url: string
        @param  url: URL name.
        @type   foreground: boolean
        @param  foreground: C{True} if the download is set to foreground,
                            C{False} if is set to background.
        @type   data: tuple
        @param  data: Tile data C{(tile_x, tile_y, zoom, [callback])}.

        @rtype: boolean
        @return: C{True} if the download with the given URL is in the queue,
                C{False} otherwise.
        """
        callback = data[3][0]
        self._queue_lock.acquire()
        has_some = False
        for i, elem in enumerate(self._queue):
            if elem.url == url:
                has_some = True

                e_list_callbacks = elem.data[3]
                if callable(callback) and not callback in e_list_callbacks:
                    e_list_callbacks.append(callback)

                if not foreground and not elem.downloading:
                    self._bg_queue.append(self._queue.pop(i))
                self._queue_lock.release()
                return has_some

        for i, elem in enumerate(self._bg_queue):
            if elem.url == url:
                has_some = True

                e_list_callbacks = elem.data[3]
                if callable(callback):
                    e_list_callbacks.append(callback)

                if foreground and not elem.downloading:
                    self._queue.append(self._bg_queue.pop(i))
                self._queue_lock.release()
                return has_some

        self._queue_lock.release()
        return has_some

    def cancel_download(self, url):
        """
        Remove a download from the queue.

        @type   url: string
        @param  url: URL name.
        """
        def helper_cancel(url, queue):
            for idx, elem in enumerate(queue):
                if elem.url == url:
                    if not elem.downloading:
                        queue.pop(idx)
                    break

        self._queue_lock.acquire()
        helper_cancel(url, self._queue)
        self._queue_lock.release()

    def stop_all_downloads(self):
        """
        Stop all file downloads. Remove all handles from the
        curl multi-object.
        """
        self._queue_lock.acquire()
        self._queue = []
        self._bg_queue = []
        self._queue_lock.release()

        self._multi_lock.acquire()
        for hand in self._multi.handles:
            if hand.fp is not None:
                self._multi.remove_handle(hand)
                hand.fp.close()
                hand.fp = None
                hand.url = None
                hand.filename = None
                hand.data = None
                self._freelist.append(hand)
        self._multi_lock.release()

        self._down_ev.clear()

    def close(self):
        """
        Close the download manager.
        """
        self._multi_lock.acquire()
        for handle in self._multi.handles:
            if handle.fp is not None:
                handle.fp.close()
                handle.fp = None
                handle.close()

        self._multi.close()

        self._running = False # FIXME How should I notify that the
                              # downloader is closed?
        self._multi_lock.release()
        self._down_ev.set()

    def has_fg_downloads(self):
        """
        Check if there are any downloads in the foreground.

        @rtype: boolean
        @return: C{True} if there are downloads in the foreground,
                C{False} otherwise.
        """
        self._queue_lock.acquire()
        if self._queue is not None:
            for elem in self._queue:
                if not elem.downloading:
                    self._queue_lock.release()
                    return True
        self._queue_lock.release()

    def has_bg_downloads(self):
        """
        Check if there are any downloads in the background.

        @rtype: boolean
        @return: C{True} if there are downloads in the background,
                C{False} otherwise.
        """
        self._queue_lock.acquire()
        if self._bg_queue is not None:
            for elem in self._bg_queue:
                if not elem.downloading:
                    self._queue_lock.release()
                    return True
        self._queue_lock.release()

    def has_downloads(self):
        """
        Check if there are any downloads.

        @rtype: boolean
        @return: C{True} if there are any downloads,
                C{False} otherwise.
        """
        return self.has_fg_downloads() or self.has_bg_downloads()

    def _setup_handle(self, handle, download):
        """
        Setup a handle for the given download.

        @type   handle: handle
        @param  handle: Curl handle.
        @type   download: _FileDownload
        @param  download: File Download object.

        @rtype: handle
        @return: Curl handle.
        """
        if not handle or not download:
            return

        new_file = download.path + self.PART_EXT
        try:
            handle.fp = open(new_file, "w+")
        except IOError:
            try:
                os.makedirs(os.path.dirname(new_file))
                handle.fp = open(new_file, "w+")
            except (IOError, OSError):
                return

        handle.setopt(pycurl.WRITEDATA, handle.fp)
        handle.setopt(pycurl.URL, download.url)
        handle.filename = new_file
        handle.url = download.url
        handle.data = download.data
        download.handle = handle

        return handle

    def clear_priority(self, foreground):
        """
        Changes the download queue order according to the priority.

        @type   foreground: boolean
        @param  foreground: Change the foreground or background
                            download queue.
        """
        self._queue_lock.acquire()
        if foreground:
            new_queue = [el for el in self._queue if not el.downloading]
            self._queue = new_queue
            self._queue_lock.release()
            return
        else:
            new_queue = [el for el in self._bg_queue if not el.downloading]
            self._bg_queue = new_queue
            self._queue_lock.release()
            return

    def _remove_from_queue(self, url):
        """
        Remove a download with the given I{URL} from the queue.

        @type   url: string
        @param  url: URL name.
        """
        self._queue_lock.acquire()
        new_queue = [elem for elem in self._queue if elem.url != url]
        self._queue = new_queue
        new_queue = [elem for elem in self._bg_queue if elem.url != url]
        self._bg_queue = new_queue
        self._queue_lock.release()
        if not self._queue and not self._bg_queue:
            self._down_ev.clear()

    def clear_fg_queue(self):
        """
        Clear the foreground download queue.
        """
        self._queue_lock.acquire()
        self._queue = []
        self._queue_lock.release()

    def clear_bg_queue(self):
        """
        Clear the background download queue.
        """
        self._queue_lock.acquire()
        self._bg_queue = []
        self._queue_lock.release()

    def clear_queues(self):
        """
        Clear the download queue.
        """
        self.clear_fg_queue()
        self.clear_bg_queue()

    def _newdownload(self):
        """
        Returns a new download from the download queue.

        @rtype: _FileDownload
        @return: File Download object.
        """
        found = None
        self._queue_lock.acquire()
        for idx, elem in enumerate(self._queue):
            if not elem.downloading:
                self._queue[idx].downloading = True
                found = self._queue[idx]
                break

        if found is None:
            for idx, elem in enumerate(self._bg_queue):
                if not elem.downloading:
                    self._bg_queue[idx].downloading = True
                    found = self._bg_queue[idx]
                    break

        self._queue_lock.release()

        if found and os.path.exists(found.path):
            self._remove_from_queue(found.url)
            found = True

        return found

    def run(self):
        """
        Execute the Download Manager Thread.
        """
        while self._running: # FIXME
            self._down_ev.wait()

            while self._freelist:
                new_down = self._newdownload()
                if new_down is None:
                    break

                if new_down is True:
                    continue

                self._multi_lock.acquire()
                free_handle = self._freelist.pop()
                new_handle = self._setup_handle(free_handle, new_down)
                if new_handle is not None:
                    self._multi.add_handle(new_handle)
                    self._multi.handles.append(new_handle)
                self._multi_lock.release()

            while self._running: # FIXME
                if not self._multi:
                    break

                self._multi_lock.acquire()
                ret, num_handles = self._multi.perform()
                self._multi_lock.release()

                if ret != pycurl.E_CALL_MULTI_PERFORM:
                    break

            while self._running: # FIXME
                if self._multi is None:
                    break

                self._multi_lock.acquire()

                num_q, ok_list, err_list = self._multi.info_read()
                for hnd in ok_list:
                    hnd.fp.close()
                    hnd.fp = None
                    self._multi.remove_handle(hnd)

                    if hnd in self._multi.handles:
                        self._multi.handles.remove(hnd)

                    self._remove_from_queue(hnd.url)
                    self._freelist.append(hnd)

                    err_code = hnd.getinfo(pycurl.HTTP_CODE)
                    if err_code != 200:
                        try:
                            os.remove(hnd.filename)
                        except:
                            pass
                        self._append_download(hnd.filename[:-5], hnd.data, err_code)
                    else:
                        try:
                            os.rename(hnd.filename, hnd.filename[:-5])
                        except:
                            pass
                        self._append_download(hnd.filename[:-5], hnd.data)

                    hnd.filename = None
                    hnd.url = None
                    hnd.data = None

                self._multi_lock.release()

                for hnd, errno, errmsg in err_list:
                    hnd.fp.close()
                    hnd.fp = None

                    if self._multi:
                        self._multi_lock.acquire()
                        self._multi.remove_handle(hnd)
                        if hnd in self._multi.handles:
                            self._multi.handles.remove(hnd)
                        self._multi_lock.release()

                    self._remove_from_queue(hnd.url)
                    self._freelist.append(hnd)
                    self._append_download(hnd.filename[:-5], hnd.data, errmsg)
                    hnd.filename = None
                    hnd.data = None
                    hnd.url = None

                if num_q == 0:
                    break
            if self._running:
                self._multi.select(1.0)

