#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{SetupAccountController}.
"""

from common.carlog import ERROR
from common.carmanconfig import CarmanConfig
from main.messagedlgctrl import MessageDlgController
from main.listradioctrl import ListRadioView
from maps.commonaccountctrl import CommonAccountController
from models.infosharingmodel import InfoSharingModel

class SetupAccountController(object):
    """
    Controls features related to "Setup account" option.

    @type title: string
    @param title: Title of screen.
    @type parent: L{ManageAccounts}
    @param parent: Parent window.
    @type cb: callback
    @param cb: When it is diferent of None, it is called in L{__save_account_info_cb} callback.
    """
    def __init__(self, title, parent, cb=None):
        self.__title = title
        self._parent = parent
        self.is_service = InfoSharingModel().service
        self.__info = None
        self.username = None
        self.old_protocol = None
        self.selected_protocol = None
        self.__get_account_info()
        self.cb = callable(cb) and cb or None

    def __get_account_info(self):
        """
        Returns account information from InfoShare service (__info, username and selected_protocol).
        """
        if self.is_service.account_exists():
            account = self.is_service.get_account()
            self.__info = self.is_service.get_account_info()
            self.username = self.__info['username']
            self.selected_protocol = \
                self.is_service.get_protocols()[account[1]][0]

    def __select_protocol(self, obj, item, param):
        """
        Called when a protocol is selected.

        @type obj: L{ListRadioView}
        @param obj: Not used.
        @type item: L{ListItemView}
        @param item: Not used.
        @type param: string
        @param param: Protocol string.
        """
        self.__get_account_info()
        self.list_protocol_view.select_item(param)

        self.old_protocol = self.selected_protocol
        self.selected_protocol = self.is_service.get_protocols()[param][0]

        if not self.old_protocol == self.selected_protocol:
            self.__info = self.is_service.get_protocol_options(param)
        self.__info['protocol_id'] = param

        setup = CommonAccountController(self.__title + \
                " - %s" % self.selected_protocol.upper(),
                self.__info, self.__save_account_info_cb)
        setup.show(True)

    def __save_account_info_cb(self, info):
        """
        Called when C{save} button in setup account option is clicked.

        @type info: dictionary
        @param info: Dictionary that contains datas related to users' account.
        """
        for key in info.keys():
            self.__info[key] = info[key]

        if self.is_service.account_exists() and \
                (self.old_protocol != self.selected_protocol or
                self.username != self.__info['username']):
            self.is_service.remove_account()
            self.username = None

        if not self.username:
            if not self.is_service.add_account((self.__info['username'], \
                    self.__info['protocol_id'])):
                msg_error = MessageDlgController()
                msg_error.show_message('Error saving account', title=self.__title
                        + " - %s" % self.selected_protocol.upper())

        if not self.is_service.set_account_info(self.__info):
            ERROR("Could not set account info")

        self.list_protocol_view.hide(True)
        if self.cb:
            self.cb()

        if self._parent:
            self._parent.hide(True)
            self._parent._parent.hide(True)

    def show(self):
        """
        Displays the protocol options screen.
        """
        self.list_protocol_view = ListRadioView("CHOOSE PROTOCOL")
        theme = CarmanConfig().get_current_theme()
        protocols = self.is_service.get_protocols()

        for id in protocols.keys():
            if self.__info and 'protocol_id' in self.__info and \
                    self.__info['protocol_id'] == id:
                is_checked = True
            else:
                is_checked = False
            self.list_protocol_view.add_item(protocols[id][0], \
                    self.__select_protocol, param=id, check=is_checked,
                    thumbnail=(theme, protocols[id][1]))
        self.list_protocol_view.show()
