#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ManageAccounts}.
"""

from main.listctrl import ListView
from main.messagedlgctrl import MessageDlgController
from maps.setupaccountctrl import SetupAccountController
from maps.connectaccountctrl import ConnectAccountController
from maps.deleteaccountctrl import DeleteAccountController
from models.infosharingmodel import InfoSharingModel


class ManageAccounts(ListView):
    """
    Implements methods used to manage Infoshare accounts.

    @ivar is_model: Infosharing model.
    @ivar is_service: Infosharing service.
    @param parent: Parent window.
    @type parent: L{GlobalOptionsView}
    """

    def __init__(self, parent):
        ListView.__init__(self, "list-normal", "list-item-normal",
                          "MANAGE ACCOUNT")
        self._parent = parent
        self.is_model = InfoSharingModel()
        self.is_service = self.is_model.service

        active_account = self.is_service.get_account()
        if self.is_service.account_is_connected():
            self.add_item("Disconnect account", \
                    self.__disconnect_account_cb, param=active_account[0])
        else:
            self.add_item("Connect account", \
                    self.__connect_account_cb)

        self.add_item("Setup account", self.__setup_account_cb)
        if self.is_service.account_exists():
            self.add_item("Delete account", self.__delete_account_cb)

    def __disconnect_account_cb(self, obj, item, param):
        """
        Disconnects from an account according to the user desire.

        @param obj: Object shown
        @type obj: object
        @param item: Not used
        @type item: string
        @param param: Connected account parameters
        @type param: string
        """
        def disconnect_account_cb():
            self.is_service.account_disconnect()

        msg = MessageDlgController(confirm_cb=disconnect_account_cb)
        msg.show_message("Do you want to disconnect from<br>%s?" % \
                param.split("/")[0], buttons=2, title="DISCONNECT ACCOUNT")
        obj.hide(True)
        self._parent.hide(True)

    def __connect_account_cb(self, obj, item, param):
        """
        Connects to an account if this one exists.

        @param obj: Object used to get the connect account controller
        @type obj: object
        @param item: Not used
        @type item: string
        @param param: Not used
        @type param: string
        """
        connect = ConnectAccountController(obj)
        if connect.exist_account():
            connect.connect_account()
        self._parent.hide(True)

    def __setup_account_cb(self, *p):
        """
        Checks if there are any account connected or connecting and configures a new one if the user wants.

        @param p: Not used
        @type p: set of arguments
        """
        def setup_cb():
            self.is_service.account_disconnect()
            setup = SetupAccountController("SETUP ACCOUNT", self)
            setup.show()
        def exit_cb():
            self.hide(True)
            self._parent.hide(True)

        if self.is_service.account_is_connecting() or \
                self.is_service.account_is_connected():
            msg = MessageDlgController(confirm_cb=setup_cb, cancel_cb=exit_cb)
            msg.show_message("Do you want to disconnect from<br>%s?"% \
                    self.is_service.get_account()[0].split("/")[0], \
                    buttons=2, title="SETUP ACCOUNT")
        else:
            setup = SetupAccountController("SETUP ACCOUNT", self)
            setup.show()

    def __delete_account_cb(self, obj, item, param):
        """
        Deletes an account if it exists.

        @param obj: Not used
        @type obj: object
        @param item: Not used
        @type item: string
        @param param: Not used
        @type param: string
        """
        delete = DeleteAccountController(self)
        if delete.exist_account():
            delete.delete_account()
