#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{Controller} and L{ContainerController}.
"""

import os, glob

from common.carlog import ERROR

class Controller(object):
    """
    Base controller class for L{MapsController}, L{TripsController}
    and L{GaugesController}.

    @cvar   NAME: Class name (Not used).
    """
    NAME = "Dummy"

    def get_name(self):
        """
        Returns the class name.

        @rtype: string
        @return: Class name L{NAME}.
        """
        return self.NAME

    def key_down_cb(self, obj, event):
        """
        Called when a key is pressed.

        @type   obj: class
        @param  obj: Not used.
        @type   event: EventKey
        @param  event: Event Key.
        """
        pass

    def set_unit_system(self, value):
        """
        Sets the unit system(METRIC or IMPERIAL).

        @type   value: string
        @param  value: Unit system type.
        """
        pass

    def show_options(self):
        """
        Displays controller options.
        """
        pass

    def device_connected(self, obd):
        """
        @deprecated: Method not used.

        @type   obd: object
        @param  obd: Not used.
        """
        pass

    def activate(self):
        """
        Activates the controller.
        """
        pass

    def deactivate(self):
        """
        Deactivates the controller.
        """
        pass

    def get_view(self):
        """
        Returns the Viewer for this Controller class.
        """
        pass

    def get_icon(self):
        """
        Returns the current module icon.
        """
        pass

    def set_theme(self, theme):
        """
        Sets the current theme.

        @type   theme: string
        @param  theme: Theme filename with full path.
        """
        pass

    def finalize(self):
        """
        Finishes the controller.
        """
        pass

class ContainerController(object):
    """
    The controller that will load all the plugins and options.

    @type   parent: L{MainController}
    @param  parent: Parent window.
    @type   canvas: L{evas.Canvas}
    @param  canvas: Evas canvas.
    @type   theme: string
    @param  theme: Theme filename with full path.
    """
    def __init__(self, parent, canvas, theme):
        self._parent = parent
        self._canvas = canvas
        self._theme = theme

    def __load_module(self, path, name):
        """
        Loads the module with the given name.

        @type   path: string
        @param  path: Modules directory full path.
        @type   name: string
        @param  name: Module name.

        @raise  Exception: Error opening I{module_name}: I{error_type}.

        @rtype: L{MapsController}, L{TripsController} or
                L{GaugesController} object
        @return: Module object.
        """
        try:
            module = __import__(os.path.join(path, name))
            container = module.get_container(self._parent, self._canvas,
                self._theme)
            if isinstance(container, Controller):
                return container
            else:
                DEBUG("Container %s invalid for maps" % name)
        except Exception, e:
            ERROR("error opening container %s: %s" % (name, e))

    def load_containers(self, path):
        """
        Load all the modules.

        @type   path: string
        @param  path: Modules directory full path.

        @rtype: list
        @return: List of modules.
        """
        containers = []
        list_aux = ["__init__"]

        files = glob.glob(os.path.join(path, "*.py*"))
        files.sort()
        for plugin in files:
            name = os.path.basename(plugin).rsplit('.', 1)[0]

            if name in list_aux:
                continue
            list_aux.append(name)

            container = self.__load_module(path, name)
            if container is not None:
                containers.append(container)

        return containers

