# _wrapper.py - Makes easier the creation of a bluetooth rfcomm devnode.
#
# Copyright (C) 2008 Instituto Nokia de Tecnologia - INdT
#
# This file is part of carman.
#
# carman is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# carman is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Implements L{DBusBT}.
"""

import dbus
import dbus.service
from common.dbus_bt import errors

class DBusBT(object):
    """
    Interface class to setup the serial connection via DBus.

    @cvar   STATE_STOPPED: State stopped.
    @cvar   STATE_STARTED: State started.
    @cvar   STATE_FINISHED: State finished.

    @raise  ValueError: The error callback must be callable.
    """
    (STATE_STOPED, STATE_STARTED, STATE_FINISHED) = range(3)

    def __init__(self, mainloop=None, error_cb=None):
        self._bus = None
        self._bmgr = None
        self._adapter = None
        self._bus_id = None
        self._device = None
        self._serial = None
        self._sec = None
        self._last_address = None
        self._dev_node = None
        self._discovery_state = self.STATE_STOPED
        self._discovery_list = []
        self._disc_started_cb = None
        self._disc_dev_name_cb = None
        self._dev_found_cb = None
        self._disc_completed_cb = None
        self._created_cb = None
        self._user_create_cb = None
        self._last_return = None
        if not callable(error_cb):
            raise ValueError("The error callback must be callable.")
        self._error_cb = error_cb
        self._main_loop = mainloop

    def set_disc_cb(self, started_cb=None, dev_name_cb=None, \
                    dev_found_cb=None, completed_cb=None):
        """
        Sets up the discovery callbacks.

        @type   started_cb: callback
        @param  started_cb: Discovery Started callback.
        @type   dev_name_cb: callback
        @param  dev_name_cb: Device Name Discovery callback.
        @type   completed_cb: callback
        @param  completed_cb: Completed Discovery callback.
        """
        self._disc_started_cb = started_cb
        self._disc_dev_name_cb = dev_name_cb
        self._dev_found_cb = dev_found_cb
        self._disc_completed_cb = completed_cb

    def _disc_started_signal(self):
        """
        Calls the discovery started callback.
        """
        self._discovery_state = self.STATE_STARTED
        if callable(self._disc_started_cb):
            self._disc_started_cb()

    def _disc_dev_name_signal(self, address, name):
        """
        Calls the device name discovery callback.
        """
        self._discovery_list.append((address, unicode(name)))
        if callable(self._disc_dev_name_cb):
            self._disc_dev_name_cb(address, name)

    def _disc_completed_signal(self):
        """
        Calls the completed discovery callback. Unregister all
        bluetooth signals.
        """
        self._discovery_state = self.STATE_FINISHED

        if callable(self._disc_completed_cb):
            self._disc_completed_cb()

        self._bus.remove_signal_receiver(self._disc_started_signal, \
                                           "DiscoveryStarted", \
                                           "org.bluez.Adapter", \
                                           "org.bluez", \
                                           "/org/bluez/hci0")
        self._bus.remove_signal_receiver(self._disc_dev_name_signal, \
                                          "RemoteNameUpdated", \
                                          "org.bluez.Adapter", \
                                          "org.bluez", \
                                          "/org/bluez/hci0")
        self._bus.remove_signal_receiver(self._disc_completed_signal, \
                                          "DiscoveryCompleted", \
                                          "org.bluez.Adapter", \
                                          "org.bluez", \
                                          "/org/bluez/hci0")

    def _start_bus(self):
        """
        Initializes the DBus interface for the bluetooth adapter.
        """
        if self._main_loop is None:
            return

        dbus.set_default_main_loop(self._main_loop)

        self._bus = dbus.SystemBus(self._main_loop)
        self._bmgr = dbus.Interface(self._bus.get_object('org.bluez',
                                                         '/org/bluez'),
                                                         'org.bluez.Manager')
        self._adapter = dbus.Interface(self._bus.get_object('org.bluez',
                                                            '/org/bluez/hci0'),
                                                            'org.bluez.Adapter')

    def start_discovery(self):
        """
        Initializes the discovery of devices. Register all
        bluetooth signals.
        """
        if not self._adapter:
            self._start_bus()

        self._bus.add_signal_receiver(self._disc_started_signal,
                                      "DiscoveryStarted",
                                      "org.bluez.Adapter",
                                      "org.bluez",
                                      "/org/bluez/hci0")
        self._bus.add_signal_receiver(self._disc_dev_name_signal,
                                      "RemoteNameUpdated",
                                      "org.bluez.Adapter",
                                      "org.bluez",
                                      "/org/bluez/hci0")
        self._bus.add_signal_receiver(self._disc_completed_signal,
                                      "DiscoveryCompleted",
                                      "org.bluez.Adapter",
                                      "org.bluez",
                                      "/org/bluez/hci0")

        self._adapter.DiscoverDevices()

    def cancel_discovery(self):
        """
        Cancel the discovery process.
        """
        if self._adapter:
            self._adapter.CancelDiscovery()

    def has_bonding(self, address):
        """
        Returns if the adapter has a bonding with the address.

        @type   address: string
        @param  address: Bluetooth device address to check if has
                         bonding or not.

        @rtype: boolean
        @return: C{True} if has bonding, C{False} otherwise.
        """
        if not self._adapter:
            self._start_bus()

        return self._adapter.HasBonding(address)

    def _bonding_created(self, address):
        """
        Calls the C{_created_cb} when the device with the given
        address is bonded.

        @type   address: string
        @param  address: Bluetooth device address.
        """
        if self._created_cb is not None:
            self._created_cb(address)
            self._bus.remove_signal_receiver(self._bonding_created,
                                             "BondingCreated",
                                             "org.bluez.Adapter",
                                             "org.bluez",
                                             "/org/bluez/hci0")

    def _wrapper_created_cb(self, path):
        """
        Calls the C{_user_create_cb} when the wrapper object is created.

        @type   path: string
        @param  path: Device path.

        @rtype: callback
        @return: Serial Connection created callback.
        """
        if callable(self._user_create_cb):
            self._bus.remove_signal_receiver(self._wrapper_created_cb,
                                             "PortCreated",
                                             "org.bluez.serial.Manager",
                                             "org.bluez",
                                             "/org/bluez/serial")

            return self._user_create_cb(path)

