# _passkeyagent.py - Makes it easier to create a custom passkey agent
#
# Copyright (C) 2008 Instituto Nokia de Tecnologia - INdT
#
# This file is part of carman
#
# carman is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# carman is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Implements L{CarmanPasskeyAgent}.

@var PASSKEY_AGENT_INTERFACE: DBus interface for bluetooth password key agent.
@var PASSKEY_AGENT_PATH: DBus object path for bluetooth password key agent.
"""

import dbus, dbus.service

PASSKEY_AGENT_INTERFACE = 'org.bluez.PasskeyAgent'
PASSKEY_AGENT_PATH = '/carman_passkeyagent/dev'

class CarmanPasskeyAgent(dbus.service.Object):
    """
    Implements the bluetooth password key agent.

    @type   main_loop: object
    @param  main_loop: Main loop used to monitor system bus.
    @type   address: string
    @param  address: Bluetooth device address.
    @type   request_cb: callback
    @param  request_cb: Callback to be called after agent request.
    """
    def __init__(self, main_loop, address, request_cb=None):
        if not address:
            raise ValueError("Error must be a bluetooth address.")
        self._address = address
        self._path = PASSKEY_AGENT_PATH + str(self._address).replace(":","")
        self._bus = dbus.SystemBus(mainloop=main_loop, private=True)
        try:
            dbus.service.Object.__init__(self, self._bus, self._path)
        except KeyError:
            pass # This path is already handled
        self._return_cb = None
        self._err_cb = None
        self._sec = None
        self._request_cb = callable(request_cb) and request_cb or None

    def success(self, value):
        """
        This is the asynchonous reply to the Request function.

        @type   value: string
        @param  value: The returned string of the function.
        """
        self._return_cb(value)
        self._request_cb = None

    def error(self, e):
        """
        This is the asynchronous error to the Request function.

        @type   e: object
        @param  e: Exception instance
        """
        self._err_cb(e)
        self._request_cb = None

    @dbus.service.method(dbus_interface=PASSKEY_AGENT_INTERFACE,
                         in_signature='ssb', out_signature='s',
                         async_callbacks=('return_cb', 'err_cb'))
    def Request(self, agent_path, remote_address, use_default, return_cb,
                err_cb):
        """
        Calls the request callback, if any.

        @type   agent_path: string
        @param  agent_path: Not used.
        @type   remote_address: string
        @param  remote_address: Remote bluetooth device address.
        @type   use_default: object
        @param  use_default: Not used.
        @type   return_cb: callback
        @param  return_cb: Callback to be called when agent replies.
        @type   err_cb: callback
        @param  err_cb: Callback to be called when an error occurs.
        """
        self._return_cb = return_cb
        self._err_cb = err_cb
        if self._request_cb is not None:
            self._request_cb(remote_address)
        self._request_cb = None
        return

    @dbus.service.method(dbus_interface=PASSKEY_AGENT_INTERFACE,
                         in_signature='', out_signature='')
    def Release(self):
        """
        Releases the password key agent.
        """
        self.unregister()

    def register(self):
        """
        Registers the password key agent.
        """
        if self._sec:
            return
        self._sec = dbus.Interface(self._bus.get_object('org.bluez',
                                                        '/org/bluez/hci0'),
                                                        'org.bluez.Security')
        self._sec.RegisterPasskeyAgent(self._path, self._address)

    def unregister(self):
        """
        Unregisters the password key agent.
        """
        if not self._sec:
            return
        self._sec.UnregisterPasskeyAgent(self._path, self._address)
        self._sec = None
