/*
 *
 * Copyright (C) 2008 Nokia Corporation. All rights reserved.
 *
 * Redistribution  and use  in source  and binary  forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * Redistributions  of source code must retain the above copyright
 * notice, this  list  of conditions and  the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation  and / or other materials  provided  with the
 * distribution. The name of the author may not be used to endorse
 * or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE  IS  PROVIDED  BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES  OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE  ARE  DISCLAIMED. IN  NO  EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON  ANY THEORY  OF  LIABILITY, WHETHER  IN  CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gpsctrl.h>
#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#include <glib.h>
#include <glib-object.h>

#include "location.h"

static LocationGPSDevice *gpsdev = NULL;
static LocationGPSDControl *control = NULL;
static gulong devhandler[3];
static gulong chandler[3];

static void inline locationcpy(location_t *dest,
			const LocationGPSDeviceFix *src)
{
	dest->time	= src->time;
	dest->mode	= src->mode;
	dest->latitude	= src->latitude;
	dest->longitude	= src->longitude;
	dest->altitude	= src->altitude;
	dest->track	= src->track;
	dest->speed	= src->speed*3.6;
}

static void control_release(LocationGPSDControl *ctrl)
{
	g_signal_handler_disconnect(ctrl, chandler[0]);
	g_signal_handler_disconnect(ctrl, chandler[1]);
	g_signal_handler_disconnect(ctrl, chandler[2]);

	memset(&chandler, 0, sizeof(chandler));

	if (ctrl->can_control)
		location_gpsd_control_stop(ctrl);

	g_object_unref(ctrl);
}

static void gps_release(LocationGPSDevice *device)
{
	g_signal_handler_disconnect(device, devhandler[0]);
	g_signal_handler_disconnect(device, devhandler[1]);
	g_signal_handler_disconnect(device, devhandler[2]);

	location_gps_device_stop(device);
	g_object_unref(device);

	memset(&devhandler, 0, sizeof(devhandler));
}

static void changed(LocationGPSDevice *device)
{
	int status = (device->status == 0 ?
			GPS_EVT_FIXING : GPS_EVT_FIXED);

	/*
	 * liblocation 0.26-1 doesn't report glib disconnect signal
	 * properly when using BT GPS. disconnect is not sent.
	 * Possible gps status values:
	 * 0 : no fix
	 * 1: fix
	 * 2: DGPS fix
	 */
	if (device->online) {
		location_t loc;
		locationcpy(&loc, device->fix);
		send_event(status, (guint8 *) &loc,
				sizeof(location_t));
	} else {
		send_event(GPS_EVT_DISCONNECTED, NULL, 0);
		if (gpsdev) {
			gps_release(gpsdev);
			gpsdev = NULL;
		}
		if (control) {
			control_release(control);
			control = NULL;
		}
	}
}

static void connected(LocationGPSDevice *device)
{
	/*
	 * Be careful: signal received if can_control == TRUE only
	 */
	fprintf(stderr, "[wrapper] GPS connected\n");
	send_event(GPS_EVT_CONNECTED, NULL, 0);
}

static void disconnected(LocationGPSDevice *device)
{
	/*
	 * The disconnected signal is only received when
	 * can_control is FALSE, and it means one of two
	 * things: 1. another application has stopped
	 * the GPS; 2. the GPS has been really disconnected.
	 */
	fprintf(stderr, "[wrapper] GPS disconnected\n");
	send_event(GPS_EVT_DISCONNECTED, NULL, 0);
}

static LocationGPSDevice *gps_init(void)
{
	LocationGPSDevice *device;

	device = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

	devhandler[0] = g_signal_connect(device, "changed",
				G_CALLBACK(changed), NULL);
	devhandler[1] = g_signal_connect(device, "connected",
				G_CALLBACK(connected), NULL);
	devhandler[2] = g_signal_connect(device, "disconnected",
				G_CALLBACK(disconnected), NULL);

	return device;
}

static void gpsd_stopped(LocationGPSDControl *ctrl)
{
	/*
	 * FIXME: If another apps has the control, which
	 * action location wrapper has to do, start the
	 * gpsd again?
	 */
	fprintf(stderr, "[wrapper] Control gpsd_stopped\n");
	if (gpsdev) {
		gps_release(gpsdev);
		gpsdev = NULL;
	}

	if (control) {
		control_release(control);
		control = NULL;
	}
}

static void gpsd_running(LocationGPSDControl *ctrl)
{
	fprintf(stderr, "[wrapper] Control gpsd_running\n");
}

static void gpsd_error(LocationGPSDControl *ctrl)
{
	/*
	 * liblocation 0.26-1 reports this signal
	 * when Bluetooth page timeout happens
	 */

	fprintf(stderr, "[wrapper] Control gpsd_error\n");

	if (gpsdev) {
		gps_release(gpsdev);
		gpsdev = NULL;
	}
	if (control) {
		control_release(control);
		control = NULL;
	}

	send_event(GPS_EVT_ERROR, NULL, 0);
}

static LocationGPSDControl *control_init(void)
{
	LocationGPSDControl *ctrl;

	ctrl = location_gpsd_control_get_default();
	/*
	 * When can_control is disabled means that there is another process
	 * controlling the gpsd, locationwrapper must be passive on this case.
	 */

	location_gpsd_control_request_status(ctrl);

	chandler[0] = g_signal_connect(ctrl, "gpsd_stopped",
				G_CALLBACK(gpsd_stopped), NULL);
	chandler[1] = g_signal_connect(ctrl, "gpsd_running",
				G_CALLBACK(gpsd_running), NULL);
	chandler[2] = g_signal_connect(ctrl, "error",
				G_CALLBACK(gpsd_error), NULL);

	return ctrl;

}

int location_gps_start(const char *address)
{

	/* 
	 * Control and GPS must be initialized always: liblocation becomes
	 * inconsistent when external applications disconnect BT GPS.
	 * If can_control is FALSE there is no glib signal, therefore send
	 * the notification here instead of wait GPS connected signal.
	 */

	if (control) {
		fprintf(stderr, "[wrapper] GPS control already active!\n");
		return -EPERM;
	}

	control = control_init();

	/* 
	 * FIXME: If is already connected and the client wants
	 * to connect to a different address it must fail
	 * FIXME: Store the address in a global variable
	 */

	if (control->can_control == TRUE) {
		if (strcasecmp("Internal", address) == 0) {
			gpsctrl_set_bt(0);
			gpsctrl_set_internal(1);
		} else {
			char *btdev[2];
			btdev[0] = (char *) address;
			btdev[1] = NULL;

			if (gpsctrl_set_internal(0) < 0) {
				int err = errno;
				fprintf(stderr, "Can't disable internal GPS: "
						"%s(%d)\n", strerror(err), err);
			}

			gpsctrl_set_bt(1);

			gpsctrl_set_bt_devices(btdev);
		}
	}

	gpsdev = gps_init();


	location_gpsd_control_start(control);
	location_gps_device_start(gpsdev);

	fprintf(stderr, "[wrapper] Starting GPS can_control:%d, ",
		control->can_control);

	fprintf(stderr, "a running:%d, online:%d\n",
		control->gpsbt_start_running, gpsdev->online);

	return 0;
}

int location_gps_stop(void)
{
	send_event(GPS_EVT_DISCONNECTED, NULL, 0);

	if (!gpsdev || !control)
		return -EPERM;

	fprintf(stderr, "[wrapper] Stopping GPS can_control:%d, ",
		control->can_control);

	fprintf(stderr, "running:%d, online:%d\n",
		control->gpsbt_start_running, gpsdev->online);
#if 0
	/* Release the objects on fixing state? */
	if (gpsdev->online == FALSE) {
		fprintf(stderr, "[wrapper] GPS not online\n");
		return -EPERM;
	}
#endif
	gps_release(gpsdev);
	gpsdev = NULL;

	control_release(control);
	control = NULL;

	return 0;
}

int location_init(void)
{
	g_type_init();

	if (!g_thread_supported())
		g_thread_init(NULL);

	return 0;
}

void location_exit(void)
{
	if (!gpsdev || !control)
		return;

	if (gpsdev->online && control->can_control)
		location_gps_device_stop(gpsdev);

	g_object_unref(gpsdev);

	if (control->can_control)
		location_gpsd_control_stop(control);

	g_object_unref(control);
}
