/*
 *
 * Copyright (C) 2008 Nokia Corporation. All rights reserved.
 *
 * Redistribution  and use  in source  and binary  forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * Redistributions  of source code must retain the above copyright
 * notice, this  list  of conditions and  the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation  and / or other materials  provided  with the
 * distribution. The name of the author may not be used to endorse
 * or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE  IS  PROVIDED  BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES  OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE  ARE  DISCLAIMED. IN  NO  EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON  ANY THEORY  OF  LIABILITY, WHETHER  IN  CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define GPS_DATA	"/home/user/gps_data.log"
#define DATA_INTERVAL		500

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <glib.h>

#include "location.h"

guint watch = 0, fixing = 0;
FILE *file = NULL;

int simulator_init(void)
{
	struct stat st;
	int err;

	if (stat(GPS_DATA, &st) < 0) {
		err = errno;
		fprintf(stderr, "GPS log file(" GPS_DATA ") not available: "
				"%s(%d)\n", strerror(err), err);
		return -1;
	}

	file = fopen(GPS_DATA, "r");
	if (file == NULL ) {
		err = errno;
		fprintf(stderr, "Can't open GPS log file: %s(%d)\n",
				strerror(err), err);
	}

	return 0;
}

void simulator_exit(void)
{
	if (file)
		fclose(file);
}

static gboolean send_gps_data(gpointer data)
{
	location_t loc;
	double alt, lat, lon, speed, track;
	int n;

	alt = lat = lon = speed = track = 0;

	/* lat, lon, alt, speed */
	n = fscanf(file, "%lf %lf %lf %lf", &lat, &lon, &track, &speed);

	if (n == EOF) {
		fprintf(stderr, "End of file reached\n");
		fseek(file, 0, SEEK_SET);
		return TRUE;
	} else if (n != 4) {
		fprintf(stderr, "Invalid GPS logging data format\n");
		return FALSE;
	}

	/* Fixing simulation: 128 seconds fixed <->8 seconds fixing*/
	fixing++;
	if (fixing & 0x80) {
		if (fixing == 0x80)
			fprintf(stderr, "Going to fixing after 128 seconds\n");
		if (fixing & 0x10) {
			fixing = 0;
			fprintf(stderr, "Returning to fixed after 16 seconds\n");
			return TRUE;
		}

		if (send_event(GPS_EVT_FIXING, NULL, 0) < 0)
			return FALSE;

		return TRUE;
	}

	memset(&loc, 0, sizeof(location_t));
	loc.mode	= 2; /* 2D mode */
	loc.latitude	= lat;
	loc.longitude	= lon;
	loc.altitude	= alt;
	loc.track	= track;
	loc.speed	= speed;

	if (send_event(GPS_EVT_FIXED, (guint8 *) &loc,
				sizeof(location_t)) < 0)
		return FALSE;

	return TRUE;
}

static void timeout_removed(gpointer data)
{
	fprintf(stderr, "GPS simulator stopped\n");
}

int simulator_gps_start(const char *address)
{
	if (file == NULL) {
		fprintf(stderr, "GPS log file not available: %s\n", GPS_DATA);
		return -ENODATA;
	} else if (watch) {
		fprintf(stderr, "GPS already running\n");
		return -EALREADY;
	}

	sleep(5);
	send_event(GPS_EVT_FIXING, NULL, 0);

	watch = g_timeout_add_full(G_PRIORITY_DEFAULT,
				DATA_INTERVAL, send_gps_data,
				NULL, timeout_removed);

	return 0;
}

int simulator_gps_stop(void)
{
	if (watch) {
		g_source_remove(watch);
		watch = 0;
	}

	send_event(GPS_EVT_DISCONNECTED, NULL, 0);

	return 0;
}
