/*
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Redistribution  and use  in source  and binary  forms, with or
 * without modification, are permitted provided that the following
 * conditions are met:
 *
 * Redistributions  of source code must retain the above copyright
 * notice, this  list  of conditions and  the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in
 * the documentation  and / or other materials  provided  with the
 * distribution. The name of the author may not be used to endorse
 * or promote products derived from this software without specific
 * prior written permission.
 * 
 * THIS SOFTWARE  IS  PROVIDED  BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES  OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE  ARE  DISCLAIMED. IN  NO  EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON  ANY THEORY  OF  LIABILITY, WHETHER  IN  CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <glib.h>

#include "location.h"

#define WRAPPER_SOCKET_NAME "\0/org/indt/location-wrapper"

static GMainLoop *main_loop = NULL;
static int clisk = -1;
static int parentpid = 0;

static gboolean client_watch(GIOChannel *io, GIOCondition cond, gpointer data)
{
	guint8 buf[148];
	gps_cmd_t *cmd = (gps_cmd_t *) buf;
	int err, sk;
	size_t n;
	guint8 type;

	if (cond & G_IO_NVAL)
		return FALSE;

	if (cond & (G_IO_HUP | G_IO_ERR))
		return FALSE;

	sk = g_io_channel_unix_get_fd(io);
	memset(&buf, 0, sizeof(buf));
	n = read(sk, buf, sizeof(buf));
	if (n < 0) {
		err = errno;
		fprintf(stderr, "[wrapper] read(): %s(%d)\n",
				strerror(err), err);
		return FALSE;
	}

	type = cmd->type;
	if (type & GPS_SIMULATOR_MODE) {
		if (type & GPS_CMD_CONNECT) {
			err = simulator_gps_start(cmd->data);
			if (err == -ENODATA) {
				send_event(GPS_EVT_ERROR, NULL, 0);
			}
		} else if (type & GPS_CMD_DISCONNECT) {
			simulator_gps_stop();
		}
	} else {
#ifdef LOCATION_ENABLED
		if (type & GPS_CMD_CONNECT) {
			location_gps_start(cmd->data);
		} else if (type & GPS_CMD_DISCONNECT) {
			location_gps_stop();
		}
#else
		fprintf(stderr, "[wrapper] GPS device not available!\n");
		send_event(GPS_EVT_ERROR, NULL, 0);
#endif
	}

	return TRUE;
}

static void client_watch_destroyed(gpointer data)
{
	fprintf(stderr, "[wrapper] client disconnected\n");
	g_main_loop_quit(main_loop);
}

static gboolean server_watch(GIOChannel *io, GIOCondition cond, gpointer data)
{
	int err, nsk, sk;
	socklen_t addrlen;
	struct sockaddr_un addr;

	if (cond & G_IO_NVAL)
		return FALSE;

	if (cond & (G_IO_HUP | G_IO_ERR))
		return FALSE;

	sk = g_io_channel_unix_get_fd(io);

	memset(&addr, 0, sizeof(addr));
	addrlen = sizeof(addr);
	nsk = accept(sk, (struct sockaddr *) &addr, &addrlen);
	if (nsk < 0) {
		err = errno;
		fprintf(stderr, "[wrapper] accept: %s (%d)\n",
					strerror(err), err);
		return TRUE;
	}

	fprintf(stderr, "[wrapper] Accepting connection\n");
	clisk = nsk;
	io = g_io_channel_unix_new(nsk);
	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_add_watch_full(io, G_PRIORITY_DEFAULT,
			G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
			client_watch, NULL, client_watch_destroyed);
	g_io_channel_unref(io);

	return TRUE;
}

static int unix_init(void)
{
	GIOChannel *io;
	int sk, err;
	struct sockaddr_un addr = {
		AF_UNIX, WRAPPER_SOCKET_NAME
	};

	sk = socket(PF_LOCAL, SOCK_SEQPACKET, 0);
	if (sk < 0) {
		err = errno;
		fprintf(stderr, "[wrapper] socket(): %s(%d)\n",
				strerror(err), err);
		return -err;
	}

	/* In case it already exists */
	unlink(addr.sun_path);
	if (bind(sk, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
		err = errno;
		fprintf(stderr, "[wrapper] bind(): %s(%d)\n",
				strerror(err), err);
		close(sk);
		return -err;
	}

	if (fcntl(sk, F_SETFL, fcntl(sk, F_GETFL, 0) | O_NONBLOCK) < 0) {
		err = errno;
		fprintf(stderr, "fcntl(): %s(%d)\n", strerror(err), err);
		close(sk);
		return -err;
	}

	if (listen(sk, 1) < 0) {
		err = errno;
		fprintf(stderr, "[wrapper] listen(): %s(%d)\n",
				strerror(err), err);
		close(sk);
		return -err;
	}

	io = g_io_channel_unix_new(sk);
	g_io_channel_set_close_on_unref(io, TRUE);
	g_io_add_watch(io, G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
			server_watch, NULL);
	g_io_channel_unref(io);

	return 0;
}

static void sig_term(int sig)
{
	if (main_loop)
		g_main_loop_quit(main_loop);
}

int send_event(int type, guint8 *data, size_t dsize)
{
	guint8 buf[148];
	gps_evt_t *evt = (gps_evt_t *) buf;
	size_t size;
	int n;

	if (clisk < 0)
		return -EIO;

	memset(&buf, 0, sizeof(buf));
	evt->type = type;
	evt->size = dsize;
	if (dsize)
		memcpy(evt->data, data, dsize);

	size = sizeof(gps_evt_t) + dsize;
	n = write(clisk, &buf, size);
	if (n < size) {
		int err = errno;
		fprintf(stderr, "[wrapper] write(): %s(%d)\n",
				strerror(err), err);
	}

	return n;
}

gboolean check_parentpid(gpointer ppid)
{
	int pid;

	pid = GPOINTER_TO_INT(ppid);

	/* Send o dummy signal, just to check if the process is alive */
	if (kill(pid, 0) < 0)
	{
		fprintf(stderr, "Ohh no, now i'm orphan :( [ppid: %d]\n", pid);
		fprintf(stderr, "nothing remains for me beyond suicide...\n");
		sig_term(SIGTERM);
	}

	return TRUE;
}

int main(int argc, char *argv[])
{
	struct sigaction sa;

	memset(&sa, 0, sizeof(sa));
	sa.sa_handler = sig_term;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	if (unix_init() < 0) {
		fprintf(stderr, "Can't init unix socket\n");
		exit(EXIT_FAILURE);
	}
#ifdef LOCATION_ENABLED
	if (location_init() < 0) {
		fprintf(stderr, "Can't init liblocation\n");
		exit(EXIT_FAILURE);
	}
#endif
	if (simulator_init() < 0) {
		fprintf(stderr, "Can't init GPS simulator\n");
	}

	main_loop = g_main_loop_new(NULL, FALSE);

	parentpid = getppid();
	g_timeout_add(5000, check_parentpid, GUINT_TO_POINTER(parentpid));

	g_main_loop_run(main_loop);

	g_main_loop_unref(main_loop);

#ifdef LOCATION_ENABLED
	location_exit();
#endif
	simulator_exit();

	return 0;
}
