#
#  Copyright (c) 2008 INdT - Instituto Nokia de Tecnologia
#
#  This file is part of carman-python.
#
#  carman-python is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  carman-python is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""
Implements L{ISService} and L{ISDBusModel}.

@var    DBUS_ADDRESS: DBus private bus address.
@var    DBUS_BUS_NAME: DBus bus name.
@var    DBUS_OBJECT_PATH: DBus object path.
@var    DBUS_IFACE_SERVICE: DBus infoshare interface service.
@var    __PROTOCOLS__: Dictionary of supported protocols C{(prpl-msn,
                       prpl-yahoo, prpl-jabber)}.
@var    __CARMAN_MSG__: Carman default status message.
@var    __CARMAN_GROUP__: Carman default buddy list group.
"""

import dbus

from common.carlog import ERROR
from models.dbusmodel import CarmandDbusModel, EDbusModel
from models.infosharingservice import InfoSharingService

DBUS_ADDRESS        = "unix:path=/var/run/carmand-dbus-path"
DBUS_BUS_NAME       = "org.indt.carmanplugin"
DBUS_OBJECT_PATH    = "/org/indt/carmand"
DBUS_IFACE_SERVICE  = "org.indt.carmanplugin.Service"

__PROTOCOLS__  = {'prpl-msn':    ('MSN',    'images/thumbnail-msn'),
                  'prpl-yahoo':  ('Yahoo',  'images/thumbnail-yahoo'),
                  'prpl-jabber': ('Jabber', 'images/thumbnail-jabber')}

__CARMAN_MSG__ = "Online from Carman"
__CARMAN_GROUP__ = "Carman"

class ISService(InfoSharingService):
    """
    Infoshare service

    Implements DBus connection with InfoShare Daemon. It is used as a
    "bridge" between Carman's InfoShare model and Infoshare Daemon.

    @ivar   __dbusmodel: Instance of L{ISDBusModel}.
    """

    def __init__(self):
        InfoSharingService.__init__(self)

        # Execute Infoshare daemon
        self.__dbusmodel = ISDBusModel()
        self.__dbusmodel.SetClientBusName(
                CarmandDbusModel().get_bus().get_unique_name())

        # Connect callbacks to DBus signals
        self.__connect_signals_callbacks(self.__dbusmodel.get_bus())

    def __connect_signals_callbacks(self, bus):
        """
        Connects InfoShare daemon DBus signals to callbacks. These callbacks
        then searches if there is a correspondent callback on L{_cbs} callback
        dictionary.

        @type bus: class
        @param bus: InfoShare daemon DBus connection.
        """
        bus.add_signal_receiver(self._on_buddy_signed_on, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'BuddySignedOn')

        bus.add_signal_receiver(self._on_buddy_signed_off, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'BuddySignedOff')

        bus.add_signal_receiver(self._on_connection_error, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'ConnectionError')

        bus.add_signal_receiver(self._on_received_msg, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'ReceivingImMessage')

        bus.add_signal_receiver(self._on_request_add_buddy, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'RequestAddBuddy')

        bus.add_signal_receiver(self._on_request_authorize_buddy, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'RequestAuthorizeBuddy')

        bus.add_signal_receiver(self._on_signed_on, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'SignedOn')

        bus.add_signal_receiver(self._on_signed_off, \
                bus_name=DBUS_BUS_NAME, path=DBUS_OBJECT_PATH, \
                dbus_interface = DBUS_IFACE_SERVICE, \
                signal_name = 'SignedOff')

    def account_exists(self):
        """
        Verify if InfoShare account exists.
        @rtype: boolean
        @return: C{True} if account exists, C{False} otherwise.
        """
        try:
            return self.__dbusmodel.AccountExists()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def authorize_buddy(self, name, authorize):
        """
        Notifies InfoShare daemon to authorize (or not) the given buddy.

        @type   name: string
        @param  name: Buddy name.
        @type   authorize: boolean
        @param  authorize: C{True} if buddy is authorized, C{False} if not.
        """
        self.__dbusmodel.AuthorizeBuddy(name, authorize)

    def send_message(self, name, message):
        """
        Sends a message to the given buddy.

        @type   name: string
        @param  name: Buddy name.
        @type   message: string
        @param  message: Message to be sent to buddy.
        """
        self.__dbusmodel.SendMsg(name, message)

    def add_account(self, acc_id):
        """
        Adds a new account to InfoShare daemon.

        @type   acc_id: tuple
        @param  acc_id: Account information (C{username}, C{protocol_id}).
        """
        try:
            self.__dbusmodel.NewAccount(acc_id[0], acc_id[1])
            return True
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def account_connect(self):
        """
        Connects to InfoShare account and set its status to available.
        """
        try:
            self.__dbusmodel.ConnectAccount()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
        try:
            self.__dbusmodel.SetAccountStatus('available', __CARMAN_MSG__)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_disconnect(self):
        """
        Disconnects from InfoShare account.
        """
        try:
            self.__dbusmodel.DisconnectAccount()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_enabled(self):
        """
        Verifies if account is enabled.

        @rtype: boolean
        @return: C{True} if account is enabled, C{False} otherwise.
        """
        try:
            return self.__dbusmodel.AccountIsEnabled()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def account_is_connected(self):
        """
        Verifies if account is connected.

        @rtype: boolean
        @return: C{True} if account is connected, C{False} otherwise.
        """
        try:
            return self.__dbusmodel.AccountIsConnected()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_connecting(self):
        """
        Verifies if account is connecting.

        @rtype: boolean
        @return: C{True} if account is connecting, C{False} otherwise.
        """
        try:
            return self.__dbusmodel.AccountIsConnecting()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def account_is_disconnected(self):
        """
        Verifies if account is disconnected.

        @rtype: boolean
        @return: C{True} if account is disconnected, C{False} otherwise.
        """
        try:
            return self.__dbusmodel.AccountIsDisconnected()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_account_password(self, password, remember):
        """
        Sets account password. Also enables or disables the
        'remember password' option.

        @type   password: string
        @param  password: Account password.
        @type   remember: boolean
        @param  remember: C{True} if account password should be remembered,
                          C{False} otherwise.
        """
        try:
            self.__dbusmodel.SetAccountPassword(password)
            self.__dbusmodel.SetAccountRemPwd(remember)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account_password(self):
        """
        Returns account password.

        @rtype: string
        @return: Account password. This only works if 'remember password'
                 option is enabled.
        """
        try:
            return self.__dbusmodel.GetAccountPassword()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account_info(self):
        """
        Returns account information.

        @rtype: dictionary
        @return: Dictionary of account information.
        """
        password = ''
        try:
            password = self.__dbusmodel.GetAccountPassword()
        except Exception, e:
            password = ''

        try:
            info = {
                'username': self.__dbusmodel.GetAccountUsername(),
                'protocol_id': self.__dbusmodel.GetAccountProtocolID(),
                'password': password,
                'alias': self.__dbusmodel.GetAccountAlias(),
                'remember_password': self.__dbusmodel.GetAccountRemPwd()
            }

            prpl_opts = self.get_protocol_options(info['protocol_id'])
            if 'connect_server' in prpl_opts:
                prpl_opts['server'] = prpl_opts['connect_server']

            for item in prpl_opts:
                info[item] = prpl_opts[item]

            return info
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return {}

    def get_protocol_options(self, protocol_id):
        """
        Returns the options from the given protocol.

        @type   protocol_id: string
        @param  protocol_id: Protocol identifier (eg. C{prpl-jabber}).
        @rtype: dictionary
        @return: Dictionary of protocol options.
        """
        try:
            if self.account_exists():
                prpl_opts = self.__dbusmodel.GetAccountProtocolOptions()
            else:
                prpl_opts = self.__dbusmodel.GetProtocolOptions(protocol_id)
                # Hack for google talk accounts
                if protocol_id == 'prpl-jabber':
                    prpl_opts['connect_server'] = 'talk.google.com'
                    prpl_opts['server'] = prpl_opts['connect_server']

            return prpl_opts
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return {}

    def set_protocol_options(self, info):
        """
        Sets the account protocol options.

        @type   info: dictionary
        @param  info: Dictionary of protocol options.
        """
        try:
            if 'port' in info:
                info['port'] = int(info['port'])
            self.__dbusmodel.SetAccountProtocolOptions(info)
            return True
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return False

    def get_accounts_active(self):
        """
        Returns a list of active accounts.

        @rtype: list
        @return: List of active accounts.
        """
        try:
            if self.__dbusmodel.AccountExists():
                account = self.__dbusmodel.GetAccountUsername()
                prot_id = self.__dbusmodel.GetAccountProtocolID()
                return [(account, prot_id)]
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return []

    def get_buddies_online(self):
        """
        Returns all online buddies.

        @rtype: list
        @return: List of online buddies.
        """
        try:
            buddy_list = []
            for buddy in set(self.__dbusmodel.GetCarmanBuddies()):
                if buddy != \
                        self.__dbusmodel.GetAccountUsername().split("/")[0]:
                    alias = self.get_buddy_alias(buddy)
                    if alias:
                        buddy_list.append((buddy, alias))
                    else:
                        buddy_list.append((buddy, ''))
                buddy_list.sort()

            return buddy_list
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return []

    def add_buddy(self, name, alias=None, group=None):
        """
        Adds a buddy to buddy list.

        @type   name: string
        @param  name: Buddy name (eg. C{john.doe@mail.com}).
        @type   alias: string
        @param  alias: Buddy alias.
        @type   group: string
        @param  group: Buddy group.
        """
        try:
            if group == None:
                group = 'Carman'
            if alias == None:
                alias = ''
            self.__dbusmodel.AddBuddy(name, alias, group)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def remove_buddy(self, name):
        """
        Removes a buddy from buddy list.

        @type   name: string
        @param  name: Buddy name.
        """
        try:
            self.__dbusmodel.RemoveBuddy(name)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_buddy_alias(self, name):
        """
        Returns the given buddy alias.

        @type   name: string
        @param  name: Buddy name.
        """
        try:
            alias = self.__dbusmodel.GetBuddyAlias(name)
            if not alias:
                alias = ''
            return alias
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_buddy_alias(self, name, alias):
        """
        Sets an alias to given buddy.

        @type   name: string
        @param  name: Buddy name.
        @type   alias: string
        @param  alias: Buddy alias to be set.
        """
        try:
            self.__dbusmodel.SetBuddyAlias(name, alias)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def get_account(self):
        """
        Returns InfoShare account.

        @rtype: tuple
        @return: InfoShare account (C{username}, C{protocol_id}) if any, None
                 otherwise.
        """
        try:
            username = self.__dbusmodel.GetAccountUsername()
            protocol_id = self.__dbusmodel.GetAccountProtocolID()
            return (username, protocol_id)
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)
            return None

    def remove_account(self):
        """
        Removes the InfoShare account.
        """
        try:
            self.__dbusmodel.RemoveAccount()
        except Exception, e:
            ERROR("Infoshare error: %s" % e.message)

    def set_account_info(self, info):
        """
        Sets the InfoShare account info according to given dictionary.

        @type   info: dictionary
        @param  info: Dictionary of account information.
        """
        if 'password' in info:
            try:
                self.__dbusmodel.SetAccountPassword(info['password'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['password']

        if 'alias' in info and info['alias'] != '':
            try:
                self.__dbusmodel.SetAccountAlias(info['alias'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['alias']

        if 'remember_password' in info:
            try:
                self.__dbusmodel.SetAccountRemPwd(info['remember_password'])
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False
            del info['remember_password']

        # Just a hack to jabber protocol
        # See funcs get_protocol_options and get_account_info
        if 'server' in info and info['protocol_id'] == 'prpl-jabber':
            info['connect_server'] = info['server']
            del info['server']

        if 'confirm' in info:
            del info['confirm']

        if 'username' in info:
            username = info['username']
            del info['username']
        else:
            username = ''

        if 'protocol_id' in info:
            protocol_id = info['protocol_id']
            del info['protocol_id']
        else:
            protocol_id = ''

        if len(info) > 0:
            if not self.set_protocol_options(info):
                return False

        if username != '':
            try:
                if not self.__dbusmodel.AccountExists() and not \
                        self.account_exists():
                    self.__dbusmodel.NewAccount(username, protocol_id)
                    return True
                else:
                    old_user = self.__dbusmodel.GetAccountUsername()
                    if old_user != username:
                        self.__dbusmodel.SetAccountUsername(username)
            except Exception, e:
                ERROR("Infoshare error: %s" % e.message)
                return False

        return True

    def get_protocols():
        """
        Returns a dictionary of supported protocols.

        @rtype: dictionary
        @return: Dictionar of supported protocols.
        """
        return __PROTOCOLS__
    get_protocols = staticmethod(get_protocols)

class ISDBusModel(dbus.Interface):
    """
    Abstraction of InfoShare daemon DBus connection interface.

    @ivar   _bus: InfoShare DBus connection.
    """

    def __init__(self):
        dbus.set_default_main_loop(EDbusModel().get_main_loop())
        self._bus = dbus.bus.BusConnection(DBUS_ADDRESS)
        self._bus_object = self._bus.get_object(DBUS_BUS_NAME,
                                                DBUS_OBJECT_PATH)
        dbus.Interface.__init__(self, self._bus_object,
                                        DBUS_IFACE_SERVICE)

        self.__name_owner_changed_cbs = []
        self._bus.add_signal_receiver(self.__name_owner_changed_cb,
                            dbus_interface="org.freedesktop.DBus",
                            signal_name="NameOwnerChanged")

    def __name_owner_changed_cb(self, *data):
        """
        Called when the owner of the DBus connection has changed.
        @type   data: set of arguments
        @param  data: Not used.
        """
        for cb in self.__name_owner_changed_cbs:
            cb(*data)

    def add_name_owner_changed_cb(self, cb):
        """
        Adds a callback to the list of callbacks to be called when the DBus
        connection owner has changed.

        @type   cb: callback
        @param  cb: Callback to be added.
        """
        if callable(cb) and cb not in self.__name_owner_changed_cbs:
            self.__name_owner_changed_cbs.append(cb)

    def del_name_owner_changed_cb(self, cb):
        """
        Removes a callback from the list of callbacks to be called when the
        DBus connection owner has changed.

        @type   cb: callback
        @param  cb: Callback to be removed.
        """
        if cb in self.__name_owner_changed_cbs:
            self.__name_owner_changed_cbs.remove(cb)

    def get_bus_object(self):
        """
        Returns DBus bus object.

        @rtype: class
        @return: DBus bus object.
        """
        return self._bus_object

    def get_bus(self):
        """
        Returns DBus connection.

        @rtype: class
        @return: DBus connection.
        """
        return self._bus
