#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import os
from time import time, mktime
from logging import getLogger

from feedparser import parse as parsefeed

from terra.core.manager import Manager
from terra.core.model import ModelFolder
from terra.core.task import Task
from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction

__all__ = ("PhotocastModelFolder", "PhotocastFeedModelFolder", )


log = getLogger("plugins.canola-core.photocast")
mger = Manager()
db = mger.canola_db
DefaultIcon = mger.get_class("Icon")
ImageModel = mger.get_class("Model/Media/Image")
DownloadManager = mger.get_class("DownloadManager")
CanolaError = mger.get_class("Model/Notify/Error")

download_mger = DownloadManager()


def whoami():
    import sys
    return sys._getframe(1).f_code.co_name

def is_supported_image_format(uri):
    for ext in ['jpg', 'jpeg', 'png']:
        if uri.lower().endswith(ext):
            return True
    return False


class PhotocastIcon(DefaultIcon):
    terra_type = "Icon/Folder/Task/Image/Photocast"
    icon = "icon/main_item/photos_photocast"


class PhotocastImageModel(ImageModel):
    terra_type = "Model/Media/Image/Photocast"
    table = "photocast_images"

    def __init__(self):
        ImageModel.__init__(self, "", None)

        self.feed_id = None
        self.title = None
        self.name = None
        self.desc = None
        self.date = None


class PhotocastFeedModelFolder(ModelFolder):
    terra_type = "Model/Media/Image/Shared/Photocast/Feed"
    table = "photocast_feeds"

    stmt_create = ("CREATE TABLE IF NOT EXISTS %s (id INTEGER "
                   "PRIMARY KEY, uri VARCHAR UNIQUE, "
                   "title VARCHAR, desc VARCHAR, author VARCHAR, "
                   "epoch INTEGER NOT NULL)") % table
    stmt_insert = ("INSERT INTO %s (uri, title, desc, author, epoch) "
                   "values (?, ?, ?, ?, ?)") % table
    stmt_update = ("UPDATE %s SET title = ?, desc = ?, author = ? "
                   "WHERE uri == ?") % table
    stmt_exists = "SELECT * FROM %s WHERE uri == ?" % table
    stmt_delete = "DELETE FROM %s WHERE id = ?" % table
    stmt_select = "SELECT id, uri, title, desc, author FROM %s " % table
    stmt_table_info = ("PRAGMA TABLE_INFO(%s)") % table

    stmt_by_images = ("SELECT * FROM photocast_images WHERE feed_id == %d "
                      "ORDER BY photocast_images.date DESC")

    def __init__(self, id, uri, title, desc, author):
        self.id = id
        self.uri = uri
        self.title = self.name = title
        self.desc = desc
        self.author = author

        self.download_process = None

        ModelFolder.__init__(self, self.title)

    def _insert(self):
        values = (self.uri, self.title, self.desc, self.author,
                  int(time()))
        cur = db.get_cursor()
        cur.execute(self.stmt_insert, values)
        self.id = cur.lastrowid
        db.commit()
        cur.close()

    def _update(self):
        values = (self.title, self.desc, self.author, self.uri)
        db.execute(self.stmt_update, values)

    def commit(self):
        rows = db.execute(self.stmt_exists, (self.uri, ))
        if not rows:
            self._insert()
        else:
            self._update()

    def _get_uri_from_summary(self, summary):
        try:
            orig_uri = summary[summary.find('<img src='):].split('"')[1]
            # hack image uri to recover medium image size from flickr:
            # _o (large), _s (small)
            uri = to_utf8(orig_uri.replace("_m.", "."))
            return uri
        except:
            return None

    def _get_uri_from_entry(self, entry):
        uri = None
        try:
            uri = to_utf8(entry.enclosures[0].href)
        except:
            if "link" in entry:
                uri = to_utf8(entry.link)

        if uri and is_supported_image_format(uri):
            return uri

        # If the image in not in the feed, try to get it from the
        # summary. Flickr works this way.
        if "summary" in entry:
            uri = self._get_uri_from_summary(entry.summary)
        return uri

    def _create_item_from_feed(self, entry):
        uri = self._get_uri_from_entry(entry)
        if uri is None:
            log.debug("Can't extract uri from entry.")
            return None

        item = PhotocastImageModel()
        item.feed_id = self.id
        item.uri = uri

        if "title" in entry:
            item.title = to_utf8(entry.title)
        else:
            item.title = ""

        if "summary" in entry:
            item.desc = to_utf8(entry.summary)
        else:
            item.desc = ""

        if "updated_parsed" in entry:
            item.date = int(mktime(entry.updated_parsed))
        else:
            item.date = 0
            log.debug("Could not retrieve updated_parsed from feed entry")

        return item

    def delete(self):
        log.debug("Deleting photocast feed '%s'", self.title)
        db.execute(self.stmt_delete, (self.id, ))

    def _entry_cmp_func(self, a, b):
        return b.date - a.date

    def do_load(self):
        self.is_loading = self.do_refresh()

    def do_refresh(self):
        self.cancel_refresh = False

        def remove_feed_file(feed_path):
            if os.path.exists(feed_path):
                try:
                    os.unlink(feed_path)
                except:
                    pass

        def parse_feed(feed_path):
            try:
                data = parsefeed(feed_path)
            except Exception, e:
                log.error("Exception during feed parsing %s", e)
                remove_feed_file(feed_path)
                return None
            remove_feed_file(feed_path)

            return data

        def disconnect():
            self.download_process.on_finished_remove(fetching_finished)
            self.download_process.on_cancelled_remove(disconnect)
            self.download_process = None

        def fetching_finished(exception, mimetype):
            feed_path = self.download_process.target
            disconnect()

            if exception is not None:
                remove_feed_file(feed_path)
                raise Exception("downloading process raised an exception: %s" \
                    % (exception))

            if self.cancel_refresh:
                remove_feed_file(feed_path)
                log.info("%s deleted before loading, canceling list " \
                         "refresh then." % self.__class__.__name__)
                self.inform_loaded()
                return

            t = ThreadedFunction(parsing_finished, parse_feed, feed_path)
            t.start()

        def parsing_finished(exception, feed):
            if exception is not None:
                raise Exception("%s() thread raised an exception: %s" \
                     % (whoami(), exception))

            if self.cancel_refresh:
                log.info("%s deleted before loading, cancelling " \
                         "refresh then." % self.__class__.__name__)

            if feed is None or self.cancel_refresh:
                self.inform_loaded()
                return

            items = []
            for entry in feed.entries:
                item = self._create_item_from_feed(entry)

                if item:
                    items.append(item)

            items.sort(self._entry_cmp_func)
            for item in items:
                item.parent = self
                self.append(item)

            self.inform_loaded()

        self.download_process = download_mger.add(self.uri,
            ignore_limit=True)
        self.download_process.on_finished_add(fetching_finished)
        self.download_process.on_cancelled_add(disconnect)
        self.download_process.start()

        return True # is still loading

    def do_unload(self):
        if self.download_process is not None:
            self.download_process.cancel()

        self.cancel_refresh = True
        ModelFolder.do_unload(self)

    def _create_item_from_db(self, row):
        item = PhotocastImageModel()

        item.id = row[0]
        item.feed_id = row[1]
        item.uri = row[2]
        item.title = item.name = to_utf8(row[3] or "")
        item.desc = to_utf8(row[4])
        # XXX: path is calculated on demand
        item.path = None

        return item

    def commit_changes(self):
        mger.canola_db.commit()


class PhotocastModelFolder(ModelFolder, Task):
    terra_task_type = "Task/Photocast"
    terra_type = "Model/Folder/Task/Image/Photocast"
    title = "Photocasts"
    select_order = "ORDER BY title"
    feed_class = PhotocastFeedModelFolder

    network_down_msg = "Network is down."

    def __init__(self, parent):
        Task.__init__(self)
        ModelFolder.__init__(self, self.title, parent)
        self._setup_network()
        self.query = self.feed_class.stmt_select + self.select_order

    def _check_network(self, network):
        if network is None or network.status == 0:
            self.cleanup_error = False
            msg = PhotocastModelFolder.network_down_msg
            self.state_reason = CanolaError(msg)
        else:
            self.cleanup_error = True
            self.state_reason = None

    def _setup_network(self):
        self._network = mger.get_status_notifier("Network")
        self._check_network(self._network)
        self._network.add_listener(self._check_network)

    def _insert_presets(self):
        log.info("Filling photocast with presets.")

        from ConfigParser import ConfigParser
        section = "Photocasts"
        presets_cfg = mger.terra_config.presets_cfg
        parser = ConfigParser()

        try:
            parser.readfp(open(presets_cfg, "r"))
            num = int(parser.get(section, "NumberOfEntries")) + 1
        except:
            log.error("Couldn't parse presets file.")
            return

        def get(prop, idx):
            try:
                return parser.get(section, prop + str(idx))
            except:
                return None

        cur = db.get_cursor()
        for i in xrange(1, num):
            values = (get("Location", i),
                      get("Title", i),
                      get("Description", i),
                      get("Author", i),
                      int(time()))

            if values[0] and values[1]:
                cur.execute(self.feed_class.stmt_insert, values)
            else:
                continue
        db.commit()
        cur.close()

    def do_load(self):
        table_exists = db.execute(self.feed_class.stmt_table_info)
        if not table_exists:
            db.execute(self.feed_class.stmt_create)
            self._insert_presets()

        rows = db.execute(self.query)
        for r in rows:
            self.children.append(self._create_item(r))

    def _create_item(self, row):
        id, uri, title, desc, author = row[0:5]
        title = to_utf8(title)
        desc = to_utf8(desc or "")
        author = to_utf8(author or "")
        return PhotocastFeedModelFolder(id, uri, title, desc, author)

    def item_exists(self, uri):
        rows = db.execute(PhotocastFeedModelFolder.stmt_exists, (uri, ))
        if rows:
            return rows[0]
        else:
            return None
