#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import logging
log = logging.getLogger("plugins.canola-core.images.options")

import ecore
from terra.core.manager import Manager
from terra.core.model import Model, ModelFolder
from terra.core.plugin_prefs import PluginPrefs

mger = Manager()
OptionsModelFolder = mger.get_class("Model/Options/Folder")
OptionsActionModel = mger.get_class("Model/Options/Action")
CheckListPanel = mger.get_class("Widget/CheckListPanel")
CheckListRenderer = mger.get_class("Widget/CheckListRenderer")
PanelButtonWidget = mger.get_class("Widget/PanelButton")
CheckListPanelController = mger.get_class("Controller/Settings/CheckedFolder")
OptionsCheckedFolderController = \
    mger.get_class("Controller/Options/CheckedFolder")


class ImagesOptionsModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Image/Fullscreen"
    title = "Slideshow Options"

    children_order = ["SlideshowMode", "SlideshowTime"]
    children_prefixes = ["Model/Options/Folder/Image/Fullscreen/Submenu"]


class SlideshowModeOptionsModel(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Image/Fullscreen/Submenu/SlideshowMode"
    title = "Presentation mode"

    children_order = ["SlideshowLoop", "SlideshowRandom"]
    children_prefixes = ["Model/Options/Action/Image/Fullscreen/SlideshowMode"]


class SlideshowTimeOptionModelFolder(OptionsModelFolder):
    terra_type = "Model/Options/Folder/Image/Fullscreen/Submenu/SlideshowTime"
    title = "Slideshow time"

    def __init__(self, parent=None):
        OptionsModelFolder.__init__(self, parent)
        self.times = [self.screen_controller.default_slideshow_time,
                      self.screen_controller.default_slideshow_time + 2,
                      10.0,
                      self.screen_controller.default_slideshow_time * 10,
                      self.screen_controller.default_slideshow_time * 20]

    def do_load(self):
        for time in self.times:
            SlideshowTimeModel(str(int(time)) + " seconds", time, self)

    def do_unload(self):
        self.current = None
        ModelFolder.do_unload(self)


class SlideshowOptionModel(OptionsActionModel):
    default_value = False

    def __init__(self, parent=None):
        OptionsActionModel.__init__(self, parent)
        self.checked_set()

    def pref_get(self):
        pref = PluginPrefs("slideshow")
        return pref.get(self.pref_key, self.default_value)

    def pref_set(self, value):
        log.info("Setting slideshow %s preference to %s", self.pref_key, value)
        pref = PluginPrefs("slideshow")
        pref[self.pref_key] = value
        pref.save()
        self.checked = value

    def checked_set(self):
        self.checked = self.pref_get()

    def execute(self):
        self.pref_set(not self.pref_get())


class SlideshowLoopOptionModel(SlideshowOptionModel):
    terra_type = "Model/Options/Action/Image/Fullscreen/SlideshowMode/" \
        "SlideshowLoop"
    name = "Loop photos"
    pref_key = "loop"

    def execute(self):
        SlideshowOptionModel.execute(self)
        self.screen_controller.slideshow_loop = self.checked


class SlideshowRandomOptionModel(SlideshowOptionModel):
    terra_type = "Model/Options/Action/Image/Fullscreen/SlideshowMode/" \
        "SlideshowRandom"
    name = "Random mode"
    pref_key = "random"


    def execute(self):
        SlideshowOptionModel.execute(self)
        self.screen_controller.slideshow_random = self.checked
        self.screen_controller.slideshow_random_idx = 0
        self.screen_controller.slideshow_random_list = []

        if self.screen_controller.slideshow_random:
            self.screen_controller.randomize_slideshow_list()


class SlideshowTimeModel(Model):
    def __init__(self, name, time, parent=None):
        Model.__init__(self, name, parent)
        self.time = time


class SlideshowTimeItemRenderer(CheckListRenderer):
    def _is_selected(self, v):
        parent = v.parent
        return parent.current is not None and \
            parent.children[parent.current] is v


class SlideshowTimePanel(CheckListPanel):
    def __init__(self, main_window, title, elements, theme=None):
        header_text = \
            "Choose the time interval between each picture in slideshow:"
        CheckListPanel.__init__(self, main_window, title, elements,
                                SlideshowTimeItemRenderer,
                                theme=theme, header_text=header_text)
        self.callback_ok = None
        self._setup_buttons()

    def _setup_buttons(self):
        ok = PanelButtonWidget(self.evas, "OK",
                               clicked_callback=self._cb_ok_clicked,
                               parent=self)
        self._setup_button_box(right=ok)

    def _cb_ok_clicked(self, *ignored):
        if self.callback_ok is not None:
            self.callback_ok()


class SlideshowModeOptionsCheckedController(OptionsCheckedFolderController):
    terra_type = "Controller/Options/Folder/Image/Fullscreen/Submenu/" \
        "SlideshowMode"

    def callback_item_data(self, model):
        return (model.name, model.checked)


class SlideshowTimerOptionsCheckedController(OptionsCheckedFolderController):
    terra_type = "Controller/Options/Folder/Image/Fullscreen/Submenu/" \
        "SlideshowTimer"

    def callback_item_data(self, model):
        return (model.name, model.checked)


class SlideshowTimeOptionController(CheckListPanelController):
    terra_type = "Controller/Options/Folder/Image/Fullscreen/Submenu/" \
        "SlideshowTime"

    def __init__(self, model, canvas, parent):
        CheckListPanelController.__init__(self, model, canvas, parent)

        def mark_selected(*args, **kargs):
            actual_time = self.model.screen_controller.slideshow_time

            for i, m in enumerate(self.model.children):
                if actual_time == m.time:
                    self.model.current = i
                    break

            if self.model.current is None:
                log.error("Slideshow timer differs from all possible values")
                self.model.current = 0

            self.view.redraw_item(self.model.current)
            return False

        ecore.idler_add(mark_selected)

    def _setup_view(self):
        title = self.model.name
        self.view = SlideshowTimePanel(self.parent.window, title,
                                       self.model.children)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.callback_escape = self.back
        self.view.callback_ok = self.cb_on_ok

    def cb_on_clicked(self, view, index):
        old_index = self.model.current
        self.model.current = index

        if old_index is not None:
            self.view.redraw_item(old_index)
        self.view.redraw_item(index)

    def _save_selected(self):
        model = self.model.children[self.model.current]
        settings = PluginPrefs("slideshow")
        settings['time'] = model.time
        settings.save()
        self.model.screen_controller.slideshow_time = model.time

    def cb_on_ok(self):
        self._save_selected()
        self.back()
