#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import ecore
import evas.decorators
import edje.decorators
import logging
from efl_utils.animations import TimelineAnimation
import math

from terra.core.controller import Controller
from terra.core.model import ModelFolder
from terra.core.manager import Manager
from terra.core.terra_object import TerraObject

from terra.ui.layout import Grid
from terra.ui.base import EdjeWidget
from terra.ui.screen import Screen
from terra.ui.grid import CellRenderer

__all__ = ("MainMenu", "MainMenuController",)

mger = Manager()
FatalError = mger.get_class("Model/Notify/Error/Fatal")
YesNoDialog = mger.get_class("Model/YesNoDialog")
SettingsController = mger.get_class("Controller/Folder/Settings")
log = logging.getLogger("canola.main_menu.ui")
KineticVGridWidget = mger.get_class("Widget/KineticVGrid")

OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
MainMenuOptionsModelFolder = mger.get_class("Model/Options/Folder/MainMenu")


class KineticCustomVGridWidget(KineticVGridWidget):

    def get_grid_size(self, w, h):
        """Get how many (rows, columns) will fit in given area.

        @return: C{(rows, columns)} fits in the given area.
        @rtype: tuple
        """
        if not self.elements:
            return (0, 0)

        if w <= 0 or h <= 0:
            return (0, 0)

        n_items = len(self.elements)
        c_width, c_height = self.renderer_size
        max_w = c_width

        n_horiz = w / max_w
        if n_horiz == 0:
            log.debug("no items fit on one line, forcing one")
            return (1, n_items)
        elif n_horiz > n_items:
            return (n_items, 1)
        else:
            n_vert = int(math.ceil(float(n_items) / float(n_horiz)))
            if n_vert > 1:
                last_line = n_items - ((n_vert - 1) * n_horiz)
                if 0 < last_line < n_horiz:
                    slots = n_horiz - last_line
                    reducing = slots / n_vert
                    n_horiz -= reducing

            return (n_horiz, n_vert)

    def _recalc_offset_y(self, x, y, w, h):
        n_elements = len(self.elements)
        c_width, c_height = self.renderer_size
        off_y = 0

        off_y = int(h - self.h_items * c_height) / 2

        return off_y

    def _reconfigure_renderers(self, x, y, w, h):
        if not self._dirty:
            return

        n_elements = len(self.elements)

        c_width, c_height = self.renderer_size
        w_items = w / c_width
        h_items = h / c_height

        if n_elements < w_items * h_items:
            w_items, h_items = self.get_grid_size(w, h)

        self.w_items = w_items
        self.h_items = h_items
        self.n_items = n_items = w_items * h_items

        if self.h_align < 0 and w_items > 1:
            self.offset_x = 0
            self.inc_x = int((w - w_items * c_width) / (w_items - 1))
        else:
            self.offset_x = int(self.h_align * (w - w_items * c_width))
            self.inc_x = 0

        if n_items >= n_elements:
            off_y = self._recalc_offset_y(x, y, w, h)
            self.current = 0
            self.offset = -c_height + off_y

        self.items_per_row = w_items
        if n_items >= n_elements:
            self.items_per_col = h_items
            self.last_top_left_visible = 0
        else:
            self.items_per_col = n_elements / w_items
            if n_elements % w_items:
                self.items_per_col += 1
            self.last_top_left_visible = self.items_per_col - h_items

        if h % c_height:
            self.spare_renderers_rows = 3
        else:
            self.spare_renderers_rows = 2
        h_items += self.spare_renderers_rows

        if self.last_top_left_visible < 0:
            self.last_top_left_visible = 0

        # delete unneeded
        if len(self.renderers):
            # delete unrequired rows of renderers
            while h_items < len(self.renderers):
                for r in self.renderers.pop():
                    r.delete()

            # delete unrequired columns of renderers
            if len(self.renderers) and w_items < len(self.renderers[0]):
                while w_items < len(self.renderers[0]):
                    for row in self.renderers:
                        cell = row.pop()
                        cell.delete()

        # reposition existing
        cy = y + self.offset
        for row in self.renderers:
            cx = x + self.offset_x
            for c in row:
                c.move(cx, cy)
                cx += c_width + self.inc_x
            cy += c_height

        # create required columns of renderers
        if len(self.renderers):
            if w_items > len(self.renderers[0]):
                cx = x + len(self.renderers[0]) * c_width + self.offset_x
                while w_items > len(self.renderers[0]):
                    cy = y + self.offset
                    for row in self.renderers:
                        c = self._renderer_new_get()
                        c.geometry_set(cx, cy, c_width, c_height)
                        row.append(c)
                        cy += c_height
                    cx += c_width + self.inc_x

        # create required rows of renderers
        cy = y + len(self.renderers) * c_height + self.offset
        while h_items > len(self.renderers):
            cx = x + self.offset_x
            row = []
            for i in xrange(w_items):
                c = self._renderer_new_get()
                c.geometry_set(cx, cy, c_width, c_height)
                row.append(c)
                cx += c_width + self.inc_x
            self.renderers.append(row)
            cy += c_height

        self._refill_renderers()
        self._dirty = False

        # reposition last row if it has less than w_items
        if n_items >= n_elements:
            row = self.renderers[self.h_items]
            w_n_items = n_items % n_elements
            dx = w_n_items * c_width / 2
            cx = x + self.offset_x + dx
            for c in row:
                tx, ty = c.pos_get()
                c.move(cx, ty)
                cx += c_width + self.inc_x

    def _index_at_xy(self, x, y):
        n_elements = len(self.elements)
        c_width, c_height = self.renderer_size
        ox, oy = self.pos
        x -= ox + self.offset_x
        y -= oy + self.offset

        row_idx = y / self.renderer_height

        if self.n_items >= n_elements and row_idx == self.h_items:
            w_n_items = self.n_items % n_elements
            dx = w_n_items * c_width / 2
            x -= dx

        col_idx = x / (self.renderer_width + self.inc_x)
        if col_idx >= self.items_per_row or col_idx < 0:
            return -1

        idx = (row_idx + self.current - 1) * self.items_per_row + col_idx

        if 0 <= idx < n_elements:
            return idx
        else:
            return -1


class CellRendererWidget(EdjeWidget, CellRenderer):

    def __init__(self, parent, theme=None):
        EdjeWidget.__init__(self, parent.evas, "main_menu_item", parent, theme)
        self._model = None
        self._edje_icon = None
        self._label = None
        self.callback_clicked = None

    def theme_changed(self):
        EdjeWidget.theme_changed(self)
        self.force_redraw()

    def force_redraw(self):
        m = self._model
        self._model = None
        self.value_set(m)

    def state_hidden(self):
        self.signal_emit("state,hidden", "")

    def state_default(self):
        self.signal_emit("state,default", "")

    def icon_new(self):
        return self._icon_cls(self)

    def icon_copy(self):
        icon = self.icon_new()
        icon.geometry_set(*self._edje_icon.geometry_get())
        return icon

    def _get_parent_type(self, type_name):
        try:
            type_name = type_name[:type_name.rindex("/")]
        except ValueError:
            return None
        return type_name

    def _get_icon_cls(self):
        icon_type = self._model.terra_type.replace("Model", "Icon", 1)
        while icon_type:
            try:
                return mger.get_class(icon_type)
            except ValueError:
                pass
            icon_type = self._get_parent_type(icon_type)

        raise ValueError("No icon found for %s", self._model.terra_type)

    def theme_changed(self):
        if self._model is not None:
            self.part_unswallow(self._edje_icon)
            self._edje_icon.delete()

        EdjeWidget.theme_changed(self)

        if self._model is not None:
            self._edje_icon = self.icon_new()
            self.part_swallow("icon", self._edje_icon)
            self.part_text_set("label", self._label)

    def value_set(self, v):
        if self._model is v or v is None:
            return

        if self._model is not None:
            self.part_unswallow(self._edje_icon)
            self._edje_icon.delete()

        self._model = v
        self._label = v.name
        self._icon_cls = self._get_icon_cls()
        self._edje_icon = self.icon_new()
        self.part_text_set("label", self._label)
        self.part_swallow("icon", self._edje_icon)

        # XXX: this should be done by edje, but trying to discover how
        lx, ly, lw, lh = self.part_geometry_get("label")
        ix, iy, iw, ih = self.part_geometry_get("icon")
        h = max(ly + lh, iy + ih)
        ow, oh = self.size_min_get()
        self.size_set(ow, h)

    @evas.decorators.del_callback
    def on_del(self):
        self.state_hidden()
        if self._edje_icon is not None:
            self._edje_icon.delete()
            self.part_unswallow(self._edje_icon)


class MainMenu(Screen):
    view_name = "menu"
    hpadding = 0
    vpadding = 20

    def __init__(self, canvas, main_window, title, elements=None, theme=None):
        Screen.__init__(self, canvas, "main_menu_grid", main_window, title,
                        theme)
        self.elements = elements
        self.callback_clicked = None
        self._setup_gui()

    def _setup_gui(self):
        def renderer_new(canvas):
            return CellRendererWidget(self)

        self._grid = KineticCustomVGridWidget(self, renderer_new, self.elements)
        self._grid.clicked_cb_set(self._cb_clicked)
        self.part_swallow("contents", self._grid)

        # Override options button
        self.bt_settings = EdjeWidget(self.evas, "bt_settings_default", self)

    @evas.decorators.del_callback
    def _cb_on_delete(self):
        self._grid.delete()

    def _cb_clicked(self, grid, index):
        if self.callback_clicked is not None:
            self.callback_clicked(grid.renderer_for_index(index))

    def freeze(self):
        self._grid.freeze()
        Screen.freeze(self)

    def thaw(self):
        self._grid.thaw()
        Screen.thaw(self)

    def theme_changed(self):
        Screen.theme_changed(self)
        self._grid.theme_changed()
        self.bt_settings.theme_changed()

    def custom_options_button(self):
        return self.bt_settings

    def MainMenuItem(self, icon, label, theme=None):
        """Factory of MainMenuItem associated with this menu.

        @rtype: L{MainMenuItem}
        """
        return MainMenuItem(self, icon, label, theme)

    def transition_from(self, old_view, end_callback=None):
        for r in self._grid.renderers:
            for c in r:
                c.state_default()
        Screen.transition_from(self, old_view, end_callback)

    def transition_to(self, new_view, end_callback=None):
        if new_view.view_name != "panel":
            Screen.transition_to(self, new_view, end_callback)
        elif end_callback:
            end_callback(self, new_view)

    def select_item(self, item, end_callback=None):
        def grow_and_fade((w, h, c, prog), icon, cx, cy):
            icon.size_set(int(w), int(h))
            icon.center_set(cx, cy)
            c = int(c)
            icon.color_set(c, c, c, c)
            c = int(c/2)
            for r in self._grid.renderers:
                for o in r:
                    o.color_set(c, c, c, c)

            if prog >= 1.0:
                if end_callback:
                    end_callback(self._parent_widget, item)
                icon.delete()
                for r in self._grid.renderers:
                    for o in r:
                        o.state_hidden()
                        o.color_set(255, 255, 255, 255)

        def move_to_center((x, y, prog), icon, duration2):
            c = int(255 - (prog * 128)) # 128 is 255 * 0.5
            for r in self._grid.renderers:
                for o in r:
                    o.color_set(c, c, c, c)

            icon.center_set(int(x), int(y))

            if prog >= 1.0:
                w, h = icon.size_get()
                start = (w, h, 255, 0.0)
                end = (w * 2, h * 2, 0, 1.0)
                TimelineAnimation(start, end, duration2,
                                  grow_and_fade, icon, int(x), int(y))

        icon = item.icon_copy()
        item.state_hidden()
        icon.show()
        start = icon.center + (0.0,)
        end = self.evas.rect.center + (1.0,)
        TimelineAnimation(start, end, 0.2, move_to_center, icon, 0.2)


class MainMenuController(Controller, OptionsControllerMixin):
    terra_type = "Controller/Folder/MainMenu"

    def __init__(self, model, canvas, parent):
        Controller.__init__(self, model, canvas, parent)
        self.animating = False

        self.models = model.children
        self._setup_view()

        OptionsControllerMixin.__init__(self)

    def _setup_view(self):
        self.view = MainMenu(self.evas, self.parent.view, "Canola", self.models)
        self.view.callback_clicked = self.cb_on_clicked
        self.view.show()

    def _show_notify(self, err):
        if isinstance(err, FatalError):
            def do_exit(*ignore):
                ecore.main_loop_quit()
            err.answer_callback = do_exit

        self.parent.show_notify(err)

    def cb_on_animation_ended(self, *ignored):
        self.animating = False

    def cb_on_clicked(self, view_item):
        if self.animating:
            log.info("click ignored: was animating")
            return

        model = view_item._model
        if not self.parent.can_use(model):
            self.parent.show_notify(model.state_reason)
            return

        self.animating = True
        if model.terra_type == "Model/Folder/Settings":
            self._open_settings(model)
        else:
            def cb(*ignored):
                #************************** decide it
                #self.evas.render() #glima: do we neet it?
                self.parent.use(model, self.cb_on_animation_ended)
            self.view.select_item(view_item, cb)

    def _open_settings(self, model):
        def cb(controller, *ignored):
            self.parent.use_options(controller, self.cb_on_animation_ended)
        sc = SettingsController(model, self.evas, self.parent,
                                end_callback=cb)

    def _decide_back(self, ignored, do_back):
        if do_back:
            ecore.main_loop_quit()

    def back(self):
        if self.animating:
            log.info("click ignored: was animating")
            return
        dialog = YesNoDialog("Do you want to exit Canola?", self._decide_back)
        self.parent.show_notify(dialog)

    def delete(self):
        OptionsControllerMixin.delete(self)
        self.model.unload()

    def reset(self):
        # XXX: this is here to deal with database cache when a rescan ends up
        # changing the schema (e.g. create a new table, already cached as
        # non existant).
        mger.canola_db.reconnect()

    def options_model_get(self):
        return MainMenuOptionsModelFolder(None, self)
