#
# This file is part of Canola
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#          Eduardo Lima (Etrunko) <eduardo.lima@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#
import re
from logging import getLogger

import evas.decorators

from terra.core.manager import Manager
from terra.ui.base import EdjeWidget


(STATE_INITIAL,
 STATE_DOWNLOAD_DIALOG_OPENING,
 STATE_DOWNLOADING,
 STATE_PAUSED,
 STATE_DOWNLOADED,
 STATE_QUEUED,
) = range(6)

log = getLogger("plugins.canola-core.ondemand.ui")
mger = Manager()
network = mger.get_status_notifier("Network")
ListScreen = mger.get_class("Widget/ListScreen")
OptionsControllerMixin = mger.get_class("OptionsControllerMixin")
ActionButton = mger.get_class("Widget/ActionButton")
ScrollableTextBlock = mger.get_class("Widget/ScrollableTextBlock")
BasicPanelController = mger.get_class("Controller/BasicPanel")


BaseRowRendererWidget = mger.get_class("Widget/BaseRowRenderer")
class BaseOnDemandRowRendererWidget(BaseRowRendererWidget):
    def __init__(self, parent, theme=None):
        BaseRowRendererWidget.__init__(self, parent, theme)

        self.action_btn = ActionButton(parent, theme)
        self.part_swallow("action_button", self.action_btn)

        self.action_btn.on_contents_box_collapsed_set(self.on_contents_box_collapsed)
        self.action_btn.on_contents_box_expanded_set(self.on_contents_box_expanded)
        self.action_btn.on_button_fetch_pressed_set(self.on_button_fetch_pressed)
        self.action_btn.on_button_pause_pressed_set(self.on_button_paused_pressed)
        self.action_btn.on_button_resume_pressed_set(self.on_button_resume_pressed)
        self.action_btn.on_button_delete_pressed_set(self.on_button_delete_pressed)
        self.action_btn.on_delete_shown_set(self.on_delete_shown)
        self.action_btn.on_delete_hidden_set(self.on_delete_hidden)

        # XXX: This parent here refers to the Screen, is there a better
        # way to get this?
        self.is_online = parent.is_online
        self.update_network_status(self.is_online)

    def theme_changed(self):
        self.action_btn.theme_changed()
        BaseRowRendererWidget.theme_changed(self)

    def on_contents_box_collapsed(self):
        self.action_btn.progress_set(0.0)
        if self._model.progress >= 1.0:
            self._model.data_state = STATE_DOWNLOADED

    def on_contents_box_expanded(self):
        if self._model.progress >= 1.0:
            self.action_btn.inform_fetch_finished()
        elif self._model.data_state == STATE_DOWNLOAD_DIALOG_OPENING:
            self._model.fetch()

    def on_button_fetch_pressed(self):
        if self._model is not None and self.is_online:
            self.action_btn.progress_set(0.0)
            self._model.delete_shown = False
            self._model._add_download_process()
            if self._model.downloader is not None:
                self._model.data_state = STATE_DOWNLOAD_DIALOG_OPENING
                self.action_btn.progress_show_animated()

    def on_button_paused_pressed(self):
        if self._model is not None:
            self._model.delete_shown = False
            self._model.pause()

    def on_button_resume_pressed(self):
        if self._model is not None and self.is_online:
            self._model.delete_shown = False
            self._model.resume()
        else:
            self.action_btn.state_set(ActionButton.STATE_PAUSED)

    def on_button_delete_pressed(self):
        if self._model is not None:
            self._model.delete_shown = False
            self._model.purge()

    def on_delete_shown(self):
        if self._model is not None:
            self._model.delete_shown = True

    def on_delete_hidden(self):
        if self._model is not None:
            self._model.delete_shown = False

    def update_network_status(self, is_online):
        self.is_online = is_online
        if is_online:
            self.action_btn.inform_online()
        else:
            self.action_btn.inform_offline()

    do_progress_value_set = None

    def progress_value_set(self, value, force_update=False):
        if value == 0.0 and not force_update:
             # sometimes the size cannot be determined
             # in the beginning of the download. Thus, ignore
             # 0.0 here. It set to 0.0 elsewhere when starting
             # the download
             return

        self.action_btn.progress_set(value)

        if self.do_progress_value_set is not None:
            self.do_progress_value_set(value, force_update)

        if value >= 1.0:
             self.action_btn.inform_fetch_finished()

    def unheard_value_set(self, value):
        if value:
            self.signal_emit("unread", "")
        else:
            self.signal_emit("read", "")

    do_state_set_model = None

    def state_set_model(self, state, model_changed=False):
        if state == STATE_INITIAL:
            self.signal_emit("inactive", "text")
            self.action_btn.state_set(ActionButton.STATE_FETCH)
        elif state == STATE_DOWNLOAD_DIALOG_OPENING:
            if model_changed:
                log.error("should never get here")
        elif state == STATE_DOWNLOADING:
            if model_changed:
                self.signal_emit("inactive", "text")
            self.action_btn.state_set(ActionButton.STATE_WORKING)
        elif state == STATE_PAUSED:
            if model_changed:
                self.signal_emit("inactive", "text")
            self.action_btn.state_set(ActionButton.STATE_PAUSED)
        elif state == STATE_QUEUED:
            if model_changed:
                self.signal_emit("inactive", "text")
            self.action_btn.state_set(ActionButton.STATE_QUEUED)
        elif state == STATE_DOWNLOADED:
            self.signal_emit("active", "text")
            self.action_btn.state_set(ActionButton.STATE_TRASH)

        if self.do_state_set_model is not None:
            self.do_state_set_model(state, model_changed)

        # do this after state has been updated, as the actions in the edje code
        # depend on the internal edje state
        if model_changed and state not in [STATE_INITIAL, STATE_DOWNLOADED]:
            self.progress_value_set(self._model.progress, force_update=True)

    def finish_animations(self, model):
         if model is None:
             return

         if model.data_state == STATE_DOWNLOAD_DIALOG_OPENING:
             model.fetch()

    @evas.decorators.del_callback
    def _on_del(self):
        self.action_btn.on_contents_box_collapsed_set(None)
        self.action_btn.on_contents_box_expanded_set(None)
        self.action_btn.on_button_fetch_pressed_set(None)
        self.action_btn.on_button_pause_pressed_set(None)
        self.action_btn.on_button_resume_pressed_set(None)
        self.action_btn.on_button_delete_pressed_set(None)
        self.action_btn.on_delete_shown_set(None)
        self.action_btn.on_delete_hidden_set(None)

        self.part_unswallow(self.action_btn)
        self.action_btn.delete()

        if not self._model:
            return

        self.value_unset(self._model)
        del self._model.renderer
        del self._model

    def value_unset(self, model):
        model.callback_state = None
        model.callback_progress = None
        model.callback_unheard = None
        model.renderer = None
        self.finish_animations(model)

    def do_value_set(self, model):
        """Called when a model changes"""
        if self._model and self._model.renderer is self:
            self.value_unset(self._model)

        self._model = model
        self._model.renderer = self

        model.callback_state = self.state_set_model
        model.callback_progress = self.progress_value_set
        model.callback_unheard = self.unheard_value_set

        self.part_text_set("text", model.name)

        self.state_set_model(model.data_state, True)
        self.unheard_value_set(model.unheard)

        if model.delete_shown:
           self.action_btn.delete_show()


ResizableRowRendererWidget = mger.get_class("Widget/ResizableRowRenderer")
class ResizableOnDemandRowRendererWidget(ResizableRowRendererWidget):
    row_group = "list_item_text_resizable"

    def do_value_set(self, model):
        self.part_text_set("text", model.name)


RowRendererWidget = mger.get_class("Widget/RowRenderer")
class OnDemandRowRendererWidget(BaseOnDemandRowRendererWidget, RowRendererWidget):
    row_group = "list_item_podcast"
    resizable_row_renderer = ResizableOnDemandRowRendererWidget


RowRendererWidget = mger.get_class("Widget/RowRenderer")
class OnDemandDescriptionRowRendererWidget(BaseOnDemandRowRendererWidget,
    RowRendererWidget):
    row_group = "list_item_podcast_subtext"
    resizable_row_renderer = ResizableOnDemandRowRendererWidget

    def do_state_set_model(self, state, model_changed=False):
        if model_changed:
            subtext = self._model.desc
            if subtext:
                subtext = re.sub("[ ]*<.*?>[ ]*", " ", subtext)
            else:
                subtext = "No description available"
            self.part_text_set("subtext", subtext)


class CurrentDownloadsRowRendererWidget(BaseOnDemandRowRendererWidget,
    RowRendererWidget):
    row_group = "list_item_podcast_subtext"
    resizable_row_renderer = ResizableOnDemandRowRendererWidget

    def on_button_paused_pressed(self):
        if self._model is not None:
            self.part_text_set("subtext", "Paused")

        BaseOnDemandRowRendererWidget.on_button_paused_pressed(self)

    def on_button_resume_pressed(self):
        if self._model is not None and self.is_online:
            self.part_text_set("subtext", "Resuming...")

        BaseOnDemandRowRendererWidget.on_button_resume_pressed(self)

    def do_state_set_model(self, state, model_changed=False):
        if state == STATE_INITIAL:
            self.part_text_set("subtext", "Click button to download episode")
        elif state == STATE_PAUSED:
            self.part_text_set("subtext", "Paused")
        elif state == STATE_QUEUED:
            self.part_text_set("subtext", "Queued")
        elif state == STATE_DOWNLOADED:
            self.part_text_set("subtext",
                               "Episode has been successfully downloaded")

    # This method is called every 0.5 seconds
    def do_progress_value_set(self, value, force_update=False):
        if self._model.data_state == STATE_DOWNLOADING:
            rstr = "Unknown"
            if self._model.filesize > 0 and self._model.velocity > 0:
                bytes_sec = (self._model.velocity * 1024)
                remaining_bytes = self._model.filesize - self._model.bytes_written
                rsec = remaining_bytes / bytes_sec
                rstr = "%d:%.2d:%.2d" % \
                    (rsec / 3600, ((rsec % 3600) / 60), rsec % 60)

            self.part_text_set("subtext", "Remaining time: %s at %d kb/s" \
                % (rstr, self._model.velocity))


class OnDemandListScreen(ListScreen):
    def __init__(self, canvas, main_window, title, elements,
                 row_renderer, group=None, theme=None):
        self.is_online = False
        ListScreen.__init__(self, canvas, main_window, title, elements,
                            row_renderer, group, theme)

    def update_network_status(self, is_online):
        self.is_online = is_online

        for r in self._list.renderers:
            r.update_network_status(is_online)


GenericListController = mger.get_class("Controller/Folder")
class OnDemandListController(GenericListController, OptionsControllerMixin):
    terra_type = "Controller/Folder/Media/OnDemand"
    list_group = "podcast_list"
    view_class = OnDemandListScreen

    fixed_renderer = False

    default_row_renderer = OnDemandRowRendererWidget
    alt_row_renderer = OnDemandDescriptionRowRendererWidget
    row_renderer = default_row_renderer

    def __init__(self, model, canvas, parent):
        self.choose_renderer(model)
        GenericListController.__init__(self, model, canvas, parent)
        self.model.callback_notify = self._show_notify
        self.model.callback_freeze = self._freeze_ui
        self.model.callback_thaw = self._thaw_ui

        self.is_online = True
        network.add_listener(self._network_status_changed)
        self._network_status_changed(network)

        OptionsControllerMixin.__init__(self)

    def _show_notify(self, err):
        self.parent.show_notify(err)

    def _freeze_ui(self, model):
        self.view.freeze()

    def _thaw_ui(self, model):
        self.view.thaw()
        self.view.force_redraw()

    def _network_status_changed(self, network_obj):
        was_online = self.is_online
        self.is_online = network_obj.status > 0.0

        self.view.update_network_status(self.is_online)
        if not was_online and self.is_online:
            # XXX: shouldn't download manager client resume the downloads
            # when the network goes back? The way it is now, only the
            # downloads being with controllers alive are resumed.
            self._resume_models()

    def _resume_models(self):
        for model in self.model.children:
            if model.data_state == STATE_DOWNLOADING:
                model.resume()

    def delete(self):
        self.model.callback_freeze = None
        self.model.callback_thaw = None
        self.model.callback_notify = None

        self.view.callback_clicked = None
        network.remove_listener(self._network_status_changed)

        GenericListController.delete(self)
        OptionsControllerMixin.delete(self)

    def choose_renderer(self, model):
        if self.fixed_renderer:
            return

        if model.description_visible:
            self.row_renderer = self.alt_row_renderer
        else:
            self.row_renderer = self.default_row_renderer

    def renderer_updated(self):
        def renderer_new(canvas):
            return self.row_renderer(self.view)

        self.view._list.row_renderer_set(renderer_new)

    def toggle_description(self):
        visible = self.model.description_visible
        self.model.description_visible = not visible

        self.choose_renderer(self.model)
        self.renderer_updated()

    def refresh_feed(self):
        # XXX: provide a better way to check this
        if self.model.refresh_finished_callback is not None:
            log.info("Ignoring refresh since is already refreshing.")

        self.view.throbber_start()
        def cb(*ignored):
            self.view.throbber_stop()
        self.model.refresh(force=True, end_callback=cb)

    def options_model_get(self):
        return self.model.options_model_get(self)


class CurrentDownloadsListController(OnDemandListController):
    terra_type = "Controller/Folder/Media/OnDemand/Generic/CurrentDownloads"
    list_group = "podcast_list"
    fixed_renderer = True
    row_renderer = CurrentDownloadsRowRendererWidget


class InfoOnDemandOptionController(BasicPanelController):
    terra_type = "Controller/Options/Folder/Player/OnDemand/Info"

    def __init__(self, model, canvas, parent, theme=None):
        BasicPanelController.__init__(self, model, canvas, parent)

        self._body = EdjeWidget(self.view.evas, "panel_info_podcast", self.view)
        self._desc = ScrollableTextBlock(self.view.evas, self.view)

        self._body.part_swallow("contents", self._desc)

        self.inner_contents_set(self._body)
        self.setup_information()

    def setup_information(self):
        model = self.model.screen_controller.model
        self._body.part_text_set("title", model.title)
        if model.artist != "":
            self._body.part_text_set("author", "From: " + model.artist)
        else:
            self._body.part_text_set("author", "")
        self._desc.text_set(model.desc)

    def delete(self):
        self._desc.delete()
        self._body.delete()
        BasicPanelController.delete(self)
