# -*- coding: utf-8 -*-
# Canola2 Youtube Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Adriano Rezende <adriano.rezende@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import re
import httplib
from cgi import parse_qs
from urlparse import urlparse
import urllib2

from terra.utils.encoding import to_utf8
from terra.core.threaded_func import ThreadedFunction

try:
    from xml.etree import cElementTree as ElementTree
except ImportError:
    try:
        import cElementTree as ElementTree
    except ImportError:
 	from elementtree import ElementTree


class Client(object):
    """YouTube Backend.

    This class provides an interface to search for videos on youtube server.

    @see YouTubeEntry
    """
    url_standardfeeds = "http://gdata.youtube.com/feeds/standardfeeds"
    url_video_search = "http://gdata.youtube.com/feeds/api/videos"
    url_video_request = "http://www.youtube.com/watch?v=%s"
    url_video_request_flv = "http://www.youtube.com/get_video?video_id=%s&t=%s"
    url_categories = "http://gdata.youtube.com/schemas/2007/categories.cat"
    url_video_by_category = "http://gdata.youtube.com/feeds/videos/-"

    def __init__(self):
        self.last_summary = {}

    def _request(self, url, *params):
        """Return feed content of a specific url."""
        xml = urllib2.urlopen(url % params).read()
        self.last_summary, entries = parse_youtube_xml(xml)
        return entries

    def search(self, query):
        """Search for video by keywords."""
        return self._request("%s?vq=%s",
                             self.url_video_search, urllib2.quote(query))

    def top_rated(self):
        """Return the top rated videos."""
        return self._request("%s/top_rated", self.url_standardfeeds)

    def most_viewed(self):
        """Return the most viewed videos."""
        return self._request("%s/most_viewed", self.url_standardfeeds)

    def most_recent(self):
        """Return the most recently posted videos."""
        return self._request("%s?vq=*&orderby=published",
                             self.url_video_search)

    def recently_featured(self):
        """Return the recently featured videos."""
        return self._request("%s/recently_featured",
                             self.url_standardfeeds)

    def category_list(self):
        """Return a list of video categories."""
        xml = urllib2.urlopen(self.url_categories).read()

        tree = ElementTree.fromstring(xml)
        categories = {}
        for child in tree.getchildren():
            categories[child.get('term')] = child.get('label')
        return categories

    def video_by_category(self, category_id):
        """Return videos from a specific category."""
        return self._request("%s/%s", self.url_video_by_category,
                             category_id)

    @classmethod
    def resolve_video_url(cls, video_id):
        std_headers = {
            'User-Agent': 'Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.0.1) Gecko/2008070208 Firefox/3.0.1',
            'Accept-Charset': 'ISO-8859-1,utf-8;q=0.7,*;q=0.7',
            'Accept': 'text/xml,application/xml,application/xhtml+xml,text/html;q=0.9,text/plain;q=0.8,image/png,*/*;q=0.5',
            'Accept-Language': 'en-us,en;q=0.5',
        }

        url = cls.url_video_request % str(video_id)

        request = urllib2.Request(url, None, std_headers)
        try:
            video_webpage = urllib2.urlopen(request).read()
        except (urllib2.URLError, httplib.HTTPException, socket.error), err:
            return None

        # "t" param
        mobj = re.search(r', "t": "([^"]+)"', video_webpage)
        if mobj is None:
            return None
        video_real_url = 'http://youtube.com/get_video?video_id=%s&t=%s' % (video_id, mobj.group(1))

        return video_real_url

    @classmethod
    def resolve_video_url_async(cls, video_id, end_callback):
        def resolve():
            return cls.resolve_video_url(video_id)

        def resolve_finished(exception, retval):
            if end_callback:
                end_callback(retval)

        ThreadedFunction(resolve_finished, resolve).start()


class InfoVideo(object):
    """Store information of a YouTube video."""

    def __init__(self, id, title):
        self.id = id
        self.title = title
        self.links = None
        self.rating = None
        self.authors = None
        self.view_count = 0
        self.thumbnails = None
        self.description = ""
        self.duration = 0

    def get_small_thumbnail(self):
        """Get the smallest thumb in size."""
        if not self.thumbnails:
            return None
        else:
            sizes = self.thumbnails.keys()
            sizes.sort()
            return self.thumbnails[sizes[0]][0]

    def get_large_thumbnail(self):
        """Get the largest thumb in size."""
        if not self.thumbnails:
            return None
        else:
            sizes = self.thumbnails.keys()
            sizes.sort()
            return self.thumbnails[sizes[-1]][0]


class InfoVideoAuthor(object):
    """Store information of a YouTube video author."""

    def __init__(self, name, uri=None, email=None):
        self.name = name
        self.uri = uri
        self.email = email


class InfoVideoRating(object):
    """Store information of a YouTube video rating."""

    def __init__(self, min=0, max=0, avg=0, num=0):
        self.min = min
        self.max = max
        self.avg = avg
        self.num = num


def get_namespaces(xml):
    space = {}
    ir = re.compile("<feed ([^>]+)")
    for c in ir.findall(xml)[0].split(' '):
        name, value = c.split("=")
        name = name.strip()
        value = value.strip()[1:-1]
        space[name] = value

    return space


def parse_youtube_xml(xml):
    """Parse an entry from youtube feed.

    Parse youtube feed and return summary and entries.
    """
    space = get_namespaces(xml)
    tree = ElementTree.fromstring(xml)

    summary = {}
    summary['total'] = int(tree.find("{%s}totalResults" % space['xmlns:openSearch']).text)
    summary['index'] = int(tree.find("{%s}startIndex" % space['xmlns:openSearch']).text)
    summary['items'] = int(tree.find("{%s}itemsPerPage" % space['xmlns:openSearch']).text)
    summary['links'] = {}

    for child in tree.findall("{%s}link" % space['xmlns']):
        rel = child.get("rel")
        if rel not in summary['links']:
            summary['links'][rel] = child.get("href")

    lst = []
    for child in tree.findall("{%s}entry" % space['xmlns']):
        id = child.find("{%s}id" % space['xmlns'])
        title = child.find("{%s}title" % space['xmlns'])

        info = InfoVideo(id=to_utf8(id.text).split("/")[-1],
                         title=to_utf8(title.text))

        info.updated = child.find('{%s}updated' % space['xmlns']).text
        info.published = child.find('{%s}published' % space['xmlns']).text

        info.links = {}
        for c in child.findall("{%s}link" % space['xmlns']):
            info.links[c.get("rel")] = ("", c.get("href"))

        info.authors = []
        for c in child.findall("{%s}author" % space['xmlns']):
            uri = c.find("{%s}uri" % space['xmlns'])
            name = c.find("{%s}name" % space['xmlns'])

            author = InfoVideoAuthor(name=to_utf8(name.text),
                                     uri=uri.text)
            info.authors.append(author)

        # rating
        tr = child.find("{%s}rating" % space['xmlns:gd'])
        info.rating = InfoVideoRating()
        if tr is not None:
            info.rating.min = float(tr.get("min", 0))
            info.rating.max = float(tr.get("max", 0))
            info.rating.avg = float(tr.get("average", 0))
            info.rating.num = float(tr.get("numRaters", 0))

        # viewcount
        tr = child.find("{%s}statistics" % space['xmlns:yt'])
        if tr is None:
            info.view_count = 0
        else:
            info.view_count = int(tr.get("viewCount", 0))

        # video thumbnails
        info.thumbnails = {}
        for tr in child.findall(".//{%s}group" % space['xmlns:media']):
            for description in tr.findall("{%s}description" % space['xmlns:media']):
                info.description = to_utf8(description.text)
                info.description = info.description.replace("\n", "<br>")

            for content in tr.findall('{%s}content' % space['xmlns:media']):
                info.duration = int(content.get('duration', 0))


            for c in tr.findall("{%s}thumbnail" % space['xmlns:media']):
                url = c.get("url")
                size = (int(c.get("width")), int(c.get("height")))
                if size not in info.thumbnails:
                    info.thumbnails[size] = [url,]
                else:
                    info.thumbnails[size].append(url)

        lst.append(info)

    return (summary, lst)
