# Canola2 UPnP Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Author: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import logging, urllib

from terra.core.task import Task

from terra.core.manager import Manager
from terra.core.model import Model, ModelFolder
from terra.core.threaded_model import ThreadedModelFolder
from terra.core.task import Task
from terra.core.plugin_prefs import PluginPrefs
from terra.utils.encoding import to_utf8 as decode

import upnpbrowser as upnp
import ecore
import time

__all__ = ("SharedModelFolder",)


log = logging.getLogger("plugins.shared")

mger = Manager()
PluginDefaultIcon = mger.get_class("Icon/Plugin")
ModelAudio = mger.get_class("Model/Media/Audio")
ModelVideo = mger.get_class("Model/Media/Video/Local")
ModelImage = mger.get_class("Model/Media/Image")
ModelDirectory = mger.get_class("Model/Folder/Directory")

CanolaError = mger.get_class("Model/Notify/Error")

class UPnPModelAudio(ModelAudio):
    terra_type = "Model/Media/Audio/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelAudio.__init__(self, name, None)

        self.parent = parent
        self.rating = None
        self.title = item.title
        urls = item.urls(["audio/mpeg",  "audio/mp3",
                          "audio/x-mp3", "audio/m4a",
                          "audio/x-m4a", "audio/mp4",
                          "audio/x-mp4", "audio/aac",
                          "audio/x-aac", "audio/x-realaudio",
                          "audio/x-pn-realaudio",
                          "audio/x-wav",
                          "audio/x-flac",
                          "audio/x-raw-int"])

        if urls:
            self.uri = urls[0]
        else:
            self.uri = None

        self.artist = decode(item.get_info("upnp:artist"))
        self.genre = decode(item.get_info("upnp:genre"))
        self.album = decode(item.get_info("upnp:album"))

        self.cover = self.get_cover()

    def load_cover_path(self):
        canola_prefs = PluginPrefs("settings")
        try:
            path = canola_prefs["cover_path"]
        except KeyError:
            path = canola_prefs["cover_path"] = \
                os.path.expanduser("~/.canola/covers/")
            canola_prefs.save()

        if not os.path.exists(path):
            os.makedirs(path)
        return path

    def get_cover(self):
        path = self.load_cover_path()
        artist = self.artist
        album = self.album
        cover = "%s/%s/cover.jpg" % (artist, album)
        cover = os.path.join(path, cover)
        return cover

class UPnPModelVideo(ModelVideo):
    terra_type = "Model/Media/Video/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelVideo.__init__(self, None)
        self.parent = parent
        self.title = item.title
        urls = item.urls(["video/x-ms-avi", "video/mpeg",
                          "video/x-msvideo", "video/x-divx",
                          "video/x-h263", "audio/x-realaudio",
                          "audio/x-pn-realaudio",
                          "video/x-realaudio",
                          "video/x-pn-realaudio"])
        if urls:
            self.uri = urls[0]
        else:
            self.uri = None

        self.name = name
        self.thumbnail = None

class UPnPModelImage(ModelImage):
    terra_type = "Model/Media/Image/Shared/UPnP"

    def __init__(self, name, parent, item):
        ModelImage.__init__(self, name, None)

        self.parent = parent
        self.title = item.title
        urls = item.urls(["image/png", "image/jpeg"])

        if urls:
            self.uri = urls[0]
        else:
            self.uri = None

        self.width  = None
        self.height = None

        resolution = item.get_attribute("resolution")
        if resolution:
            (width, height) = resolution.lower().split("x")
            self.width  = int(width)
            self.height = int(height)


class UPnPModelFolder(ThreadedModelFolder, ModelDirectory):
    terra_type = "Model/Folder/Directory/Shared/UPnP"

    def __init__(self, name, parent, device, folder_id, folder_name):
        ThreadedModelFolder.__init__(self, name, None)
        ModelDirectory.__init__(self, None, name, None)
        self.parent = parent
        self.device = device
        self.folder_id = folder_id
        self.folder_name = folder_name
        self.is_daemon = True

    def insert_new_items(self, items):
        array = []
        for item in items:
            if self.must_die:
                log.warn("Forced loading stop (must die)")
                return False

            upnp_item = None
            name = decode(item.title)
            klass = decode(item.klass.lower())

            if "playlist" in klass:
                continue
            elif "container" in klass:
                upnp_item = UPnPModelFolder(name, self, self.device, item.id, name)
            elif "audio" in klass:
                upnp_item = UPnPModelAudio(name, self, item)
            elif "video" in klass:
                upnp_item = UPnPModelVideo(name, self, item)
            elif "image" in klass:
                upnp_item = UPnPModelImage(name, self, item)
            else:
                log.info("UPnP item not match, item class:%s", klass)
                continue

            if isinstance(upnp_item, UPnPModelFolder) or upnp_item.uri:
                self.append(upnp_item)
                time.sleep(0.1)

        return True

    def do_load(self):
        time.sleep(1)

        start = 0
        screen_count = 30

        items = self.device.list_folder(self.folder_id,
                                        start=start,
                                        max_count=screen_count)

        if items and self.insert_new_items(items):
            array = []
            max_count = screen_count
            while items and len(items) == max_count:
                if self.must_die:
                    log.warn("Forced loading stop (must die)")
                    break

                start += len(items)
                max_count = 1
                items = self.device.list_folder(self.folder_id,
                                                start=start,
                                                max_count=max_count)
                array.extend(items)
                time.sleep(0.3)

                if len(array) > screen_count:
                    self.insert_new_items(array)
                    array = []

            self.insert_new_items(array)

        if not self.is_loaded:
            self.inform_loaded()


    def do_unload(self):
        log.info("%d UPnP items removed", len(self.children))
        ThreadedModelFolder.do_unload(self)

class UPnPSearchDevicesIcon(PluginDefaultIcon):
    terra_type = "Icon/Folder/Task/Shared/UPnPSearchDevices"
    icon = "icon/main_item/upnp_search"
    plugin = "upnp"

class UPnPSearchDevicesModelFolder(ThreadedModelFolder, Task):
    terra_type = "Model/Folder/Task/Shared/UPnPSearchDevices"
    terra_task_type = "Task/Shared/UPnP"

    icon = "upnp_search"
    server_down_msg  = "Server is going down."
    network_down_msg = "Network is down."

    def __init__(self, parent):
        Task.__init__(self)
        ThreadedModelFolder.__init__(self, "Search Servers", parent)
        self.parent = parent
        self._setup_network()
        self.is_daemon = True
        self.cleanup_error = True
        self.control_point = upnp.Browser()
        self.fd_handler = None
        self.callback_fd = None

    def _get_state_reason(self):
        return ThreadedModelFolder._get_state_reason(self)

    def _set_state_reason(self, value):
        msg = UPnPSearchDevicesModelFolder.server_down_msg
        if value and value.message == msg:
            log.info('Invalid state reason message: "%s"', value.message)
            return

        ThreadedModelFolder._set_state_reason(self, value)

    state_reason = property(_get_state_reason, _set_state_reason)

    def _check_network(self, network):
        msg = UPnPSearchDevicesModelFolder.network_down_msg
        if network.status == 0:
            self.cleanup_error = False
            self.state_reason = CanolaError(msg)
        else:
            self.cleanup_error = True
            self.state_reason = None

    def _setup_network(self):
        self._network = Manager().get_status_notifier("Network")
        self._check_network(self._network)
        self._network.add_listener(self._check_network)

    def new_device_found(self, device):
        if not 'ContentDirectory' in device.list_services():
            log.info("UPnP device: %s doesn't have ContentDirectory service", device.name)
            return

        for item in self.children:
            if item.device.udn == device.udn:
                log.info("UPnP device: %s already in list", device.name)
                return

        folder = UPnPModelFolder(decode(device.name), self, device, "0", "")
        self.append(folder)
        time.sleep(0.1)

    def del_device(self, udn):
        msg = UPnPSearchDevicesModelFolder.server_down_msg
        for item in self.children:
            if item.device.udn == udn:
                if self.is_valid and item.is_valid:
                    item.state_reason = CanolaError(msg)
                self.remove(item)
                break

    def start_search(self):
        self.control_point.start_search()

    def stop_search(self):
        self.control_point.stop_search()

    def callback_devices(self, fd_handler):
        udn, status = upnp.read_data(fd_handler.fd)
        status = status.upper()
        if status == "ADDED" or status == "UPDATED":
            device  = self.control_point.get_device(udn)
            self.new_device_found(device)
        elif status == "REMOVED":
            self.del_device(udn)
        else:
            log.info("Device: %s %s", udn, status)

        if not self.is_loaded:
            self.inform_loaded()

        return True

    def do_load(self):
        time.sleep(1)

        self.callback_fd = self.control_point.notification_fd()
        self.start_search()

        for device in self.control_point.list_devices():
            if self.must_die:
                log.warn("Forced loading stop (must die)")
                return

            self.new_device_found(device)

        self.fd_handler = ecore.fd_handler_add(self.callback_fd,
                                               ecore.ECORE_FD_READ,
                                               self.callback_devices)

    def do_unload(self):
        log.info("%d UPnP devices removed", len(self.children))
        self.stop_search()

        if self.fd_handler:
            self.fd_handler.stop()

        if self.callback_fd:
            self.control_point.close_fd(self.callback_fd)

        if self.cleanup_error:
            log.info("Cleanning state reason")
            self.state_reason = None

        ThreadedModelFolder.do_unload(self)
