#! /usr/bin/python
# - coding: utf-8 -*-

# Copyright (c) 2007-2008 Rnato Chencarek. All rights reserved.
#
# Author: Renato Chencarek <renato.chencarek@gmail.com>
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED,
# THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
# PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR
# CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
# EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
# PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


import re
import urllib
import os, sys, shutil
import optparse
import socket
import signal
import string

from Queue import Queue
from threading import Thread
from threading import currentThread

import ecore, ecore.evas, edje, evas

import pyid3lib

import cPickle

process_list = Queue()
ui_updates = Queue()
rfd, wfd = os.pipe()

is_done = False

accents_table = string.maketrans( 'ѡ',
                                  'aaaaaeeeeiiiiooooouuuuconAAAAEEEEIIIIOOOOUUUUCON!?')

try:
    canola_data = cPickle.load(open(os.path.expanduser('~/.canola/prefs/settings')))
except Exception, e:
    print "Settings Error! Run Canola to fix this."
    print e
    sys.exit()

try:
    from sqlite3 import dbapi2 as sqlite
except ImportError:
    from pysqlite2 import dbapi2 as sqlite

class AlbumCover(object):
    def __init__(self, artist, album):
        self.artist = artist
        self.album  = album

class LastFMAlbumCover(AlbumCover):
    base_url = "http://www.last.fm/music/%s/%s"
    img_re = re.compile(r'''<div class="cover">.*img src="([^"]+)"''')

    def __init__(self, artist=None, album="+albums"):
        artist = artist.replace(" ", "+")
        album  = album.replace(" ", "+")
        AlbumCover.__init__(self, artist, album)

    def fetch(self):
        if self.artist.upper() in ['COMPILATION', 'VARIOUS', 'VARIOUS ARTISTS']:
            artist = "Various+Artists"
        else:
            artist = self.artist

        url = self.base_url % (urllib.quote(artist), urllib.quote(self.album))
        try:
            data = urllib.urlopen(url).read()
        except:
            return []
        if not data:
            return []

        return self.img_re.findall(data)

class AmazonAlbumCover(AlbumCover):
    re_large  = re.compile(r'''<ImageUrlLarge>(.*)</ImageUrlLarge>''')
    re_medium = re.compile(r'''<ImageUrlMedium>(.*)</ImageUrlMedium>''')
    re_small  = re.compile(r'''<ImageUrlSmall>(.*)</ImageUrlSmall>''')
    license_key = "D1ESMA5AOEZB24"
    host_addr = socket.gethostbyname("xml.amazon.com")

    def __init__(self, artist=None, album="+albums"):
        AlbumCover.__init__(self, artist, album)

    def fetch(self):
        #xml.amazon.com
        url = "http://%s/onca/xml3?f=xml&t=webservices-20" % self.host_addr
        url += "&dev-t=%s" % self.license_key
        url += "&type=lite&page=1&mode=music&KeywordSearch="
        url += "%s%s" % (urllib.quote(self.artist + " "), urllib.quote(self.album))

        try:
            data = urllib.urlopen(url).read()
        except:
            return []

        if not data:
            return []

        ret = self.re_large.findall(data)
        if not len(ret):
            ret = self.re_medium.findall(data)
            if not len(ret):
                ret = self.re_small.findall(data)

        return ret

def remove_unused_covers(path):

    cover = os.path.join(path, "cover.jpg")
    cover_small = os.path.join(path, "cover-small.jpg")

    if not os.path.exists(cover) or not os.path.exists(cover_small):
        return False

    try:
        relink = os.readlink(cover)
        keep_cover = os.path.basename(relink)
    except:
        keep_cover = None

    try:
        relink = os.readlink(cover_small)
        keep_cover_small = os.path.basename(relink)
    except:
        keep_cover_small = None

    files = os.listdir(path)

    files.remove("cover.jpg")
    files.remove("cover-small.jpg")

    if keep_cover:
        files.remove(keep_cover)
    if keep_cover_small:
        files.remove(keep_cover_small)

    for f in files:
        os.unlink(os.path.join(path, f))

    return True

def cleanup_covers(data, options):
    destdir = options.destdir
    artists = os.listdir(destdir)

    for artist in artists:
        albums = os.listdir(os.path.join(destdir, artist))
        keep_artist = False
        for album in albums:
            if (decode(artist), decode(album)) not in data.keys():
                shutil.rmtree(os.path.join(destdir, artist, album))
            else:
                if remove_unused_covers(os.path.join(destdir, artist, album)):
                    keep_artist = True
                else:
                    shutil.rmtree(os.path.join(destdir, artist, album))
        if not keep_artist:
            shutil.rmtree(os.path.join(destdir, artist))


def download_covers(cover_store, artist, album, i, path):
    for url in cover_store(artist, album).fetch():
        url = url.strip()
        if not url:
            continue

        try:
            data = urllib.urlopen(url).read()
        except:
            continue

        if not data:
            continue

        name = "%s/cover-%d.jpg" % (path, i)
        name_small = "%s/cover-small-%d.jpg" % (path, i)
        f = open(name, "wr")
        f.write(data)
        f.close()

        process_list.put((150, 150, name, name_small))
        process_list.put((300, 300, name, name))

        i = i + 1

    return i

def find_embeded_covers(path, i, files, all_id3):
    for f in files:
        try:
            id3 = pyid3lib.tag(f)
            data = id3[id3.index('APIC')]['data']

            name = "%s/cover-%d.jpg" % (path, i)
            name_small = "%s/cover-small-%d.jpg" % (path, i)

            f = open(name, "wr")
            f.write(data)
            f.close()

            process_list.put((150, 150, name, name_small))
            process_list.put((300, 300, name, name))

            i = i + 1
            if not all_id3:
                return i
        except:
            continue

    return i

def find_local_covers(path, i, dirs):
    files = []
    for d in dirs:
        files.extend([os.path.join(d, p) for p in os.listdir(d)])

    for f in files:
        low = f.lower()
        if low.endswith(".jpg") or low.endswith(".jpeg") or low.endswith(".png") or low.endswith(".bmp"):
            name = "%s/cover-%d.jpg" % (path, i)
            name_small = "%s/cover-small-%d.jpg" % (path, i)
            try:
                shutil.copyfile(f, name)
            except Exception, e:
                print "Copy Error", f, name
                print e
                continue
            else:
                process_list.put((150, 150, name, name_small))
                process_list.put((300, 300, name, name))

                i = i + 1
    return i

def process_resize(set_text, set_pos):
    global is_done

    while True:
        w, h, input, output = process_list.get()

        if w < 0 and h < 0:
            set_text("Done")
            set_pos(1)
            is_done = True
            return

        ee = ecore.evas.Buffer(w=w, h=h)
        ee.show()

        canvas = ee.evas

        try:
            img1 = canvas.Image(file=input, size=(w, h))
        except:
            continue

        img1.smooth_scale_set(True)
        img1.fill_set(0, 0, w, h)
        img1.show()

        img2 = canvas.Image(size=(w, h))
        img2.image_size_set(w, h)
        img2.image_data_set(ee)
        img2.save(output)

        img2.delete()
        img1.delete()

def process_items(data, options, set_text, set_pos):

    pos = 0
    total = len(data)
    for artist,album in data.keys():
        pos += 1

        files = data[(artist, album)]
        artist = decode(artist)
        album  = decode(album)

        if "??" in artist or "??" in album:
            continue

        path = options.destdir + "/%s/%s" % (artist, album)
        exists = os.path.exists(os.path.join(path, "cover.jpg"))

        if options.skip and exists:
            continue

        if not os.path.exists(path):
            os.makedirs(path)

        set_text("%s - %s" % (artist, album))

        i = 1

        if "id3" in options.source:
            i = find_embeded_covers(path, i, files, options.all_id3)

        if "local" in options.source:
            i = find_local_covers(path, i, unique_dirname_list(files))

        if "amazon" in options.source:
            i = download_covers(AmazonAlbumCover, artist, album, i, path)

        if "lastfm" in options.source:
            i = download_covers(LastFMAlbumCover, artist, album, i, path)

        set_pos(float(pos) / float(total))

        if i == 1:
            continue

        link = os.path.join(path, "cover.jpg")
        link_small = os.path.join(path, "cover-small.jpg")

        if not os.path.exists(link):
            try:
                os.symlink("cover-1.jpg", link)
            except:
                shutil.copyfile(os.path.join(path, "cover-1.jpg"), link)
            try:
                os.symlink("cover-small-1.jpg", link_small)
            except:
                shutil.copyfile(os.path.join(path, "cover-small-1.jpg"), link_small)

    if options.clean:
        set_text("Cleaning unused covers")
        cleanup_covers(data, options)
        set_pos(1)
        set_text("Done")

    process_list.put((-1, -1, None, None))

def decode(data):
    ret = data
    for codec in ["ISO-8859-15", "ISO-8859-1", "UTF-8"]:
        try:
            ret = data.decode(codec)
        except LookupError:
            continue
        except UnicodeEncodeError:
            continue
        else:
            break

    return ret.encode("UTF-8")

def generate_data(location):
    try:
        con = sqlite.connect(location)
        cur = con.cursor()
        cur.execute("SELECT DISTINCT audio_albums.name,audio_artists.name,files.path\
                     FROM audios,files,audio_albums,audio_artists WHERE audios.id = files.id AND files.dtime = 0\
                     AND audios.album_id = audio_albums.id AND audio_albums.artist_id = audio_artists.id")
        items = cur.fetchall()
        con.close()
    except Exception, e:
        print "Database Error! Run Canola to fix this."
        print e
        sys.exit()

    data = {}
    for p in items:
        data.setdefault((p[1], p[0]), []).append(str(p[2]))
    return data

def unique_dirname_list(files):
    s = set(os.path.dirname(e) for e in files)
    return s

def cb_done(*a):
    global is_done
    if is_done:
        ecore.main_loop_quit()

def load_ui(edj):
    if ecore.evas.engine_type_supported_get("software_x11_16") and \
    '--x11' not in sys.argv:
        cls = ecore.evas.SoftwareX11_16
    else:
        cls = ecore.evas.SoftwareX11

    ee = cls(w=720, h=420)
    ee.title = "Album Cover Download"
    ee.name_class = ("AlbumCover", "AlbumCover")
    ee.evas.font_hinting_set(evas.EVAS_FONT_HINTING_AUTO)
    canvas = ee.evas

    bg = canvas.Rectangle(color=(255, 255, 255, 255), size=canvas.size)
    bg.show()

    pb = edje.Edje(canvas, file=edj, group="progressbar")
    pb.signal_callback_add("action,clicked", "done", cb_done)

    ee.data["pb"] = pb

    def resize_cb(ee):
        h = ee.data["pb"].size_min_get()[1]
        ee.data["pb"].geometry_set(10, (ee.evas.size[1] - h) / 2, ee.evas.size[0] - 20, h)

    resize_cb(ee)
    ee.callback_resize = resize_cb

    pb.part_drag_value_set("knob", 0, 0)
    pb.part_text_set("label", "Loading")

    pb.show()
    ee.show()

    return ee, pb

def process_input(parser):

    parser.add_option("-f", "--from", dest="source",
                      action="append", metavar="[LOCAL, ID3, AMAZON, LASTFM]",
                      help="specify one or more sources to get covers.",)

    parser.add_option("", "--all_id3", dest="all_id3",
                      action="store_true",
                      help="Search covers inside id3 tags in all songs of same album.")

    parser.add_option("-d", "--db", dest="db", default=os.path.expanduser('~/.canola/canola.db'),
                      action="store", metavar="CANOLA_DATABASE",
                      help="specify canola's database location [%default]")

    parser.add_option("-a", "--artist", dest="artist",
                      action="store", help="artist name")

    parser.add_option("-b", "--album", dest="album",
                      action="store", help="album name")

    parser.add_option("-p", "--path", dest="path",
                      action="store", metavar="PATH",
                      help="path to audio file or directory to search covers")

    parser.add_option("-o", "--output", dest="destdir", default=canola_data["cover_path"],
                      action="store", metavar="CANOLA_COVERS_DIR",
                      help="specify canola's covers location [%default]")

    parser.add_option("-s", "--skip", dest="skip",
                      action="store_true",
                      help="skip already downloaded album covers")

    parser.add_option("", "--x11", dest="x11",
                      action="store_true",
                      help="use x11 engine")

    parser.add_option("-q", "--quiet", action='store_true',
                      dest="quiet",
                      help="Print dots and sharps")

    parser.add_option("-e", "--edje", dest="edj", default="/usr/share/canola-tuning/pb.edj",
                      action="store", metavar="THEME_LOCATION",
                      help="theme location [%default]")

    parser.add_option("-n", "--no-ui", dest="no_ui",
                      action="store_true",
                      help="no User Interface")

    parser.add_option("-c", "--clean", dest="clean",
                      action="store_true",
                      help="Cleanup unused covers and empty folders")


    return parser.parse_args()

def move_default_cover_location(src, dest):
    try:
        shutil.copytree(src, dest)
        shutil.rmtree(src)
    except Exception, e:
        print "Cannot move default cover location to %s" % dest
        print e
        sys.exit(-1)

    try:
        canola_data["cover_path"] = dest
        cPickle.dump(canola_data,
                     open(os.path.expanduser('~/.canola/prefs/settings'), "wb"),
                     cPickle.HIGHEST_PROTOCOL)
    except Exception, e:
        print "Cannot set Canola settings"
        print e
        sys.exit(-1)

if __name__ == "__main__":
    parser = optparse.OptionParser("usage: python albumcover.py [options]")
    options, args = process_input(parser)

    artist = options.artist
    album = options.album
    database = options.db
    options.destdir = options.destdir.strip('"').strip("'")

    if options.destdir != canola_data["cover_path"]:
        move_default_cover_location(canola_data["cover_path"],
                                    options.destdir)

    if options.source:
        options.source = [s.lower() for s in options.source]
    else:
        options.source = []

    ui = not options.no_ui

    if ui:
        ee, pb = load_ui(options.edj)

    if artist or album:
        database = None
        if not artist or not album:
            parser.error("missing option artist/album")

        artist = artist.strip('"').strip("'")
        album  = album.strip('"').strip("'")
        data = {(artist,album):[options.path]}
    else:
        data = generate_data(options.db)

    def set_text(text):
        if ui:
            ui_updates.put(text)
            os.write(wfd, "0")

    def set_pos(pos):
        if ui:
            ui_updates.put(pos)
            os.write(wfd, "1")

    def update_ui(fd_handler, bar):
        tag = os.read(fd_handler.fd, 1)
        val = ui_updates.get()

        if tag == "0":
            bar.part_text_set("label", val)
        elif tag == "1":
            bar.part_drag_value_set("knob", val, 0)
        else:
            return False

        return True

    resizer = Thread(target=process_resize, args=(set_text, set_pos))
    resizer.setDaemon(True)
    resizer.start()

    downloader = Thread(target=process_items, args=(data, options, set_text, set_pos))
    downloader.setDaemon(True)
    downloader.start()

    def abort(signum, frame):
        sys.exit(-1)

    signal.signal(signal.SIGINT, abort)

    if ui:
        ecore.fd_handler_add(rfd, ecore.ECORE_FD_READ, update_ui, pb)
        ecore.main_loop_begin()
        del ee
    else:
        resizer.join()

