# -*- coding: utf-8 -*-
# Canola2 Tuning Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Renato Chencarek <renato.chencarek@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import ecore
import etk
import evas
import logging

from terra.core.manager import Manager

log = logging.getLogger("plugins.canola-tuning.ui")

mger = Manager()

CheckItemRenderer = mger.get_class("Renderer/EtkList/CheckItem")
EtkThrobber = mger.get_class("Widget/EtkThrobber")
ModalController = mger.get_class("Controller/Modal")
ModalSlider = mger.get_class("Widget/Settings/ModalSlider")
ModalThrobber = mger.get_class("Widget/Settings/ModalThrobber")
SliderContentList = mger.get_class("Widget/Settings/SliderContentList")


class ModalThrobberController(ModalController):
    terra_type = "Controller/Folder/Settings/ModalThrobber"
    wait_message = ""

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        self.view = ModalThrobber(self.parent.last_panel.view,
                                  self.wait_message,
                                  hborder=16, vborder=100)
        self.view.callback_escape = self.cb_on_cancel
        self.view.callback_clicked = self.cb_on_cancel
        self.view.show()

        self.update()

    def update(self):
        def end(retval, error):
            self.stop_finished()
            self.do_finish()

        def cb_callback(retval, error):
            # delay for message
            ecore.timer_add(1.5, end, retval, error)

        self.do_start(cb_callback)

    def do_start(self, end_callback):
        raise NotImplementedError("must be implemented by subclasses")

    def do_finish(self):
        pass

    def on_cancel(self):
        pass

    def cb_on_cancel(self):
        self.on_cancel()
        self.stop_finished()

    def stop_finished(self):
        self.view.hide(self.parent.back)

    def delete(self):
        self.view.delete()

#############################################################################
# CheckList Modal Controller
#############################################################################

class CheckContentList(SliderContentList):
    def __init__(self, evas_obj, item_clicked=None):
        SliderContentList.__init__(self, evas_obj)
        self.callback_item_clicked = item_clicked

    def columns_get(self):
        def cb_ui_func(row):
            return (row.name, row.checked)

        def cb_item_click(row, list):
            if self.callback_item_clicked:
                self.callback_item_clicked(row)

        renderer = CheckItemRenderer(ui_func=cb_ui_func,
                                     item_click=cb_item_click)
        return [(100, renderer, True)]


class CheckListView(ModalSlider):
    HEADER_AREA_HEIGHT = 50

    def __init__(self, parent, title, cb_item_clicked):
        ModalSlider.__init__(self, parent.last_panel.view, title,
                             parent.last_panel.view.theme, hborder=16,
                             vborder=20)

        self.lst = CheckContentList(self.evas,
                                    cb_item_clicked)
        self.lst.show()
        self.center_contents_set(self.lst.object)

        self.header_label = etk.Label("")
        self.header_label.alignment_set(0.0, 0.5)
        self.header_label.show()

        self.embed = None
        self.throbber = None
        self.title_separator.show()

        self.caption = etk.HBox()
        self.caption.border_width_set(15)
        self.caption.append(self.header_label, etk.HBox.START,
                            etk.HBox.EXPAND_FILL, 0)
        self.caption.show()
        self.header_vbox.append(self.caption, etk.VBox.START,
                                etk.VBox.EXPAND_FILL, 0)

        self.header_area_show()

        self.button_box_show()
        self.ok_button = self.button_add("   OK   ")
        self.back_button = self.button_add("   Cancel   ")

        self.collapsed = False
        self.callback_ok_clicked = None
        self.callback_back_clicked = None
        self.callback_cancel_clicked = None

    def add_item(self, model):
        if model.children:
            self.lst.item_append(model.children[-1])

    def _on_button_clicked(self, bt):
        if bt == self.ok_button:
            if self.callback_ok_clicked:
                self.callback_ok_clicked()
        elif bt == self.back_button:
            if self.callback_back_clicked:
                self.callback_back_clicked()

    def _cancel_clicked(self, *ignored):
        if self.callback_cancel_clicked:
            self.callback_cancel_clicked()

    def _check_destroy_embed(self):
        if self.embed is not None:
            self.embed.destroy()
            self.embed = None

    def _colapse_embed(self, content):
        self._check_destroy_embed()
        self.title = ""
        self.embed = etk.Embed(self.evas)
        self.embed.add(content)
        self.contents = self.embed.object
        if not self.collapsed:
            self.half_expand()
        self.embed.show()
        self.slider_contents.hide()
        self.collapsed = True

    def colapse_wait(self, message):
        self.throbber = EtkThrobber(message)
        self.throbber_align = \
            etk.Alignment(0.5, 0.4, 0.0, 0.0, child=self.throbber)
        self.throbber_align.show()

        self.vbox = etk.VBox()
        self.vbox.border_width = 25
        self.vbox.append(self.throbber_align, etk.VBox.START, etk.VBox.FILL, 0)

        self.cancel_button = etk.Button(label="   Cancel   ")
        self.cancel_button.on_clicked(self._cancel_clicked)
        self.cancel_button.show()
        self.button_align = etk.Alignment(0.5, 0.5, 0.0, 0.0,
                                          child=self.cancel_button)
        self.button_align.show()
        self.vbox.append(self.button_align, etk.VBox.END, etk.VBox.FILL, 0)
        self.vbox.show()
        self.throbber.start()
        self.throbber.show()

        self._colapse_embed(self.vbox)

    def colapse_message(self, message):
        label = etk.Label(message)
        label.theme_group = "modal_label"
        label.show()
        label_align = etk.Alignment(0.5, 0.4, 0.0, 0.0, child=label)
        label_align.show()

        vbox = etk.VBox()
        vbox.border_width = 25
        vbox.append(label_align, etk.VBox.START, etk.VBox.FILL, 0)

        ok_button = etk.Button(label="      OK      ")
        ok_button.on_clicked(self._cancel_clicked)
        ok_button.show()
        button_align = etk.Alignment(0.5, 0.5, 0.0, 0.0, child=ok_button)
        button_align.show()
        vbox.append(button_align, etk.VBox.END, etk.VBox.FILL, 0)
        vbox.show()

        self._colapse_embed(vbox)

    @evas.decorators.del_callback
    def _destroy_throbber(self):
        self._check_destroy_embed()
        self.lst.destroy()
        self.lst = None


class ModalCheckList(ModalController):
    terra_type = "Controller/Folder/Settings/ModalCheckList"
    title = None
    header_message = ""

    def __init__(self, model, canvas, parent):
        ModalController.__init__(self, model, canvas, parent)

        self.view = CheckListView(self.parent, self.title or model.title,
                                  self.cb_item_clicked)
        self.view.callback_ok_clicked = self.cb_ok
        self.view.callback_back_clicked = self.back
        self.view.callback_cancel_clicked = self.cb_cancel
        self.view.header_label.text = self.header_message
        self.view.show()

    def item_clicked(self):
        pass

    def ok_clicked(self):
        pass

    def cancel_clicked(self):
        pass

    def add_item(self, model):
        self.view.add_item(model)

    def colapse_wait(self, message):
        self.view.colapse_wait(message)

    def _get_ok_button_text(self):
        return self.view.ok_button.label

    def _set_ok_button_text(self, value):
        self.view.ok_button.label = value

    ok_button_text = property(_get_ok_button_text,
                              _set_ok_button_text)

    def _get_cancel_button_text(self):
        return self.view.back_button.label

    def _set_cancel_button_text(self, value):
        self.view.back_button.label = value

    cancel_button_text = property(_get_cancel_button_text,
                                  _set_cancel_button_text)

    def _get_ok_button_enabled(self):
        return self.view.ok_button.disabled != 0

    def _set_ok_button_enabled(self, value):
        if value:
            self.view.ok_button.disabled = 0
        else:
            self.view.ok_button.disabled = 1

    ok_button_enabled = property(_get_ok_button_enabled,
                                 _set_ok_button_enabled)

    def _get_cancel_button_enabled(self):
        return self.view.back_button.disabled != 0

    def _set_cancel_button_enabled(self, value):
        if value:
            self.view.back_button.disabled = 0
        else:
            self.view.back_button.disabled = 1

    cancel_button_enabled = property(_get_cancel_button_enabled,
                                     _set_cancel_button_enabled)

    def cb_ok(self):
        self.ok_clicked()

    def cb_cancel(self):
        self.cancel_clicked()

    def cb_item_clicked(self, model):
        self.item_clicked(model)

    def close(self):
        if self.view is not None:
            self.view.hide(self.parent.back)

    def delete(self):
        self.view.delete()
        self.view = None

    def back(self):
        self.close()


class ScanCoversController(ModalCheckList):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/ScanCovers"
    header_message = "Choose the cover art sources."

    def __init__(self, model, canvas, parent):
        ModalCheckList.__init__(self, model, canvas, parent)
        self.ok_button_text = " Update "
        self.model.changed_callback_add(self.view.add_item)
        self.model.load()
        self.check_disable_button()

    def ok_clicked(self):
        self.colapse_wait(" Updating covers... ")
        self.model.update(self.on_finished, self.on_progress)

    def on_progress(self, fname, progress, status):
        if self.view and self.view.throbber:
            self.view.throbber.text = " Updating covers (%s)" % progress

    def cancel_clicked(self):
        self.model.cancel()
        self.close()
        self.refresh_view()

    def item_clicked(self, model):
        model.checked = not model.checked
        model.updated = False
        self.check_disable_button()

    def on_finished(self, error, retval):
        self.close()
        self.refresh_view()

    def refresh_view(self):
        pass

    def check_disable_button(self):
        self.ok_button_enabled = (self.model.select_count() > 0)

    def delete(self):
        self.model.changed_callback_del(self.view.add_item)
        ModalCheckList.delete(self)
        self.model.unload()


class ScanCoversItemController(ScanCoversController):
    terra_type = "Controller/Options/Folder/Media/Audio/AlbumArt/ScanCovers"

    def __init__(self, model, canvas, parent):
        ScanCoversController.__init__(self, model, canvas, parent)
        self.parent.screen_controller.view.album_set_show()

    def ok_clicked(self):
        model = self.parent.screen_controller.model
        model = model.children[self.parent.screen_controller.current]

        self.colapse_wait(" Updating covers... ")
        self.model.update(model.artist, model.name, self.on_finished)

    def on_finished(self, error, retval):
        if retval == 0:
            self.view.colapse_message("No covers found for this album"
                                      "<br>using the selected sources")
            ecore.timer_add(3.0, ScanCoversController.on_finished,
                            self, error, retval)
        else:
            ScanCoversController.on_finished(self, error, retval)

    def refresh_view(self):
        view = self.parent.screen_controller.view
        view.current_album.cb_cover_right()


class CleanUnusedCoversController(ModalThrobberController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/CleanUnusedCovers"
    wait_message = " Removing unused covers..."

    def do_start(self, end_callback):
        self.model.update(end_callback)

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()


class CleanUnusedCoversItemController(ModalThrobberController):
    terra_type = "Controller/Options/Folder/Media/Audio/AlbumArt/CleanUnused"
    wait_message = " Removing unused covers..."

    def do_start(self, end_callback):
        view = self.parent.screen_controller.view
        view.album_set_show()
        self.model.update(view.current_album._cover_path, end_callback)

    def do_finish(self):
        view = self.parent.screen_controller.view
        view.current_album.cb_cover_right()

    def on_cancel(self):
        self.model.cancel()


class VideoThumbnailController(ModalThrobberController):
    terra_type = "Controller/Settings/Folder/MediaLibrary/Tuning/VideoThumbnail"
    wait_message = " Updating thumbs..."

    def do_start(self, end_callback):
        self.model.update(end_callback, self.on_progress)

    def on_progress(self, fname, progress, status):
        if self.view and self.view.throbber:
            self.view.throbber.text = " Updating thumbs (%s)" % progress

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()


class VideoPlayerThumbnailController(ModalThrobberController):
    terra_type = "Controller/Options/Folder/Player/Video/Thumbnail"
    wait_message = "  Updating thumbnail..."

    def do_start(self, end_callback):
        controller = self.parent.screen_controller
        self.model.update(controller.model.uri,
                          controller.get_position(), end_callback)

    def do_finish(self):
        pass

    def on_cancel(self):
        self.model.cancel()
