# -*- coding: utf-8 -*-
# Canola2 Tuning Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Renato Chencarek <renato.chencarek@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import os
import md5
import Image
import ecore
import shutil
import urllib
import pyid3lib

from Queue import Queue
from remote_cover_searchers import AmazonCoverSearcher, LastfmCoverSearcher


def resize_image(input, output, w, h):
    try:
        im = Image.open(input)
        im.resize((w, h)).save(output, "JPEG")
    except:
        pass


def decode(data):
    ret = data
    for codec in ["ISO-8859-15", "ISO-8859-1", "UTF-8"]:
        try:
            ret = data.decode(codec)
        except LookupError:
            continue
        except UnicodeEncodeError:
            continue
        else:
            break
    return ret.encode("UTF-8")


class TuningManager(object):
    remote_searchers = {"amazon": AmazonCoverSearcher,
                        "lastfm": LastfmCoverSearcher}

    def __init__(self, data, get_continue=None, set_progress=None):
        self.data = data
        self._count = 0
        self._get_continue = get_continue
        self._set_progress_cb = set_progress

        self.ui_updates = Queue()
        self.rfd, self.wfd = os.pipe()
        ecore.fd_handler_add(self.rfd, ecore.ECORE_FD_READ, self._update_ui)

    def _update_ui(self, fd_handler):
        tag = os.read(fd_handler.fd, 1)

        if tag != "1":
            return False

        values = self.ui_updates.get()
        if self._set_progress_cb is not None:
            self._set_progress_cb(*values)

        return True

    def _set_progress(self, label, progress, status):
        if self._set_progress_cb is not None:
            self.ui_updates.put((label, progress, status))
            os.write(self.wfd, "1")

    def cleanup_covers(self, destdir):
        artists = os.listdir(destdir)
        artist_count = len(artists)
        for i, artist in enumerate(artists):
            keep_artist = False
            if not self._get_continue():
                return
            for album in os.listdir(os.path.join(destdir, artist)):
                if (decode(artist), decode(album)) not in self.data.keys():
                    shutil.rmtree(os.path.join(destdir, artist, album))
                elif TuningManager.remove_unused_covers(os.path.join(destdir, artist, album)):
                    keep_artist = True
                else:
                    shutil.rmtree(os.path.join(destdir, artist, album))
            if not keep_artist:
                shutil.rmtree(os.path.join(destdir, artist))

            progress = "%d/%d" % (i + 1, artist_count)
            self._set_progress(artist, progress, "success")

    @classmethod
    def remove_unused_covers(cls, path):
        cover = os.path.join(path, "cover.jpg")
        cover_small = os.path.join(path, "cover-small.jpg")

        if not os.path.exists(cover) or not os.path.exists(cover_small):
            for f in os.listdir(path):
                os.unlink(os.path.join(path, f))
            return True

        try:
            relink = os.readlink(cover)
            keep_cover = os.path.basename(relink)
        except:
            keep_cover = None

        try:
            relink = os.readlink(cover_small)
            keep_cover_small = os.path.basename(relink)
        except:
            keep_cover_small = None

        files = os.listdir(path)
        files.remove("cover.jpg")
        files.remove("cover-small.jpg")

        if keep_cover:
            files.remove(keep_cover)

        if keep_cover_small:
            files.remove(keep_cover_small)

        for f in files:
            os.unlink(os.path.join(path, f))

        return True

    def _create_covers(self, path, data, count):
        name = os.path.join(path, "cover-%d.jpg" % count)
        name_small = os.path.join(path, "cover-small-%d.jpg" % count)
        f = open(name, "wr")
        f.write(data)
        f.close()
        resize_image(name, name_small, 150, 150)
        resize_image(name, name, 300, 300)

    def _fetch_id3_covers(self, path, files, all_id3=True):
        for f in files:
            try:
                id3 = pyid3lib.tag(f)
                for frame in id3:
                    if frame['frameid'] == 'APIC':
                        data = frame['data']
                        self._create_covers(path, data, (self._count + 1))
                        self._count += 1

                if not all_id3:
                    return
            except:
                continue

    def _fetch_local_covers(self, path, dirs):
        files = []
        for d in dirs:
            files.extend([os.path.join(d, p) for p in os.listdir(d)])

        for f in files:
            low = f.lower()
            if low.endswith(".jpg") or low.endswith(".jpeg"):
                name = os.path.join(path, "cover-%d.jpg" % (self._count + 1))
                name_small = os.path.join(path, "cover-small-%d.jpg" % (self._count + 1))
                try:
                    resize_image(f, name_small, 150, 150)
                    resize_image(f, name, 300, 300)
                    self._count += 1
                except:
                    continue

    def _fetch_remote_covers(self, path, cover_store, artist, album):
        for url in cover_store(artist, album).fetch():
            try:
                data = urllib.urlopen(url.strip()).read()
            except:
                continue

            if not data:
                continue

            self._create_covers(path, data, self._count + 1)
            self._count += 1

    def process_albums(self, destdir, source, skip=False, partist=None, palbum=None):
        if partist and palbum:
            lstdata = [(partist, palbum)]
        else:
            lstdata = self.data.keys()

        result = 0
        total = len(lstdata)
        for i, (artist, album) in enumerate(lstdata):
            files = self.data[(artist, album)]
            artist, album = decode(artist), decode(album)

            if not self._get_continue():
                return

            if "??" in artist or "??" in album:
                progress = "%d/%d" % (i + 1, total)
                self._set_progress("%s - %s" % (artist, album),
                                   progress, "invalid")
                continue

            path = os.path.join(destdir, artist, album)
            exists = os.path.exists(os.path.join(path, "cover.jpg"))

            if skip and exists:
                progress = "%d/%d" % (i + 1, total)
                self._set_progress("%s - %s" % (artist, album),
                                   progress, "skip")
                continue

            if not os.path.exists(path):
                os.makedirs(path)

            self._count = 0

            if "id3" in source:
                self._fetch_id3_covers(path, files)

            if "local" in source:
                unique_dirlist = set(os.path.dirname(e) for e in files)
                self._fetch_local_covers(path, unique_dirlist)

            for key, searcher in self.remote_searchers.items():
                if key in source:
                    self._fetch_remote_covers(path, searcher, artist, album)

            if self._count == 0:
                progress = "%d/%d" % (i + 1, total)
                self._set_progress("%s - %s" % (artist, album),
                                   progress, "nothing found")
                continue

            result += self._count

            link = os.path.join(path, "cover.jpg")
            link_small = os.path.join(path, "cover-small.jpg")

            if not os.path.exists(link):
                try:
                    os.symlink("cover-1.jpg", link)
                except:
                    shutil.copyfile(os.path.join(path, "cover-1.jpg"), link)

                try:
                    os.symlink("cover-small-1.jpg", link_small)
                except:
                    shutil.copyfile(os.path.join(path, "cover-small-1.jpg"), link_small)

            progress = "%d/%d" % (i + 1, total)
            self._set_progress("%s - %s" % (artist, album),
                               progress, "found %d covers" % self._count)

        return result

    def process_videos(self, files, destdir, time, width=150, height=90, skip=True):
        if not os.path.exists(destdir):
            os.makedirs(destdir)

        i, fcount = 2, len(files)
        for j, f in enumerate(files):
            # check to stop thread
            if not self._get_continue():
                return

            filename = md5.new(f).hexdigest()
            filename = os.path.join(destdir, filename + ".jpg")

            if os.path.exists(filename) and skip:
                progress = "%d/%d" % (j + 1, fcount)
                self._set_progress(f, progress, "skip")
                continue

            cmd = "mplayer -idx -really-quiet -zoom -ss %s -nosound " \
                "-vo jpeg:outdir=%s -vf scale=%d:%d -frames 2 \"%s\" > /dev/null 2>&1" \
                % (str(time), destdir, width, height, f)

            os.system(cmd)

            if not os.path.exists(os.path.join(destdir, "%08d.jpg" % i)):
                # mplayer error. thumb not generated
                progress = "%d/%d" % (j + 1, fcount)
                self._set_progress(f, progress, "error")
                continue

            shutil.move(os.path.join(destdir, "%08d.jpg" % i), filename)
            os.unlink(os.path.join(destdir, "%08d.jpg" % (i - 1)))

            progress = "%d/%d" % (j + 1, fcount)
            self._set_progress(f, progress, "success")
