# Canola2 Last.fm Plugin
# Copyright (C) 2008 Instituto Nokia de Tecnologia
# Authors: Adriano Rezende <adriano.rezende@openbossa.org>
#          Artur Duque de Souza <artur.souza@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# Additional permission under GNU GPL version 3 section 7
#
# If you modify this Program, or any covered work, by linking or combining it
# with Canola2 and its core components (or a modified version of any of those),
# containing parts covered by the terms of Instituto Nokia de Tecnologia End
# User Software Agreement, the licensors of this Program grant you additional
# permission to convey the resulting work.

import time
import ecore
import logging
from time import mktime, localtime

from Queue import Queue
from terra.core.manager import Manager
from terra.core.plugin_prefs import PluginPrefs
from terra.core.threaded_func import ThreadedFunction

from manager import LastfmManager


mger = Manager()
lastfm_manager = LastfmManager()
PlayerHook = mger.get_class("Hook/Player")
network = mger.get_status_notifier("Network")

log = logging.getLogger("plugins.canola-lastfm.scrobbler")


class AudioScrobbler(PlayerHook):
    terra_type = "Hook/Player/Audio"
    time_cons = 240
    submit_max_cache = 400
    submit_delay = 35 # (seconds)
    submit_max_posted = 40
    submit_max_retries = 5

    def __init__(self):
        PlayerHook.__init__(self)
        self._model = None
        self._timer = None
        self._length = 0
        self.start_time = None
        self._timer_paused = False
        self.prefs = PluginPrefs("lastfm")
        self.sending_submits = False
        self.pending_submits = Queue()
        self.submit_max_cache = int(self.prefs.get("scrobbler_max_cache",
                                                   self.submit_max_cache))
        self.submit_max_posted = int(self.prefs.get("scrobbler_max_posted",
                                                    self.submit_max_posted))
        self.submit_max_retries = int(self.prefs.get("scrobbler_max_retries",
                                                     self.submit_max_retries))

    def media_changed(self, model):
        """Function that is called everytime that the Player's Controller
        changes the model.
        """
        log.warning("media changed to %s" % model.title)

        self.remove_timer()
        self._model = model
        self._length = 0
        self.start_time = None

    def _validate_cmd(self, submit=False, tracks=None):
        dsc = bool(submit) and "submit" or "now playing"

        if submit:
            complement = str(tracks)
        else:
            complement = "%s - %s" % (self._model.name, self._model.album)

        if not lastfm_manager.get_username() or not lastfm_manager.get_password():
            log.warning("%s ignored (user or pass empty): %s" % (dsc, complement))
            return False

        if not lastfm_manager.scrobble_enabled:
            log.warning("%s ignored (scrobble disabled): %s" % (dsc, complement))
            return False

        if lastfm_manager.session_id and not lastfm_manager.post_session_id:
            log.warning("%s ignored (no post sessiond id): %s" % (dsc, complement))
            return False

        log.warning("sending %s: %s" % (dsc, complement))
        return True

    def _connected(self):
        return (network and network.status > 0.0)

    def _now_playing(self):
        def cb_finished(exception, retval):
            if exception is None:
                log.warning("now playing sent")
            else:
                log.error("error sending now playing: %s" % str(exception))

        if self._connected() and self._validate_cmd():
            ThreadedFunction(cb_finished, lastfm_manager.now_playing,
                             self._model.name, self._model.artist,
                             self._model.album, self._model.trackno,
                             self._length).start()

    def _update_offline_cache(self):
        # create offline cache if not exists
        if not self.prefs.has_key('submit_cache'):
            self.prefs['submit_cache'] = []

        # refresh offline cache from queue
        while self.pending_submits.qsize() > 0:
            if len(self.prefs['submit_cache']) >= self.submit_max_cache:
                self.prefs['submit_cache'] = \
                    self.prefs['submit_cache'][-self.submit_max_cache + 1:]
            args = self.pending_submits.get()
            self.prefs['submit_cache'].append(args)

        # save to file
        self.prefs.save()
        log.warning("updated cache: %s" % str(self.prefs['submit_cache']))

    def _cache_submit_send(self, submit_method, connected=False):
        # update cache from pending queue
        self._update_offline_cache()

        if not connected:
            log.warning("submit cache ignored (not connected)")
            return

        # send all cache
        error_count = 0
        while self.prefs['submit_cache']:
            # update cache with last pending
            if self.pending_submits.qsize() > 0:
                self._update_offline_cache()

            tracks = self.prefs['submit_cache'][:self.submit_max_posted]

            # validate
            if not self._validate_cmd(submit=True, tracks=tracks):
                break

            # XXX: do not overwhelm the server
            time.sleep(self.submit_delay)

            try:
                submit_method(tracks)
                log.warning("tracks submitted: %s" % str(tracks))
                self.prefs['submit_cache'] = \
                    self.prefs['submit_cache'][self.submit_max_posted:]
                error_count = 0
            except Exception, e:
                log.error("error submiting tracks %s: %s" % \
                              (str(tracks), str(e)))
                error_count += 1

            self._update_offline_cache()
            if error_count >= self.submit_max_retries:
                break

    def _submit(self):
        if self._length > 0:
            args = (self._model.name or "", self._model.artist,
                    self._model.album or "", self._model.trackno,
                    self._length, self.start_time)
            self.pending_submits.put(args)
        else:
            log.warning("submit ignored %s (invalid length)" % (self._model.name))

        if not self.sending_submits:
            def cb_finished(exception, retval):
                self.sending_submits = False
                log.warning("submit cache done")
                if exception is not None:
                    log.error("error sending submit cache: %s" % str(exception))

            self.sending_submits = True
            ThreadedFunction(cb_finished, self._cache_submit_send,
                             lastfm_manager.submit, self._connected()).start()

    def create_timer(self, time, func):
        self.remove_timer()
        self._start_time = int(ecore.time_get())
        self._total_time = time
        self._timer_func = func
        self._timer = ecore.timer_add(time, func)
        return self._timer

    def remove_timer(self):
        if self._timer is not None:
            self._timer.delete()
            self._timer = None
            self._timer_paused = False

    def playing(self):
        """Function that is called everytime that the Player's Controller
        changes it's state to 'PLAYING'.
        """
        log.warning("media playing")

        if self._timer is not None and self._timer_paused:
            self._timer_paused = False
            time = self._total_time - (self._stop_time - self._start_time)
            self.create_timer(time, self._timer_func)

        if self.start_time is None:
            self.start_time = int(mktime(localtime()))
            self._now_playing()

    def paused(self):
        """Function that is called everytime that the Player's Controller
        changes it's state to 'PAUSED'.
        """
        log.warning("media paused")

        if self._timer is not None:
            self._stop_time = int(ecore.time_get())
            self._timer.stop()
            self._timer_paused = True

    def duration_updated(self, duration):
        """Function that is called everytime that the Player's Controller
        updates it's length.
        """
        self._length = duration
        log.warning("duration updated: %d" % duration)

        if duration < 30:
            return
        elif duration > self.time_cons:
            time = self.time_cons
        else:
            time = duration / 2

        def cb_finished(*ignored):
            self.remove_timer()
            self._submit()

        self.create_timer(time, cb_finished)
