/*
** Bombermanclient SDL (c) 2000-2002 by Robert Hockauf
**
** ported 2003-08-15 to Sharp Zaurus (linux w/ qtopia + SDL) by Klaus Rotter kr@rotters.de
** Bombermanbgaiclient.c
** HISTORY:
** 2003-08-30 	Initial work to convert Roberts Java programm to C
*/

#include <stdbool.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <SDL/SDL_thread.h>
#include <SDL/SDL.h>

#define clock() (SDL_GetTicks())
#define SOCKET int

//static int min(int a, int b) { if (a < b) return a; return b; }

#define BOMBERMAN_PACKETLENGTH 48
#define BOMBERMAN_REQUEST 5
#define BOMBERMAN_PING 4
#define BOMBERMAN_UP 2
#define BOMBERMAN_DOWN 0
#define BOMBERMAN_LEFT 3
#define BOMBERMAN_RIGHT 1
#define BOMBERMAN_FIRE 16
#define BOMBERMAN_PLAYER 6
#define BOMBERMAN_ARENA 7
#define BOMBERMAN_IDLE 5
#define BOMBERMAN_TEXT 8
#define BOMBERMAN_STATUS 9
#define BOMBERMAN_SOUND 10
#define BOMBERMAN_MUSIC 11
#define BOMBERMAN_AI 12

#define DONT_WALK_HERE 	-100000
#define WALK_HERE 			100000
#define WALL_SCORE		1000
#define FLAME_SCORE 		-250
#define BONUS_FACTOR		100			// 0.1
#define SKULL_FACTOR		1000		// 1.0
#define ROUTING_STEPS		16
#define ROUTING_FACTOR	800
#define MIN_OFFENSIVE		20
#define RND_OFFENSIVE		5
#define NEARBY_SCORE		-4000
#define THREAD_SLEEP		25

#define NEUTRAL	0
#define DEFENSIVE	1
#define OFFENSIVE	2

#define ARENA_NULL		0
#define ARENA_BONUS	3
#define ARENA_SKULL	4
#define ARENA_DEBRIS	12
#define ARENA_FLAME	13
#define ARENA_WALL	14
#define ARENA_STEEL	15
#define ARENA_BOMB	16

#define DIRECTION_DOWN	0
#define DIRECTION_RIGHT	1
#define DIRECTION_UP		2
#define DIRECTION_LEFT		3
#define DIRECTION_IDLE		4
#define DIRECTION_DIE		5
#define DIRECTION_WIN		6

#define ACTION_KEYUP		1
#define ACTION_KEYDOWN	2
#define ACTION_KEYLEFT	3
#define ACTION_KEYRIGHT	4

#define MAXBOMBLIST 100

extern bool terminate;
extern char *server_host, *client_host;
extern int server_port;

typedef signed char byte;

struct Point {
	int x, y;
};

struct AIClient {
	SOCKET 			socket;
	struct sockaddr_in  	addr;
	int 					client_port;

	SDL_Thread 	*receiver_thread;
	SDL_Thread 	*sender_thread;
	SDL_Thread *ai_thread;

	bool keyUp;
	bool keyDown;
	bool keyLeft;
	bool keyRight;
	bool keyFire;

	byte theme_arena[15][13];
	byte player_id;
	byte arena[15][13][2];
	byte info_x, info_y;

	int 		playerX[5];
	int		playerY[5];
	byte		playerD[5];
	int		player_x, player_y;
	int 		startX[5];
	int 		startY[5];

	int 		tactics;
	int 		action;
	int 		offensive_counter;

	int 		score[15][13];

	int 		route_list[15*13*4];
	int 		route_first, route_last;
	bool 	not_routed[15][13];

	int 		flame_list[15*13*3*2];
	int 		flame_first, flame_last;

	struct Point 	bomb_list[49][MAXBOMBLIST];	// FIXME: Should make this dynamic, static about 20k
	int			bomb_pos[49];				//  actual position
};

static void Error(struct AIClient *ai, char *msg)
{
	printf("ID %d: %s\n", ai->player_id, msg);
	terminate = true;
	SDL_Delay(1000);
	exit(0);
}

static void Message(struct AIClient *ai, char *msg)
{
	printf("ID %d: %s\n", ai->player_id, msg);
}


static void addElement(struct AIClient *ai, int k, int x, int y)
{
	int p;

	p = ai->bomb_pos[k];

	if (p > MAXBOMBLIST)
		Error(ai, "Error: addElement() bomb_list[] full!");

	ai->bomb_list[k][p].x = x;
	ai->bomb_list[k][p].y = y;
	ai->bomb_pos[k]++;
}

static bool isEmpty(struct AIClient *ai, int k)
{
	if  (ai->bomb_pos[k] == 0)
		return true;
	else
		return false;
}

#if 0
static struct Point firstElement(struct AIClient *ai, int k)
{
	struct Point p;

	p.x = ai->bomb_list[k][0].x;
	p.y = ai->bomb_list[k][0].y;

	return p;
}
#endif

static void removeElementAt(struct AIClient *ai, int k, int p)
{
	int i, mp;

	mp = ai->bomb_pos[k];

	if (p < mp)
	{
		for (i = p; i<mp; i++)
		{
			ai->bomb_list[k][i].x = ai->bomb_list[k][i+1].x;
			ai->bomb_list[k][i].y = ai->bomb_list[k][i+1].y;
			i++;
		}
		ai->bomb_pos[k]--;
	}
	else
		Error(ai,"removeElemet: Already last Element!");
}

static void ai_arena(struct AIClient *ai, byte x, byte y, byte arena_type)
{
	if (arena_type <= 2) ai->arena[x][y][0] = ARENA_STEEL;					// steel 15
	else if(arena_type <= 3) ai->arena[x][y][0] = ARENA_WALL;				// wall 14
	else if(arena_type <= 8) ai->arena[x][y][0] = ARENA_DEBRIS;				// destroyed wall 12
	else if(arena_type <= 11)											// bomb 16-64
		if(ai->arena[x][y][0] < ARENA_BOMB) ai->arena[x][y][0] = ARENA_BOMB;
			else ai->arena[x][y][0]++;
		else if(arena_type == 12) ai->arena[x][y][0] = ARENA_NULL;				// nothing 0
		else if(arena_type <= 15) ai->arena[x][y][0] = (byte)(16 - arena_type);		// bonus 1-3
		else if(arena_type == 16) ai->arena[x][y][0] = ARENA_SKULL;				// skull 4
		else if(arena_type <= 52) ai->arena[x][y][0] = ARENA_FLAME;				// explosion 13
		else ai->arena[x][y][0] = ARENA_NULL;									// smoke 0

		ai->info_x = x;
		ai->info_y = y;
}

static void ai_player(struct AIClient *ai, byte i, int x, int y, byte direction)
{
		ai->playerX[i] = (x + 8) / 16;
		ai->playerY[i] = (y + 22) / 16;

		if (direction < 7) ai->playerD[i] = DIRECTION_DOWN;
		else if (direction < 14) ai->playerD[i] = DIRECTION_RIGHT;
		else if (direction < 21) ai->playerD[i] = DIRECTION_UP;
		else if (direction < 28) ai->playerD[i] = DIRECTION_LEFT;
		else if (direction < 31) ai->playerD[i] = DIRECTION_IDLE;
		else if (direction < 35) ai->playerD[i] = DIRECTION_DIE;
		else ai->playerD[i] = DIRECTION_WIN;

		if(i == ai->player_id && ai->playerY[i] <= 11)
		{
			ai->player_x = ai->playerX[i];
			ai->player_y = ai->playerY[i];
		}
}

static void ai_info(struct AIClient *ai, byte bomb_flamelength)
{
	ai->arena[ai->info_x][ai->info_y][1] = bomb_flamelength;
}

static bool alive(struct AIClient *ai)
{
	return (ai->playerD[ai->player_id] <= DIRECTION_IDLE);
}

static int score_static(struct AIClient *ai, int i, int j)
{
	int sc = 0;
	int k;

	switch(ai->arena[i][j][0])
	{
		case ARENA_NULL:
			switch(ai->tactics)
			{
				case NEUTRAL:
					/**
					 *  avoid dead ends if another player nearby
					 */
					/*if(player_nearby(player_x, player_y, 2) && dead_end(i, j))
						return 0;*/
					/**
					 *  the more walls the better (even skulls)
					 */
					if (ai->arena[i-1][j][0] == ARENA_WALL) sc = WALL_SCORE;
					if (ai->arena[i-1][j][0] == ARENA_SKULL) sc += WALL_SCORE / 4;
					if (ai->arena[i+1][j][0] == ARENA_WALL) sc += WALL_SCORE;
					if (ai->arena[i+1][j][0] == ARENA_SKULL) sc += WALL_SCORE / 4;
					if (ai->arena[i][j-1][0] == ARENA_WALL) sc += WALL_SCORE;
					if (ai->arena[i][j-1][0] == ARENA_SKULL) sc += WALL_SCORE / 4;
					if (ai->arena[i][j+1][0] == ARENA_WALL) sc += WALL_SCORE;
					if (ai->arena[i][j+1][0] == ARENA_SKULL) sc += WALL_SCORE / 4;
					break;
				case DEFENSIVE:
					/**
					 *  the more ways the better (even flames)
					 */
					if(ai->arena[i-1][j][0] <= ARENA_BONUS) sc = WALL_SCORE;
					if(ai->arena[i-1][j][0] == ARENA_FLAME) sc += WALL_SCORE / 4;
					if(ai->arena[i+1][j][0] <= ARENA_BONUS) sc += WALL_SCORE;
					if(ai->arena[i+1][j][0] == ARENA_FLAME) sc += WALL_SCORE / 4;
					if(ai->arena[i][j-1][0] <= ARENA_BONUS) sc += WALL_SCORE;
					if(ai->arena[i][j-1][0] == ARENA_FLAME) sc += WALL_SCORE / 4;
					if(ai->arena[i][j+1][0] <= ARENA_BONUS) sc += WALL_SCORE;
					if(ai->arena[i][j+1][0] == ARENA_FLAME) sc += WALL_SCORE / 4;
					break;
				case OFFENSIVE:
					/**
					 *  the more other player the better
					 */
					for(k = 0; k < 5; k++)
						if(k != ai->player_id && i == ai->playerX[k] && j == ai->playerY[k])
							sc += WALL_SCORE;
					break;
				default:
					;
			}
			break;
		case 1:
		case 2:
		case ARENA_BONUS:
			sc = (WALK_HERE * BONUS_FACTOR) / 1000;
			break;
		case ARENA_SKULL:
			sc = (DONT_WALK_HERE * SKULL_FACTOR) / 1000;
			break;
		default:
			sc = DONT_WALK_HERE;
	}
	return sc;
}

static void clear_route(struct AIClient *ai)
{
	int i, j;

	for(i = 0; i < 15; i++)
		for(j = 0; j < 13; j++)
			ai->not_routed[i][j] = true;
}

static void do_route(struct AIClient *ai, int i, int j, int steps, int sc)
{
	if ((--steps) == 0) return;
	sc = (sc * ROUTING_FACTOR) / 1000;

	if (ai->not_routed[i-1][j] && ai->arena[i-1][j][0] <= ARENA_BONUS)
	{
		ai->not_routed[i-1][j] = false;
		if(sc > ai->score[i-1][j]) ai->score[i-1][j] = sc;
		ai->route_list[ai->route_last++] = i-1;
		ai->route_list[ai->route_last++] = j;
		ai->route_list[ai->route_last++] = steps;
		ai->route_list[ai->route_last++] = sc;
	}
	if (ai->not_routed[i+1][j] && ai->arena[i+1][j][0] <= ARENA_BONUS)
	{
		ai->not_routed[i+1][j] = false;
		if(sc > ai->score[i+1][j]) ai->score[i+1][j] = sc;
		ai->route_list[ai->route_last++] = i+1;
		ai->route_list[ai->route_last++] = j;
		ai->route_list[ai->route_last++] = steps;
		ai->route_list[ai->route_last++] = sc;
	}
	if (ai->not_routed[i][j-1] && ai->arena[i][j-1][0] <= ARENA_BONUS)
	{
		ai->not_routed[i][j-1] = false;
		if(sc > ai->score[i][j-1]) ai->score[i][j-1] = sc;
		ai->route_list[ai->route_last++] = i;
		ai->route_list[ai->route_last++] = j-1;
		ai->route_list[ai->route_last++] = steps;
		ai->route_list[ai->route_last++] = sc;
	}
	if (ai->not_routed[i][j+1] && ai->arena[i][j+1][0] <= ARENA_BONUS)
	{
		ai->not_routed[i][j+1] = false;
		if(sc > ai->score[i][j+1]) ai->score[i][j+1] = sc;
		ai->route_list[ai->route_last++] = i;
		ai->route_list[ai->route_last++] = j+1;
		ai->route_list[ai->route_last++] = steps;
		ai->route_list[ai->route_last++] = sc;
	}
}

static void score_route(struct AIClient *ai)
{
	int i, j, k;

	/**
	 *  reset score and fill bomb lists
	 */
	for (i = 0; i < 15; i++)
		for (j = 0; j < 13; j++)
		{
			ai->score[i][j] = DONT_WALK_HERE;
			k = ai->arena[i][j][0]-ARENA_BOMB;
			if (k > 48) k = 48;
			if (ai->arena[i][j][0] >= ARENA_BOMB)
				addElement(ai, k, i, j);
		}

	for(i = 1; i <= 13; i++)
		for(j = 1; j <= 11; j++)
		{
			int sc = score_static(ai, i, j);
			if(sc < 0 || sc > ai->score[i][j]) ai->score[i][j] = sc;
			if(ai->arena[i][j][0] <= ARENA_BONUS)
			{
				clear_route(ai);
				ai->not_routed[i][j] = false;
				ai->route_first = 0;
				ai->route_last = 0;
				ai->route_list[ai->route_last++] = i;
				ai->route_list[ai->route_last++] = j;
				ai->route_list[ai->route_last++] = ROUTING_STEPS;
				ai->route_list[ai->route_last++] = sc;
				while(ai->route_first < ai->route_last)
					do_route(ai, ai->route_list[ai->route_first++], ai->route_list[ai->route_first++], ai->route_list[ai->route_first++], ai->route_list[ai->route_first++]);
			}
		}
}

static void do_bombs(struct AIClient *ai)
{
	int i, j;
	//struct Point p;

	clear_route(ai);

	for(i = 48; i >= 0; i--)
		while(!isEmpty(ai, i))
		{
			//Point p = (Point)bomb_list[i].firstElement();

			struct Point p;
			//p = firstElement(ai, i);

			p.x = ai->bomb_list[i][0].x;
			p.y = ai->bomb_list[i][0].y;

			if(ai->not_routed[p.x][p.y])
			{
				ai->not_routed[p.x][p.y] = false;
				ai->flame_list[ai->flame_last++] = p.x;
				ai->flame_list[ai->flame_last++] = p.y;
				ai->flame_list[ai->flame_last++] = -1;

				for (j = 1; j <= ai->arena[p.x][p.y][1]; j++)
					if (ai->arena[p.x-j][p.y][0] >= ARENA_BOMB)
					{
						if (ai->not_routed[p.x-j][p.y])
						{
							if((ai->arena[p.x][p.y][0] - ai->arena[p.x-j][p.y][0]) > 1)
								addElement(ai, i-1, p.x-j, p.y); // bomb_list[i-1].addElement(new Point(p.x-j, p.y));
							break;
						}
						ai->flame_list[ai->flame_last++] = p.x-j;
						ai->flame_list[ai->flame_last++] = p.y;
						ai->flame_list[ai->flame_last++] = i;
					}
					else if(ai->arena[p.x-j][p.y][0] <= ARENA_BONUS || ai->arena[p.x-j][p.y][0] == ARENA_FLAME)
					{
						ai->flame_list[ai->flame_last++] = p.x-j;
						ai->flame_list[ai->flame_last++] = p.y;
						ai->flame_list[ai->flame_last++] = i;
					}
					else break;

					for(j = 1; j <= ai->arena[p.x][p.y][1]; j++)
						if(ai->arena[p.x+j][p.y][0] >= ARENA_BOMB)
						{
							if(ai->not_routed[p.x+j][p.y])
							{
								if((ai->arena[p.x][p.y][0] - ai->arena[p.x+j][p.y][0]) > 1)
								{
									addElement(ai, i-1, p.x+j, p.y); // bomb_list[i-1].addElement(new Point(p.x+j, p.y));
								}
								break;
							}
							ai->flame_list[ai->flame_last++] = p.x+j;
							ai->flame_list[ai->flame_last++] = p.y;
							ai->flame_list[ai->flame_last++] = i;
						}
						else if(ai->arena[p.x+j][p.y][0] <= ARENA_BONUS || ai->arena[p.x+j][p.y][0] == ARENA_FLAME)
						{
							ai->flame_list[ai->flame_last++] = p.x+j;
							ai->flame_list[ai->flame_last++] = p.y;
							ai->flame_list[ai->flame_last++] = i;
						} else break;

					for(j = 1; j <= ai->arena[p.x][p.y][1]; j++)
						if(ai->arena[p.x][p.y-j][0] >= ARENA_BOMB)
						{
							if(ai->not_routed[p.x][p.y-j])
							{
								if((ai->arena[p.x][p.y][0] - ai->arena[p.x][p.y-j][0]) > 1)
									addElement(ai, i-1, p.x+j, p.y); // bomb_list[i-1].addElement(new Point(p.x, p.y-j));
								break;
							}
							ai->flame_list[ai->flame_last++] = p.x;
							ai->flame_list[ai->flame_last++] = p.y-j;
							ai->flame_list[ai->flame_last++] = i;
						}
						else if (ai->arena[p.x][p.y-j][0] <= ARENA_BONUS || ai->arena[p.x][p.y-j][0] == ARENA_FLAME) {
							ai->flame_list[ai->flame_last++] = p.x;
							ai->flame_list[ai->flame_last++] = p.y-j;
							ai->flame_list[ai->flame_last++] = i;
						}
						else break;

					for(j = 1; j <= ai->arena[p.x][p.y][1]; j++)
						if(ai->arena[p.x][p.y+j][0] >= ARENA_BOMB)
						{
							if(ai->not_routed[p.x][p.y+j])
							{
								if((ai->arena[p.x][p.y][0] - ai->arena[p.x][p.y+j][0]) > 1)
									addElement(ai, i-1, p.x+j, p.y); // bomb_list[i-1].addElement(new Point(p.x, p.y+j));
								break;
							}
							ai->flame_list[ai->flame_last++] = p.x;
							ai->flame_list[ai->flame_last++] = p.y+j;
							ai->flame_list[ai->flame_last++] = i;
						}
						else	if(ai->arena[p.x][p.y+j][0] <= ARENA_BONUS || ai->arena[p.x][p.y+j][0] == ARENA_FLAME)
						{
							ai->flame_list[ai->flame_last++] = p.x;
							ai->flame_list[ai->flame_last++] = p.y+j;
							ai->flame_list[ai->flame_last++] = i;
						}
						else break;
				}
				removeElementAt(ai, i,0); // bomb_list[i].removeElementAt(0);
			}
	}

static void score_flames(struct AIClient *ai)
{
	bool run_away;

		/**
		 *  get list of (future) flames ordered by time
		 */
		ai->flame_first = 0;
		ai->flame_last = 0;
		do_bombs(ai);
		clear_route(ai);

		run_away = false;

		while(ai->flame_first < ai->flame_last)
		{
			int x, y, sc;

			x = ai->flame_list[ai->flame_first++];
			y = ai->flame_list[ai->flame_first++];
			sc = ai->flame_list[ai->flame_first++];

			/**
			 *  become defensive when going to be flamed
			 */
			if (x == ai->player_x && y == ai->player_y)
				run_away = true;

			if (ai->not_routed[x][y])
				ai->score[x][y] += (sc + 1) * FLAME_SCORE;

			ai->not_routed[x][y] = false;
		}
		if (run_away) ai->tactics = DEFENSIVE; else ai->tactics = NEUTRAL;
	}


static bool drop_bomb(struct AIClient *ai)
{
	int w, e;

		switch(ai->tactics)
		{
			case NEUTRAL:
				if(ai->action >= 0) break;

				/**
				 *  only drop bomb if at least one wall (or skull) will be destroyed and there is a way to escape
				 */
				w = 0, e = 0;
 				if (ai->arena[ai->player_x-1][ai->player_y][0] == ARENA_WALL || ai->arena[ai->player_x-1][ai->player_y][0] == ARENA_SKULL) w++;
				else if(ai->score[ai->player_x-1][ai->player_y] >= 0) e++;
				if(ai->arena[ai->player_x+1][ai->player_y][0] == ARENA_WALL || ai->arena[ai->player_x+1][ai->player_y][0] == ARENA_SKULL) w++;
				else if(ai->score[ai->player_x+1][ai->player_y] >= 0) e++;
				if(ai->arena[ai->player_x][ai->player_y-1][0] == ARENA_WALL || ai->arena[ai->player_x][ai->player_y-1][0] == ARENA_SKULL) w++;
				else if(ai->score[ai->player_x][ai->player_y-1] >= 0) e++;
				if(ai->arena[ai->player_x][ai->player_y+1][0] == ARENA_WALL || ai->arena[ai->player_x][ai->player_y+1][0] == ARENA_SKULL) w++;
				else if(ai->score[ai->player_x][ai->player_y+1] >= 0) e++;

				if(w > 0 && e > 0) return true;
				break;
			case OFFENSIVE:
				if((rand() % RND_OFFENSIVE) == 0) return true;
				break;
			case DEFENSIVE:
				/**
				 *  don't drop a bomb if in defensive mode
				 */
			default:
				;
		}
		return false;
	}

int sender_thread(struct AIClient *ai)
{
	int i;
	byte data[BOMBERMAN_PACKETLENGTH];

	for (i=0; i<BOMBERMAN_PACKETLENGTH; i++) data[i]=0;

	data[1] = ai->player_id;

	while(!terminate)
	{
		data[0] = BOMBERMAN_IDLE;
		if (ai->keyUp) data[0] = BOMBERMAN_UP;
		else if (ai->keyDown) data[0] = BOMBERMAN_DOWN;
		else if (ai->keyLeft) data[0] = BOMBERMAN_LEFT;
		else if (ai->keyRight) data[0] = BOMBERMAN_RIGHT;
		if (ai->keyFire) data[0] |= BOMBERMAN_FIRE;
		if (sendto(ai->socket, data, 2, 0, (struct sockaddr *)&(ai->addr), sizeof(struct sockaddr)) == -1)
			Message(ai,"send to failed!");
		//printf("ID %d: Sending byte %d\n", ai->player_id, (int) data[0]);
		SDL_Delay(100);
	}

	data[0] = BOMBERMAN_PING;
	sendto(ai->socket, data, 2, 0, (struct sockaddr *)&(ai->addr), sizeof(struct sockaddr));
	return 0;
}

int ai_thread(struct AIClient *ai)
{
	//struct sockaddr_in _addr;
	//int data_len;
	byte data[BOMBERMAN_PACKETLENGTH];

	int i, j;

	//this.controller = controller;
	//player_id = (byte *) *id;
	//printf("Player id=%d\n", (int) player_id);

	// Message(ai, "AI thread started...");

	for (i = 0; i < 15; i++)
		for (j = 0; j < 13; j++)
			ai->arena[i][j][0] = ARENA_STEEL;

	for (i = 0; i < 49; i++)
		ai->bomb_pos[i] = 0;	// Init position, empty list

	ai->player_x = ai->startX[(int) ai->player_id];
	ai->player_y = ai->startY[(int) ai->player_id];

	SDL_Delay(500);

	//new Thread(this).start();
	//System.out.println("BombermanRobertAI version 26.jul.2002 by robert hockauf.");
	// Mainloop of AI Thread
	while(!terminate)
	{
		SDL_Delay(50);

		ai->keyUp = false;
		ai->keyDown = false;
		ai->keyLeft= false;
		ai->keyRight = false;
		ai->keyFire = false;

		if (alive(ai))
		{
			int max;

			// Message(ai, "AI send mainloop entered.");

			score_route(ai);
			score_flames(ai);
			//log();
			ai->action = -1;
			max = ai->score[ai->player_x][ai->player_y];
			//data[0] = BOMBERMAN_IDLE;

			if (ai->score[ai->player_x][ai->player_y-1] > max)
			{
				max = ai->score[ai->player_x][ai->player_y-1];
				ai->action = ACTION_KEYUP; // action = BombermanController.KEY_UP;
				//data[0] = BOMBERMAN_UP;
			}

			if (ai->score[ai->player_x][ai->player_y+1] > max)
			{
				max = ai->score[ai->player_x][ai->player_y+1];
				ai->action = ACTION_KEYDOWN; //action = BombermanController.KEY_DOWN;
				//data[0] = BOMBERMAN_DOWN;
			}

			if (ai->score[ai->player_x-1][ai->player_y] > max)
			{
				max = ai->score[ai->player_x-1][ai->player_y];
				ai->action = ACTION_KEYLEFT; //action = BombermanController.KEY_LEFT;
				//data[0] = BOMBERMAN_LEFT;
			}

			if (ai->score[ai->player_x+1][ai->player_y] > max)
			{
				max = ai->score[ai->player_x+1][ai->player_y];
				ai->action = ACTION_KEYRIGHT; //action = BombermanController.KEY_RIGHT;
				//data[0] = BOMBERMAN_RIGHT;
			}

			/**
			*  become offensive if nothing else to do
			*/
			if (ai->action < 0 /*data[0] == 5*/ && max == 0 && ai->tactics != OFFENSIVE) ai->offensive_counter++;
			if (ai->offensive_counter > MIN_OFFENSIVE)
			{
				ai->offensive_counter = MIN_OFFENSIVE;
				ai->tactics = OFFENSIVE;
			}
			// controller.key(action, true);

			switch (ai->action)
			{
				case ACTION_KEYUP:
					ai->keyUp = true;
					break;
				case ACTION_KEYDOWN:
					ai->keyDown = true;
					break;
				case ACTION_KEYLEFT:
					ai->keyLeft = true;
					break;
				case ACTION_KEYRIGHT:
					ai->keyRight = true;
					break;
			}

			ai->keyFire = drop_bomb(ai);  // controller.key(BombermanController.KEY_FIRE, drop_bomb());
/*
			if (drop_bomb(ai))  // controller.key(BombermanController.KEY_FIRE, drop_bomb());
				data[0] |= BOMBERMAN_FIRE;
*/
/*
			if (keyUp) data[0] = BOMBERMAN_UP;
			else if (keyDown) data[0] = BOMBERMAN_DOWN;
			else if (keyLeft) data[0] = BOMBERMAN_LEFT;
			else if (keyRight) data[0] = BOMBERMAN_RIGHT;
			if (keyFire) data[0] |= BOMBERMAN_FIRE;
*/
/*
			printf("ID %d: Sending %02x\n",ai->player_id, data[0]);

			sendto(ai->socket, data, 2, 0, (struct sockaddr *)&(ai->addr), sizeof(struct sockaddr));
*/
		}
		else
		{
			/**
			*  move player to start position if died or won
			*/
			ai->player_x = ai->startX[ai->player_id];
			ai->player_y = ai->startY[ai->player_id];
			ai->tactics = NEUTRAL;
		}
	}
	data[0] = BOMBERMAN_PING;
	sendto(ai->socket, data, 2, 0, (struct sockaddr *) &(ai->addr), sizeof(struct sockaddr));

	if (ai->receiver_thread) SDL_WaitThread(ai->receiver_thread, NULL);
	shutdown(ai->socket, SHUT_RDWR);
	SDL_Delay(100);
	free((void *) ai);	// Dangerous !

	return 0;
}

int receiver_thread(struct AIClient *ai)
{
	int i; //, j, k;
	int playerX[5], playerY[5];
	byte playerImage[5];
	//byte zbuf[5];
	//long time = 0, t;

	struct sockaddr_in addr;

	int data_len;
	byte data[BOMBERMAN_PACKETLENGTH];

	//SDL_Rect srect;

	Message(ai, "Receiver thread started...");

	while(!terminate)
	{
		//SDL_Delay(50);

		data_len = sizeof(struct sockaddr);
		if (((data_len = recvfrom(ai->socket, data, sizeof(data), 0, (struct sockaddr *) &addr, &data_len)) == -1))
			Error(ai, "Error receiving socket.\n");
		i = 0;

		while(data[i] != BOMBERMAN_IDLE)
		{
			//printf("ID %d: Data %d\n",ai->player_id, data[i]);

			switch(data[i])
			{
				//int t;
				case BOMBERMAN_PLAYER:
					i++;
					playerX[data[i]] = (((int)data[i+1])+256)%256;
					playerY[data[i]] = ((((int)data[i+2])+256)%256)+20;
					playerImage[data[i]] = data[i+3];
                                        ai_player(ai, data[i], playerX[data[i]], playerY[data[i]]-20, playerImage[data[i]]);
					i += 4;
					break;

				case BOMBERMAN_ARENA:
					ai_arena(ai, data[i+1], data[i+2], data[i+3]); // Manage AI
					i += 4;
					break;

				case BOMBERMAN_AI:
					i++;
					ai_info(ai, data[i]);
					i++;
					break;

				case BOMBERMAN_TEXT:
					if(data[++i] < 5);
					i += data[i+1]+2;
					break;

				case BOMBERMAN_STATUS:
					if(data[++i] == 5)
					{
						i += 3;	// Dont care in Background AI mode
/*
						char status_time[5] = "X:XX";
						byte m = data[++i];
						byte s = data[++i];
						status_time[0] = '0'+m;
						status_time[2] = '0'+s/10;
						status_time[3] = '0'+s%10;
						i++;
*/
					}
					else
					{
						i += 2;
/*
						char status_frags[4] = "-XX";
						byte p = data[i++];
						byte f = data[i++];
						if(f >= 0) status_frags[0] = ' '; else status_frags[0] = '-';
						f = abs(f);
*/
					}
					break;

				case BOMBERMAN_SOUND:
					i += 2;
					break;

				case BOMBERMAN_MUSIC:
					i++;
					break;

				default:
					Error(ai, "Error handling data.");
					exit(-101);
					break;
			}
		}
	}
	return 0;
}

int ai_bg_init(int port)
{
	int 				i;
	//struct hostent 	*h;
	int 				data_len;
	byte 			data[BOMBERMAN_PACKETLENGTH];
	int startx[] = {1, 13, 13, 1, 7};
	int starty[] = {1, 11, 1, 11, 6};

	struct AIClient 	*ai;

	if (!(ai = (struct AIClient *) malloc((size_t) sizeof(struct AIClient))))
		Error(ai,"Malloc failed!");

	// Init ai
	ai->tactics = NEUTRAL;
	ai->action = -1;
	ai->offensive_counter = 0;
	ai->route_first = 0;
	ai->route_last = 0;
	ai->flame_first = 0;
	ai->flame_last = 0;
	ai->player_id = -1;
	ai->client_port = port;
	ai->keyUp = false;
	ai->keyDown = false;
	ai->keyLeft = false;
	ai->keyRight = false;
	ai->keyFire = false;

	for (i=0; i<5; i++) {
		ai->playerY[i] =17;
		ai->startX[i] = startx[i];
		ai->startY[i] = starty[i];
	}

	if ((ai->socket = socket(AF_INET, SOCK_DGRAM, 0)) == -1)
		Error(ai, "ID %d: Could not open socket.");

/*
	gethostname(data, sizeof(data)-1);
	h = gethostbyname(data);
	client_host = inet_ntoa(*((struct in_addr *) h->h_addr));
	if (!server_host) server_host = client_host;
*/
	ai->addr.sin_family = AF_INET;
	ai->addr.sin_port = htons(ai->client_port);
	ai->addr.sin_addr.s_addr = inet_addr(client_host);
	memset(&(ai->addr.sin_zero), '\0', 8);
	if (bind(ai->socket, (struct sockaddr *) &(ai->addr), sizeof(struct sockaddr)) == -1)
		Error(ai, "Could not bind Socket");

	//SDL_Delay(100);

	data[0] = BOMBERMAN_PING;
	data[1] = BOMBERMAN_REQUEST;
	data[2] = 0;

	ai->addr.sin_family = AF_INET;
	ai->addr.sin_port = htons(server_port);
	ai->addr.sin_addr.s_addr = inet_addr(server_host);
	memset(&(ai->addr.sin_zero), '\0', 8);
	sendto(ai->socket, data, data[2]+3, 0, (struct sockaddr *) &(ai->addr), sizeof(struct sockaddr));

	printf("ID %d: Connecting to bombermanserver %s\n", ai->player_id, server_host);

	data_len = sizeof(struct sockaddr);

	if (((data_len = recvfrom(ai->socket, data, sizeof(data), 0, (struct sockaddr *) &(ai->addr), &data_len)) == -1) || data[1] > 4) {
		printf("ID %d: Server not found or too many players", ai->player_id);
	}
	else {
		ai->player_id = data[1];

		// Start threads
		if (!(ai->receiver_thread = SDL_CreateThread((void *) receiver_thread, (void *) ai)))
			Error(ai, "Could not start receiver thread");

		if (!(ai->sender_thread = SDL_CreateThread((void *) sender_thread, (void *) ai)))
			Error(ai, "Could not start sender thread");

		if (!(ai->ai_thread = SDL_CreateThread((void *) ai_thread, (void *) ai)))
			Error(ai, "Could not start ai thread");
	}
	return 0;
}
