/*
 * Copyright © 2006-2007 Daniel Stone
 * Copyright © 2009 Oliver McFadden
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 * Author: Daniel Stone <daniel@fooishbar.org>
 * Author: Oliver McFadden <oliver.mcfadden@nokia.com>
 */

#ifdef HAVE_DIX_CONFIG_H
#include <dix-config.h>
#endif

#include <dbus/dbus.h>
#include <string.h>

#include <X11/X.h>

#include "config-backends.h"

#include "xorg-config.h"
#include "globals.h"

#ifdef DPMSExtension
#include "dpmsproc.h"
#endif

#ifdef DPMSExtension
#define DPMS_SERVER
#include <X11/extensions/dpms.h>
#endif

#include <mce/dbus-names.h>
#include <mce/mode-names.h>

#define MATCH_RULE \
  "type='signal',interface='" MCE_SIGNAL_IF "',"\
  "member='" MCE_DISPLAY_SIG "'"

struct connection_info
{
  DBusConnection *connection;
};

static DBusHandlerResult
message_handler (DBusConnection * connection, DBusMessage * message,
		 void *data)
{
  DBusMessageIter iter;
  char *s = NULL;

  if (!dbus_message_iter_init (message, &iter))
    {
      return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

  dbus_message_iter_get_basic (&iter, &s);
  if (!s)
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

  DebugF ("[config/osso] received an \"%s\" message\n", s);

  if (strncmp (s, MCE_DISPLAY_ON_STRING,
	       strlen (MCE_DISPLAY_ON_STRING) + 1) == 0)
    {
#ifdef DPMSExtension
      if (DPMSEnabled)
	DPMSSet (serverClient, DPMSModeOff);
#endif
      ResetOsBuffers ();
      dixSaveScreens (serverClient, SCREEN_SAVER_OFF, ScreenSaverReset);
    }
  else
    if (strncmp (s, MCE_DISPLAY_OFF_STRING,
		 strlen (MCE_DISPLAY_OFF_STRING) + 1) == 0)
    {
#ifdef DPMSExtension
      if (DPMSEnabled)
	DPMSSet (serverClient, DPMSModeOn);
#endif
      ResetOsBuffers ();
      dixSaveScreens (serverClient, SCREEN_SAVER_ON, ScreenSaverActive);
    }

  return DBUS_HANDLER_RESULT_HANDLED;
}

static void
connect_hook (DBusConnection * connection, void *data)
{
  DBusError error;
  DBusObjectPathVTable vtable = {.message_function = message_handler, };
  struct connection_info *info = data;

  info->connection = connection;

  dbus_error_init (&error);

  dbus_bus_request_name (info->connection, MCE_SIGNAL_IF, 0, &error);
  if (dbus_error_is_set (&error))
    {
      ErrorF ("[config/osso] couldn't take over org.x.config: %s (%s)\n",
	      error.name, error.message);
      goto err_start;
    }

  /* blocks until we get a reply. */
  dbus_bus_add_match (info->connection, MATCH_RULE, &error);
  if (dbus_error_is_set (&error))
    {
      ErrorF ("[config/osso] couldn't add match: %s (%s)\n", error.name,
	      error.message);
      goto err_name;
    }

  if (!dbus_connection_register_object_path
      (info->connection, MCE_SIGNAL_PATH, &vtable, info))
    {
      ErrorF ("[config/osso] couldn't register object path\n");
      goto err_match;
    }

  dbus_error_free (&error);

  return;

err_match:
  dbus_bus_remove_match (info->connection, MATCH_RULE, &error);
err_name:
  dbus_bus_release_name (info->connection, MCE_SIGNAL_IF, &error);
err_start:
  dbus_error_free (&error);
}

static void
disconnect_hook (void *data)
{
}

static struct connection_info connection_data;
static struct config_dbus_core_hook core_hook = {
  .connect = connect_hook,
  .disconnect = disconnect_hook,
  .data = &connection_data,
};

int
config_osso_init (void)
{
  return config_dbus_core_add_hook (&core_hook);
}

void
config_osso_fini (void)
{
  config_dbus_core_remove_hook (&core_hook);
}
