/*-------------------------------------------------------------------------
 * Filename:      serial.c
 * Version:       $Id: serial.c,v 1.18 2000/07/10 14:20:17 erikm Exp $
 * Copyright:     Copyright (C) 1999, Erik Mouw
 * Author:        Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Description:   Serial utilities for blob
 * Created at:    Tue Aug 24 20:25:00 1999
 * Modified by:   Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Modified at:   Mon Oct  4 20:11:14 1999
 *-----------------------------------------------------------------------*/
/*
 * serial.c: Serial utilities for blob
 *
 * Copyright (C) 1999  Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ident "$Id: serial.c,v 1.18 2000/07/10 14:20:17 erikm Exp $"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

/* Tell SA-1100.h to shut up; we're including it anyway. Nyah nyah ;-) */
#include "serial.h"
#include "time.h"




/* number of nibbles in a word */
#define NIBBLES_PER_WORD (8)




/*
 * Initialise the serial port with the given baudrate. The settings
 * are always 8 data bits, no parity, 1 stop bit, no start bits.
 *
 */
void SerialInit(eBauds baudrate)
{
	/* Theory of operations:
	 * - Flush the output buffer
	 * - switch receiver and transmitter off
	 * - clear all sticky bits in control register 3
	 * - set the port to sensible defaults (no break, no interrupts,
	 *   no parity, 8 databits, 1 stopbit, transmitter and receiver 
	 *   enabled
	 * - set the baudrate to the requested value
	 * - turn the receiver and transmitter back on
	 */

}


#define TDFE 0x20

/*
 * Output a single byte to the serial port.
 */
void SerialOutputByte(const char c)
{
    char *SCFTDR2;
    volatile unsigned short *SCFSR2;
    
    SCFTDR2 = 0xffe8000c;
    SCFSR2 = 0xffe80010;
    if(c==13)SerialOutputByte(10);

    while ( ((*SCFSR2) & TDFE) == 0 );
    
    *SCFTDR2 = c;
    *SCFSR2 = ~TDFE;
}




/*
 * Write a null terminated string to the serial port.
 */
void SerialOutputString(const char *s) {

	while(*s != 0)
		SerialOutputByte(*s++);
		
} /* SerialOutputString */




/*
 * Write the argument of the function in hexadecimal to the serial
 * port. If you want "0x" in front of it, you'll have to add it
 * yourself.
 */
void SerialOutputHex(const u32 h)
{
	char c;
	int i;
	
	for(i = NIBBLES_PER_WORD - 1; i >= 0; i--) {
		c = (char)((h >> (i * 4)) & 0x0f);

		if(c > 9)
			c += ('A' - 10);
		else
			c += '0';

		SerialOutputByte(c);
	}
}




/*
 * Write the argument of the function in decimal to the serial port.
 * We just assume that each argument is positive (i.e. unsigned).
 */
void SerialOutputDec(const u32 d)
{
	int leading_zero = 1;
	u32 divisor, result, remainder;
        char b[11];

        if(d==0) {
            SerialOutputByte('0');
        } else {
            int i;
            i=0;
            while(d!=0) {
                b[i] = d % 10;
                d=d/10;
                i++;
                if(i==10)d=0;
            }
            while(i>0) {
                i--;
                SerialOutputByte('0'+b[i]);
            }
        }

/*
        remainder = d;

	for(divisor = 1000000000; 
	    divisor > 0; 
	    divisor /= 10) {
		result = remainder / divisor;
		remainder %= divisor;

		if(result != 0 || divisor == 1)
			leading_zero = 0;

		if(leading_zero == 0)
			SerialOutputByte((char)(result) + '0');
	}
*/
}




/*
 * Write a block of data to the serial port. Similar to
 * SerialOutputString(), but this function just writes the number of
 * characters indicated by bufsize and doesn't look at termination
 * characters.
 */
void SerialOutputBlock(const char *buf, int bufsize)
{
	while(bufsize--)
		SerialOutputByte(*buf++);
}


#define RDF 3

/*
 * Read a single byte from the serial port. Returns 1 on success, 0
 * otherwise. When the function is succesfull, the character read is
 * written into its argument c.
 */
int SerialInputByte(char *c)
{
    char *SCFRDR2;
    volatile unsigned short *SCFSR2;
    
    SCFRDR2 = 0xffe80014;
    SCFSR2 = 0xffe80010;

    if ( (*SCFSR2) & RDF ) {
        *c = *SCFRDR2;
        *SCFSR2=~RDF;
        return 1;
    } else {
        return 0;
    }

} /* SerialInputByte */




/*
 * read a string with maximum length len from the serial port
 * using a timeout of timeout seconds
 *
 * len is the length of array s _including_ the trailing zero,
 * the function returns the number of bytes read _excluding_
 * the trailing zero
 */
int  SerialInputString(char *s, const int len, const int timeout)
{
	u32 startTime, currentTime;
	char c;
	int i;
	int numRead;
	int skipNewline = 1;
	int maxRead = len - 1;
	
	startTime = TimerGetTime();

	for(numRead = 0, i = 0; numRead < maxRead;) {
		/* try to get a byte from the serial port */
		while(!SerialInputByte(&c)) {
			currentTime = TimerGetTime();

			/* check timeout value */
			if((currentTime - startTime) > 
			   (timeout * TICKS_PER_SECOND)) {
				/* timeout */
				s[i++] = '\0';
				return(numRead);
			}
		}

		/* eat newline characters at start of string */
		if((skipNewline == 1) && (c != '\r') && (c != '\n'))
			skipNewline = 0;

		if(skipNewline == 0) {
			if((c == '\r') || (c == '\n')) {
				s[i++] = '\0';
				return(numRead);
			} else {
				s[i++] = c;
				numRead++;
			}
		}
	}

	return(numRead);
}




/* 
 * SerialInputBlock(): almost the same as SerialInputString(), but
 * this one just reads a block of characters without looking at
 * special characters.
 */
int  SerialInputBlock(char *buf, int bufsize, const int timeout)
{
	u32 startTime, currentTime;
	char c;
	int i, j;
	int numRead;
	int maxRead = bufsize;
	
	startTime = TimerGetTime();

	j=10000;
	
	for(numRead = 0, i = 0; numRead < maxRead;) {
		/* try to get a byte from the serial port */
		while(!SerialInputByte(&c)) {
			j--;
			/* check timeout value */
			if(!j)
				return(numRead);
		}
		
		buf[i++] = c;
		numRead ++;
	}

	return(numRead);
}
