/*-------------------------------------------------------------------------
 * Filename:      main.c
 * Version:       $Id: main.c,v 1.26 2000/07/10 14:20:17 erikm Exp $
 * Copyright:     Copyright (C) 1999, Jan-Derk Bakker
 * Author:        Jan-Derk Bakker <J.D.Bakker@its.tudelft.nl>
 * Description:   Main file for the trivially simple bootloader for the 
 *                LART boards
 * Created at:    Mon Aug 23 20:00:00 1999
 * Modified by:   Erik Mouw <J.A.K.Mouw@its.tudelft.nl>
 * Modified at:   Sat Mar 25 14:31:16 2000
 *-----------------------------------------------------------------------*/
/*
 * main.c: main file for the blob bootloader
 *
 * Copyright (C) 1999  Jan-Derk Bakker (J.D.Bakker@its.tudelft.nl) and
 *                     Erik Mouw (J.A.K.Mouw@its.tudelft.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ident "$Id: main.c,v 1.26 2000/07/10 14:20:17 erikm Exp $"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "clock.h"
#include "command.h"
#include "flash.h"
#include "serial.h"
#include "time.h"
#include "main.h"
#include "util.h"
#include "uucodec.h"
#include "st.h"


typedef enum {
	fromFlash = 0,
	fromDownload = 1
} blockSource;




typedef struct {
	int kernelSize;
	blockSource kernelType;

	int ramdiskSize;
	blockSource ramdiskType;

	u32 blockSize;

	eBauds downloadSpeed;
} blobStatus;


void memtest1(void);
void memtest2(void);
void memtest3(void);
void st(void);

void BootKernel(char *commandline);
void Download(char *commandline, blobStatus *status);
void PrintHelp(void);
void SetDownloadSpeed(char *commandline, blobStatus *status);
void PrintStatus(blobStatus *status);
void ResetTerminal(void);
void Reload(char *commandline, blobStatus *status);
void PrintSerialSpeed(eBauds speed);

int init_pcic();

void c_main(char *blockBase, u32 blockSize)
{
	int numRead = 0;
	char commandline[128];
	blobStatus status;
	int i;
	int retval = 0;

        blockSize = 8*1024*1024;
	
	/* We really want to be able to communicate, so initialise the
	 * serial port at 9k6 (which works good for terminals)
	 */
	SerialInit(baud115k2);
	TimerInit();
	
	/* initialise status */
	status.kernelSize = 0;
	status.kernelType = fromFlash;
	status.ramdiskSize = 0;
	status.ramdiskType = fromFlash;
	status.blockSize = blockSize;
	status.downloadSpeed = baud115k2;

	init_pcic();
	
	/* Load kernel and ramdisk from flash to RAM */
SerialOutputString("\rMy Version 106\r");
	Reload("kernel", &status);
	Reload("ramdisk", &status);

	/* Print the required GPL string */
	SerialOutputString("\r" PACKAGE " version " VERSION  "\r"
			   "Copyright (C) 1999 2000 "
			   "Jan-Derk Bakker and Erik Mouw\r"
			   "Copyright (C) 2000 "
			   "Johan Pouwelse.\r");
	SerialOutputString(PACKAGE " comes with ABSOLUTELY NO WARRANTY; "
			   "read the GNU GPL for details.\r");
	SerialOutputString("This is free software, and you are welcome "
			   "to redistribute it\r");
	SerialOutputString("under certain conditions; "
			   "read the GNU GPL for details.\r\r");

	/* and some information */
#ifdef BLOB_DEBUG
	SerialOutputString("Running from ");
	if(RunningFromInternal())
		SerialOutputString("internal flash\r");
	else
		SerialOutputString("external flash\r");

	SerialOutputString("blockBase = 0x");
	SerialOutputHex((int) blockBase);
	SerialOutputString(", blockSize = 0x");
	SerialOutputHex(blockSize);
	SerialOutputByte('\r');
#endif
        
	/* wait 3 seconds before starting autoboot */
	SerialOutputString("Autoboot in progress, press any key to stop ");
#if 1
	for(i = 0; i < 3; i++) {
		SerialOutputByte('.');

 		retval = SerialInputBlock(commandline, 1, 1); 

		if(retval > 0)
			break;
	}
#else
        retval = SerialInputByte(commandline);
#endif
	/* no key was pressed, so proceed booting the kernel */
	if(retval == 0) {
		BootKernel(" console=ttySC1,115200");
	}

	SerialOutputString("\rAutoboot aborted\r");

	SerialOutputString("Type \"help\" to get a list of commands\r");

	/* the command loop. endless, of course */
	for(;;) {
		DisplayPrompt(NULL);

		/* wait an hour to get a command */
		numRead = GetCommand(commandline, 128, 3600);

		if(numRead > 0) {
			if(MyStrNCmp(commandline, "boot", 4) == 0) {
				BootKernel(commandline + 4);
			} else if(MyStrNCmp(commandline, "download ", 9) == 0) {
				Download(commandline + 9, &status);
			} else if(MyStrNCmp(commandline, "memtest1", 8) == 0) {
                            	memtest1();
			} else if(MyStrNCmp(commandline, "memtest2", 8) == 0) {
                            	memtest2();
			} else if(MyStrNCmp(commandline, "memtest3", 8) == 0) {
                            	memtest3();
			} else if(MyStrNCmp(commandline, "help", 4) == 0) {
				PrintHelp();

			} else if(MyStrNCmp(commandline, "dmb ", 4) == 0) {
				dmb(commandline+3);
			} else if(MyStrNCmp(commandline, "dmw ", 4) == 0) {
				dmw(commandline+3);
			} else if(MyStrNCmp(commandline, "dml ", 4) == 0) {
				dml(commandline+3);

			} else if(MyStrNCmp(commandline, "smb ", 4) == 0) {
                            	smb(commandline+3);
			} else if(MyStrNCmp(commandline, "smw ", 4) == 0) {
				smw(commandline+3);
			} else if(MyStrNCmp(commandline, "sml ", 4) == 0) {
                            	sml(commandline+3);

			} else if(MyStrNCmp(commandline, "bra ", 4) == 0) {
                            	bra(commandline+3);

			} else if(MyStrNCmp(commandline, "reload ", 7) == 0) {
				Reload(commandline + 7, &status);
			} else if(MyStrNCmp(commandline, "reset", 5) == 0) {
				ResetTerminal();
			} else if(MyStrNCmp(commandline, "speed ", 6) == 0) {
				SetDownloadSpeed(commandline + 6, &status);
			} else if(MyStrNCmp(commandline, "status", 6) == 0) {
				PrintStatus(&status);
			} else if(MyStrNCmp(commandline, "stt", 3) == 0) {
                            	st();
			} else {
				SerialOutputString("*** Unknown command: ");
				SerialOutputString(commandline);
				SerialOutputByte('\r');
			}
		}
	}
} /* c_main */


void BootKernel(char *commandline)
{
    	unsigned char *param;
        unsigned int *p;
		void (*theKernel)(int zero, int arch) = (void (*)(int, int))KERNEL_RAM_BASE;

        param=(unsigned char *)0xa8001000;
        p=(unsigned int *)param;
//		MyMemCpyChar(param+256,"keepinitrd",12);
        p[0]=1;
        p[1]=0;
        p[2]=0x100;
        p[3]=3;
        p[4]=0x300000;
        p[5]=0x600000;
        p[6]=1;
        p[7]=1;
        p[8]=1;

        if(*commandline == 0) {
			MyMemCpyChar(param+256, "console=ttySC1,115200", 30);
		} else {
			MyMemCpyChar(param+256, commandline+1, 256 );
		}
        

	/* we assume that the kernel is in place */
	/* See linux/include/asm-arm/system.h for architecture numbers */
	SerialOutputString("\rStarting kernel ...\r\r");

	theKernel(0,18);

	SerialOutputString("Hey, the kernel returned! This should not happen.\r");
}




void Download(char *commandline, blobStatus *status)
{
	u32 startAddress = 0;
	int bufLen;
	int *numRead = 0;

	if(MyStrNCmp(commandline, "kernel", 6) == 0) {
		/* download kernel */
		startAddress = KERNEL_RAM_BASE;
		bufLen = status->blockSize - KERNEL_BLOCK_OFFSET;
		numRead = &status->kernelSize;
		status->kernelType = fromDownload;
	} else if(MyStrNCmp(commandline, "ramdisk", 7) == 0) {
		/* download ramdisk */
		startAddress = RAMDISK_RAM_BASE;
		bufLen = status->blockSize - RAMDISK_BLOCK_OFFSET;
		numRead = &status->ramdiskSize;
		status->ramdiskType = fromDownload;
	} else {
		SerialOutputString("*** Don't know how to download \"");
		SerialOutputString(commandline);
		SerialOutputString("\"\r");
		return;
	}


	SerialInit(status->downloadSpeed);

	*numRead = UUDecode((char *)startAddress, bufLen);
	
	if(*numRead < 0) {
		/* something went wrong */
		SerialOutputString("*** Uudecode receive failed\r");
		
		/* reload the correct memory */
		Reload(commandline, status);

		SerialInit(baud9k6);
		return;
	}
	SerialOutputString("Received ");
	SerialOutputDec(*numRead);
	SerialOutputString(" (0x");
	SerialOutputHex(*numRead);
	SerialOutputString(") bytes.\r");


	SerialInit(baud9k6);
}




void PrintHelp(void)
{
	SerialOutputString("Help for " PACKAGE " " VERSION ", the LART bootloader\r");
	SerialOutputString("The following commands are supported:\r");
	SerialOutputString("* boot [kernel options]      Boot Linux with optional kernel options\r");
	SerialOutputString("* download {kernel|ramdisk}  Download kernel or ramdisk image to RAM\r");
	SerialOutputString("* flash {kernel|ramdisk}     Copy kernel or ramdisk from RAM to flash\r");
	SerialOutputString("* help                       Get this help\r");
	SerialOutputString("* reload {kernel|ramdisk}    Reload kernel or ramdisk from flash to RAM\r");
	SerialOutputString("* reset                      Reset terminal\r");
	SerialOutputString("* speed                      Set download speed\r");
	SerialOutputString("* status                     Display current status\r");
	SerialOutputString("* stt                        go to sdram monitor\r");
}




void SetDownloadSpeed(char *commandline, blobStatus *status)
{
	if(MyStrNCmp(commandline, "1200", 4) == 0) {
		status->downloadSpeed = baud1k2;
	} else if(MyStrNCmp(commandline, "1k2", 3) == 0) {
		status->downloadSpeed = baud1k2;
	} else if(MyStrNCmp(commandline, "9600", 4) == 0) {
		status->downloadSpeed = baud9k6;
	} else if(MyStrNCmp(commandline, "9k6", 3) == 0) {
		status->downloadSpeed = baud9k6;
	} else if(MyStrNCmp(commandline, "19200", 5) == 0) {
		status->downloadSpeed = baud19k2;
	} else if(MyStrNCmp(commandline, "19k2", 4) == 0) {
		status->downloadSpeed = baud19k2;
	} else if(MyStrNCmp(commandline, "38400", 5) == 0) {
		status->downloadSpeed = baud38k4;
	} else if(MyStrNCmp(commandline, "38k4", 4) == 0) {
		status->downloadSpeed = baud38k4;
	} else if(MyStrNCmp(commandline, "57600", 5) == 0) {
		status->downloadSpeed = baud57k6;
	} else if(MyStrNCmp(commandline, "57k6", 4) == 0) {
		status->downloadSpeed = baud57k6;
	} else if(MyStrNCmp(commandline, "115200", 6) == 0) {
		status->downloadSpeed = baud115k2;
	} else if(MyStrNCmp(commandline, "115k2", 5) == 0) {
		status->downloadSpeed = baud115k2;
	} else {
		SerialOutputString("*** Invalid download speed value \"");
		SerialOutputString(commandline);
		SerialOutputString("\"\r*** Valid values are:\r");
		SerialOutputString("*** 1200, 9600, 19200, 38400, 57600, 115200,\r");
		SerialOutputString("*** 1k2, 9k6, 19k2, 38k4, 57k6, and 115k2\r");
	}

	SerialOutputString("Download speed set to ");
	PrintSerialSpeed(status->downloadSpeed);
	SerialOutputString(" baud\r");
}




void PrintStatus(blobStatus *status)
{
	SerialOutputString("Bootloader    : " PACKAGE "\r");
	SerialOutputString("Version       : " VERSION "\r");

	SerialOutputString("Running from  : ");
	if(RunningFromInternal())
		SerialOutputString("internal");
	else
		SerialOutputString("external");

	SerialOutputString(" flash\rBlocksize     : 0x");
	SerialOutputHex(status->blockSize);
	SerialOutputString("\r");

	SerialOutputString("Download speed: ");
	PrintSerialSpeed(status->downloadSpeed);
	SerialOutputString(" baud\r");

	SerialOutputString("Kernel        : ");
	if(status->kernelType == fromFlash) {
		SerialOutputString("from flash\r");
	} else {
		SerialOutputString("downloaded, ");
		SerialOutputDec(status->kernelSize);
		SerialOutputString(" bytes\r");
	}

	SerialOutputString("Ramdisk       : ");
	if(status->ramdiskType == fromFlash) {
		SerialOutputString("from flash\r");
	} else {
		SerialOutputString("downloaded, ");
		SerialOutputDec(status->ramdiskSize);
		SerialOutputString(" bytes\r");
	}

}




void ResetTerminal(void)
{
	int i;

	SerialInit(baud9k6);
	SerialOutputString("          c");
	for(i = 0; i < 100; i++)
		SerialOutputByte('\r');

	SerialOutputString("c");
}




void Reload(char *commandline, blobStatus *status)
{
	u32 *src = 0;
	u32 *dst = 0;
	int numWords;

	if(MyStrNCmp(commandline, "kernel", 6) == 0) {
            	src = (u32 *)KERNEL_RAM_BASE;
		dst = (u32 *)KERNEL_START;
		numWords = KERNEL_LEN / 4;
		status->kernelSize = 0;
		status->kernelType = fromFlash;
		SerialOutputString("Loading kernel from flash ");
	} else if(MyStrNCmp(commandline, "ramdisk", 7) == 0) {
		src = (u32 *)RAMDISK_RAM_BASE;
		dst = (u32 *)INITRD_START;
		numWords = INITRD_LEN / 4;
		status->ramdiskSize = 0;
		status->ramdiskType = fromFlash;
		SerialOutputString("Loading ramdisk from flash ");
	} else {
		SerialOutputString("*** Don't know how to reload \"");
		SerialOutputString(commandline);
		SerialOutputString("\"\r");
		return;
	}

	MyMemCpy(src, dst, numWords);
	SerialOutputString(" done\r");
}




void PrintSerialSpeed(eBauds speed)
{
	switch(speed) {

	case baud1k2:
		SerialOutputHex(1200);
		break;

	case baud9k6:
		SerialOutputHex(9600);
		break;

	case baud19k2:
		SerialOutputHex(19200);
		break;

	case baud38k4:
		SerialOutputHex(38400);
		break;

	case baud57k6:
		SerialOutputHex(57600);
		break;

	case baud115k2:
		SerialOutputDec(115200);
		break;

	default:
		SerialOutputString("(unknown)");
		break;
	}
}

void waitaminute(void);

void memtest1(void) {
    int i,k;
    unsigned char *b;
    
    b=(unsigned char *)0x88000000;
    k=0;
    
    for(i=0;i<15*512*1024;i++) {
        *(b+i)=k;
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("written ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
    
    waitaminute();
    
    k=0;
    for(i=0;i<15*512*1024;i++) {
        if(*(b+i)!=k) {
            SerialOutputHex(i);        
            SerialOutputString(" ");
            SerialOutputHex(*(b+i));        
            SerialOutputString("\r");
        }
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("read ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
}

void memtest2(void) {
    int i,k;
    unsigned char *b;
    
    b=(unsigned char *)0x88000000;
    k=0;
    
    for(i=0;i<15*512*1024;i++) {
        *(b+i)=0xff;
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("written ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
    
    waitaminute();
    
    k=0;
    for(i=0;i<15*512*1024;i++) {
        if(*(b+i)!=0xff) {
            SerialOutputHex(i);        
            SerialOutputString(" ");
            SerialOutputHex(*(b+i));        
            SerialOutputString("\r");
        }
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("read ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
}

void memtest3(void) {
    int i,k;
    unsigned char *b;
    
    b=(unsigned char *)0x88000000;
    k=0;
    
    for(i=0;i<15*512*1024;i++) {
        *(b+i)=0;
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("written ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
    
    waitaminute();
    
    k=0;
    for(i=0;i<15*512*1024;i++) {
        if(*(b+i)!=0) {
            SerialOutputHex(i);        
            SerialOutputString(" ");
            SerialOutputHex(*(b+i));        
            SerialOutputString("\r");
        }
        k++;
        if(k==201)k=0;
        if((i&0xfffff)==0){
            SerialOutputString("read ");
            SerialOutputHex(i);
            SerialOutputString(" bytes\r");
        }
    }
}


void waitaminute(void) {
    volatile register unsigned char * s;
    register int t,r;
    
    s=(unsigned char *)0xffc80004;
    r=*s;
    t=60;
    
    while(t>0) {
        while(r==*s){};
        r=*s;
        t--;
    }
}

void st(void) {
    volatile register unsigned char *s;
    int i;
    void (*_st)(void) = (void (*)(void))0xa8010000;
    
    s=(unsigned char *)0xa8010000;
    for(i=0;i<sizeof(smalltest);i++) {
        *(s+i)=smalltest[i];
    }
    
    _st();
    
}

#define R64CNT  	0xffc80000
#define RTC_BIT_CHANGE 0x40	/* We have bug on SH-4 */
#define RCR1_CF		0x80	/* Carry Flag             */
#define p4_inb(addr)  *(unsigned char *)(addr)

void
sleep128 (unsigned int count)
{
  unsigned int cur128 = p4_inb (R64CNT) ^ RTC_BIT_CHANGE;
  unsigned int n = (cur128 + count) % 128;
  unsigned int m = (cur128 + count) / 128;

  if (count == 0)
    return;

  while (m != 0)
    {
      /* Wait next one tick */
      while ((p4_inb (R64CNT) ^ RTC_BIT_CHANGE) == cur128)
	/* Do nothing */
	asm volatile ("" : : :"memory");

      m--;

      /* Wait next 127 ticks */
      while ((p4_inb (R64CNT) ^ RTC_BIT_CHANGE) != cur128)
	/* Do nothing */
	asm volatile ("" : : :"memory");
    }

  while ((p4_inb (R64CNT) ^ RTC_BIT_CHANGE) != n)
    /* Do nothing */
    asm volatile ("" : : :"memory");
}

#define CONFIG_MEMORY_SIZE 0x8000000

/* Register addresses and such */
#define SH7751_BCR1     (volatile unsigned long *)0xFF800000
  #define BCR1_BREQEN   0x00080000
#define SH7751_BCR2     (volatile unsigned short*)0xFF800004
#define SH7751_WCR1     (volatile unsigned long *)0xFF800008
#define SH7751_WCR2     (volatile unsigned long *)0xFF80000C
#define SH7751_WCR3     (volatile unsigned long *)0xFF800010
#define SH7751_MCR      (volatile unsigned long *)0xFF800014

#define SH7751_PCICONF0 (volatile unsigned long *)0xFE200000
  #define PCI_SH7751_ID 0x35051054
#define SH7751_PCICONF1 (volatile unsigned long *)0xFE200004
  #define SH7751_PCICONF1_WCC 0x00000080
  #define SH7751_PCICONF1_PER 0x00000040
  #define SH7751_PCICONF1_BUM 0x00000004
  #define SH7751_PCICONF1_MES 0x00000002
  #define SH7751_PCICONF1_CMDS 0x000000C6
#define SH7751_PCICONF2 (volatile unsigned long *)0xFE200008
  #define SH7751_PCI_HOST_BRIDGE 0x6
#define SH7751_PCICONF3 (volatile unsigned long *)0xFE20000C
#define SH7751_PCICONF4 (volatile unsigned long *)0xFE200010
#define SH7751_PCICONF5 (volatile unsigned long *)0xFE200014
#define SH7751_PCICONF6 (volatile unsigned long *)0xFE200018

#define SH7751_PCICR    (volatile unsigned long *)0xFE200100
  #define SH7751_PCICR_PREFIX 0xa5000000
  #define SH7751_PCICR_PRST    0x00000002
  #define SH7751_PCICR_CFIN    0x00000001
#define SH7751_PCILSR0  (volatile unsigned long *)0xFE200104
#define SH7751_PCILSR1  (volatile unsigned long *)0xFE200108
#define SH7751_PCILAR0  (volatile unsigned long *)0xFE20010C
#define SH7751_PCILAR1  (volatile unsigned long *)0xFE200110
#define SH7751_PCIMBR   (volatile unsigned long *)0xFE2001C4
#define SH7751_PCIIOBR  (volatile unsigned long *)0xFE2001C8
#define SH7751_PCIPINT  (volatile unsigned long *)0xFE2001CC
  #define SH7751_PCIPINT_D3  0x00000002
  #define SH7751_PCIPINT_D0  0x00000001
#define SH7751_PCIPINTM (volatile unsigned long *)0xFE2001D0
#define SH7751_PCICLKR  (volatile unsigned long *)0xFE2001D4
  #define SH7751_PCICLKR_PREFIX 0xa5000000
#define SH7751_PCIBCR1  (volatile unsigned long *)0xFE2001E0
#define SH7751_PCIBCR2  (volatile unsigned long *)0xFE2001E4
#define SH7751_PCIWCR1  (volatile unsigned long *)0xFE2001E8
#define SH7751_PCIWCR2  (volatile unsigned long *)0xFE2001EC
#define SH7751_PCIWCR3  (volatile unsigned long *)0xFE2001F0
#define SH7751_PCIMCR   (volatile unsigned long *)0xFE2001F4
#define SH7751_PCI_MEM_BASE 0xFD000000
#define SH7751_PCI_MEM_SIZE 0x01000000
#define SH7751_PCI_IO_BASE  0xFE240000
#define SH7751_PCI_IO_SIZE  0x00040000

#define SH7751_CS3_BASE_ADDR   0x08000000
#define SH7751_P2CS3_BASE_ADDR 0xA8000000

#define SH7751_PCIPAR   (volatile unsigned long *)0xFE2001C0
#define SH7751_PCIPDR   (volatile unsigned long *)0xFE200220

unsigned long pci_nextio;
unsigned long pci_nextmem;

#define p4_out(addr,data)  *(addr) = data
#define p4_in(addr)  *(addr)

/* Return 0 if ok, else err code */
int init_pcic(void)
{
  /* Double-check that we're a 7751 chip */
  if (p4_in(SH7751_PCICONF0) != PCI_SH7751_ID)
    return 1;

SerialOutputHex(p4_in(SH7751_BCR2));

  /* Double-check some BSC config settings */
  /* (Area 3 non-MPX 32-bit, PCI bus pins) */
  if ((p4_in(SH7751_BCR1) & 0x20008) == 0x20000)
    return 2;
  if ((p4_in(SH7751_BCR2) & 0xC0) != 0xC0)
    return 3;
  if (p4_in(SH7751_BCR2) & 0x01)
    return 4;

  /* Force BREQEN in BCR1 to allow PCIC access */
  p4_out(SH7751_BCR1, (p4_in(SH7751_BCR1) | BCR1_BREQEN));

  /* Toggle PCI reset pin */
  p4_out(SH7751_PCICR, (SH7751_PCICR_PREFIX | SH7751_PCICR_PRST));
  sleep128(32);
  p4_out(SH7751_PCICR, SH7751_PCICR_PREFIX);
	
  /* Set cmd bits: WCC, PER, BUM, MES */
  /* (Addr/Data stepping, Parity enabled, Bus Master, Memory enabled) */
  p4_out(SH7751_PCICONF1, SH7751_PCICONF1_CMDS);

  /* Define this host as the host bridge */
  p4_out(SH7751_PCICONF2, (SH7751_PCI_HOST_BRIDGE << 24));

  /* Force PCI clock(s) on */
  p4_out(SH7751_PCICLKR, SH7751_PCICLKR_PREFIX);

  /* Clear powerdown IRQs, also mask them (unused) */
  p4_out(SH7751_PCIPINT, (SH7751_PCIPINT_D0 | SH7751_PCIPINT_D3)); 
  p4_out(SH7751_PCIPINTM, 0);

  /* Set up target memory mappings (for external DMA access) */
  /* Map both P0 and P2 range to Area 3 RAM for ease of use */

  p4_out(SH7751_PCILSR0, CONFIG_MEMORY_SIZE-1);
  p4_out(SH7751_PCILAR0, SH7751_CS3_BASE_ADDR);
  p4_out(SH7751_PCILSR1, CONFIG_MEMORY_SIZE-1);
  p4_out(SH7751_PCILAR1, SH7751_CS3_BASE_ADDR);

  p4_out(SH7751_PCICONF5, SH7751_CS3_BASE_ADDR);
  p4_out(SH7751_PCICONF6, SH7751_P2CS3_BASE_ADDR);

  /* Map memory window to same address on PCI bus */
  p4_out(SH7751_PCIMBR, SH7751_PCI_MEM_BASE);
  pci_nextmem = SH7751_PCI_MEM_BASE;
  
  /* Map IO window to same address on PCI bus */
  p4_out(SH7751_PCIIOBR, SH7751_PCI_IO_BASE);
  pci_nextio = SH7751_PCI_IO_BASE;

  /* Copy BSC registers into PCI BSC */
  p4_out(SH7751_PCIBCR1, p4_in(SH7751_BCR1));
  p4_out(SH7751_PCIBCR2, p4_in(SH7751_BCR2));
  p4_out(SH7751_PCIWCR1, p4_in(SH7751_WCR1));
  p4_out(SH7751_PCIWCR2, p4_in(SH7751_WCR2));
  p4_out(SH7751_PCIWCR3, p4_in(SH7751_WCR3));
  p4_out(SH7751_PCIMCR , p4_in(SH7751_MCR));

  /* Finally, set central function init complete */
  p4_out(SH7751_PCICR, (SH7751_PCICR_PREFIX | SH7751_PCICR_CFIN));
  return 0;
}


